package csbase.client.facilities.configurabletable;

import csbase.client.facilities.configurabletable.UI.ORIENTATION;
import csbase.client.facilities.configurabletable.UI.Panel4Tables;
import csbase.client.facilities.configurabletable.UI.SplitPane4Tables;
import csbase.client.facilities.configurabletable.UI.TabbedPane4Tables;
import csbase.client.facilities.configurabletable.UI.UI4Tables;
import csbase.client.facilities.configurabletable.model.Config;
import csbase.client.facilities.configurabletable.model.Panel;
import csbase.client.facilities.configurabletable.model.Splitpane;
import csbase.client.facilities.configurabletable.model.Tab;
import csbase.client.facilities.configurabletable.model.Table;
import csbase.client.facilities.configurabletable.model.TableRef;
import csbase.client.facilities.configurabletable.model.Tabs;
import csbase.client.facilities.configurabletable.stringprovider.IStringProvider;
import csbase.client.facilities.configurabletable.table.ConfigurableTable;

/**
 * Fbrica de elementos de interface.
 * 
 * @see ConfigurableTable
 * 
 * @author Tecgraf
 */
public class UIFactory {

  /**
   * Objeto que contm a configurao das interfaces do usurio.
   */
  private Config config;

  /**
   * Provedor de strings.
   */
  private IStringProvider stringProvider;

  /**
   * Fbrica de tabelas configurveis.
   */
  private ConfigurableTableFactory tableFactory;

  /**
   * Construtor default.
   * 
   * @param stringProvider - provedor de strings.
   */
  public UIFactory(IStringProvider stringProvider) {
    this.stringProvider = stringProvider;

    if (this.stringProvider == null) {
      this.stringProvider = new IStringProvider() {
        @Override
        public String getString(String key) {
          return key;
        }
      };
    }
  }

  /**
   * Construtor.
   * 
   * @param config - configurao de tabelas.
   * @param stringProvider - usado para internacionalizar o nome das colunas.
   */
  public UIFactory(Config config, IStringProvider stringProvider) {
    this(stringProvider);
    this.config = config;

    tableFactory = new ConfigurableTableFactory(config, stringProvider);
  }

  /**
   * Obtm o container de tabelas (painel simples ou painel de abas) cujo
   * identificador  <code>id</code>.
   * 
   * @param id - identificador do container de tabelas.
   * @return container de tabelas.
   */
  public UI4Tables getUI4Tables(String id) {

    UI4Tables result = null;
    for (Object obj : config.getUI().getTabsAndPanelAndSplitpane()) {
      if (obj instanceof Panel && ((Panel) obj).getId().equals(id)) {
        result = getPanel(id);
        break;
      }

      if (obj instanceof Tabs && ((Tabs) obj).getId().equals(id)) {
        result = getTabbedPane(id);
        break;
      }

      if (obj instanceof Splitpane && ((Splitpane) obj).getId().equals(id)) {
        result = getSplitPane(id);
        break;
      }
    }
    return result;
  }

  /**
   * Obtm o painel cujo identificador  <code>id</code>.
   * 
   * @param id - identificador do painel.
   * @return painel com suas respectivas tabelas.
   */
  public Panel4Tables getPanel(String id) {

    Panel4Tables panel = null;

    for (Object obj : config.getUI().getTabsAndPanelAndSplitpane()) {
      if (obj instanceof Panel && ((Panel) obj).getId().equals(id)) {
        Panel panelConf = (Panel) obj;

        ORIENTATION orientation = ORIENTATION.value(panelConf.getOrientation());
        panel = new Panel4Tables(panelConf.getId(), orientation);

        for (TableRef tableRef : panelConf.getTableRef()) {

          Table tableConf = getTableConfById(tableRef.getId());
          try {
            Class<?> clazz = Class.forName(tableConf.getDatatype());
            ConfigurableTable<?> table =
              tableFactory.getTable(tableConf.getId(), clazz);

            String label = stringProvider.getString(tableRef.getId());

            panel.addTable(table, label, clazz);

          }
          catch (ClassNotFoundException e) {
            throw new RuntimeException("No existe a classe: "
              + tableConf.getDatatype());
          }

        }

      }
    }

    panel.buildUI();

    return panel;
  }

  /**
   * Obtm o painel de abas cujo identificador  <code>id</code>.
   * 
   * @param id - identificador do painel de abas.
   * @return painel de abas com suas respectivas tabelas.
   */
  public TabbedPane4Tables getTabbedPane(String id) {

    TabbedPane4Tables tabbedPane = null;

    for (Object obj : config.getUI().getTabsAndPanelAndSplitpane()) {
      if (obj instanceof Tabs && ((Tabs) obj).getId().equals(id)) {
        Tabs tabsConf = (Tabs) obj;

        tabbedPane = new TabbedPane4Tables(tabsConf.getId());

        for (Tab tabConf : tabsConf.getTab()) {

          ORIENTATION orientation = ORIENTATION.value(tabConf.getOrientation());

          String tabTitle = stringProvider.getString(tabConf.getId());
          tabbedPane.addTab(tabTitle, orientation);

          for (TableRef tableRef : tabConf.getTableRef()) {

            Table tableConf = getTableConfById(tableRef.getId());

            try {
              Class<?> clazz = Class.forName(tableConf.getDatatype());
              ConfigurableTable<?> table =
                tableFactory.getTable(tableConf.getId(), clazz);

              String label = stringProvider.getString(tableRef.getId());

              tabbedPane.addTable(tabTitle, table, label, clazz);

            }
            catch (ClassNotFoundException e) {
              throw new RuntimeException("No existe a classe: "
                + tableConf.getDatatype());
            }

          }

        }

        tabbedPane.buildUI();
      }
    }

    return tabbedPane;
  }

  /**
   * Obtm o painel com separador cujo identificador  <code>id</code>.
   * 
   * @param id - identificador do painel com separador.
   * @return painel com separador com suas respectivas tabelas.
   */
  public SplitPane4Tables getSplitPane(String id) {
    SplitPane4Tables splitPane = null;

    for (Object obj : config.getUI().getTabsAndPanelAndSplitpane()) {
      if (obj instanceof Splitpane && ((Splitpane) obj).getId().equals(id)) {
        Splitpane splitpaneConf = (Splitpane) obj;

        ORIENTATION orientation =
          ORIENTATION.value(splitpaneConf.getOrientation());
        splitPane = new SplitPane4Tables(splitpaneConf.getId(), orientation);

        for (TableRef tableRef : splitpaneConf.getTableRef()) {

          Table tableConf = getTableConfById(tableRef.getId());
          try {
            Class<?> clazz = Class.forName(tableConf.getDatatype());
            ConfigurableTable<?> table =
              tableFactory.getTable(tableConf.getId(), clazz);

            String label = stringProvider.getString(tableRef.getId());

            splitPane.addTable(table, label, clazz);

          }
          catch (ClassNotFoundException e) {
            throw new RuntimeException("No existe a classe: "
              + tableConf.getDatatype());
          }

        }

      }
    }

    splitPane.buildUI();

    return splitPane;
  }

  /**
   * Mtodo auxiliar que obtm a configurao de uma tabela pelo seu
   * identificador.
   * 
   * @param tableId - identificador da tabela.
   * 
   * @return modelo da tabela.
   */
  private Table getTableConfById(String tableId) {
    for (Table table : config.getTables().getTable()) {
      if (table.getId().equals(tableId)) {
        return table;
      }
    }
    return null;
  }
}
