/*
 * UserGroupInfoDialog.java
 * 
 * $Author$ $Revision$ - $Date: 2007-11-22 19:04:50 -0200 (Thu,
 * 22 Nov 2007) $
 */
package csbase.client.ias;

import java.awt.BorderLayout;
import java.awt.GridLayout;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.text.MessageFormat;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JTextField;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.StandardDialogs;
import csbase.client.desktop.DesktopComponentDialog;
import csbase.client.remote.srvproxies.UserGroupProxy;
import csbase.client.util.StandardErrorDialogs;
import csbase.logic.UserGroup;
import csbase.logic.UserGroupInfo;

/**
 * A classe <code>UserGroupInfoDialog</code>  responsvel por incluir ou
 * modificar as informaes de um grupo de usurio, atravs de uma interface
 * especfica.
 * 
 * @author Marcus Leal
 * @version $Revision$
 */
public class UserGroupInfoDialog {
  /** Janela de cadastro de dados do Grupo de Usurio */
  protected DesktopComponentDialog window;
  /** Janela "pai" */
  protected Window owner;
  /** Campo nome do grupo de usurio */
  protected JTextField name;
  /** Campo descrio do grupo de usurio */
  protected JTextField description;
  /** Boto de alterao dos dados do grupo de usurio */
  protected JButton modifyUserGroupButton;
  /** Ttulo da janela */
  protected String windowLabel;
  /** Objeto para persistir os dados de grupos de usurios fornecidos */
  protected UserGroup userGroup;
  /** Indica se  um grupo de usurio que est sendo criado ou alterado */
  protected boolean isNew;

  /**
   * Cria e mostra a janela de incluso/modificao do grupo de usurio.
   */
  protected void display() {
    // Cria o painel com os campos e o boto limpar.
    JPanel mainPane = new JPanel(new BorderLayout());
    String borderTitle =
      (isNew ? LNG.get("IAS_USERGROUP_REGISTRATION") : LNG
        .get("IAS_USERGROUP_UPDATE_TITLE"));
    mainPane.setBorder(BorderFactory.createTitledBorder(borderTitle));
    mainPane.add(makeMainPanel(), BorderLayout.NORTH);
    mainPane.add(makeClearButtonPanel(), BorderLayout.SOUTH);
    // Cria o painel de fundo
    JPanel backPane = new JPanel(new BorderLayout());
    backPane.add(mainPane);
    backPane.add(makeButtonPanel(), BorderLayout.SOUTH);
    // Cria a janela    
    window = new DesktopComponentDialog(owner, windowLabel);
    window.getContentPane().add(backPane);
    window.setDefaultCloseOperation(DesktopComponentDialog.DISPOSE_ON_CLOSE);
    window.pack();
    window.center(owner);
    window.setVisible(true);
  }

  /**
   * Constri o painel para preenchimento ou alterao dos dados do grupo de
   * usurio.
   * 
   * @return o painel principal da janela
   */
  protected JPanel makeMainPanel() {
    // nome
    name = new JTextField(20);
    JLabel nameL = new JLabel(LNG.get("IAS_USERGROUP"));
    // descricao
    description = new JTextField();
    JLabel descriptionL = new JLabel(LNG.get("IAS_USERGROUP_DESCRIPTION"));
    // Preenche os campos
    initFields();
    // Monta os labels
    GridLayout layout = new GridLayout(0, 1);
    layout.setVgap(10);
    JPanel labelsPanel = new JPanel(layout);
    labelsPanel.add(nameL);
    labelsPanel.add(descriptionL);
    // Monta os campos de entrada
    JPanel fieldsPanel = new JPanel(new GridLayout(0, 1));
    fieldsPanel.add(name);
    fieldsPanel.add(description);
    // Monta o painel principal
    JPanel panel = new JPanel();
    panel.add(labelsPanel);
    panel.add(fieldsPanel);
    return panel;
  }

  /**
   * Inicializa os campos da janela com os dados do grupo de usurio a ser
   * modificada.
   */
  protected void initFields() {
    if (isNew) {
      return;
    }
    // Alterao do Grupo de Usurio
    name.setText(userGroup.getName());
    description.setText(userGroup.getDesc());
    ChangeListener changeList = new ChangeListener();
    name.addKeyListener(changeList);
    description.addKeyListener(changeList);
  }

  /**
   * Cria um painel contendo o boto limpar.
   * 
   * @return o painel criado com o boto de limpar campos
   */
  private JPanel makeClearButtonPanel() {
    JPanel panel = new JPanel();
    panel.add(makeClearButton());
    return panel;
  }

  /**
   * Constri o painel com os botes da janela de incluso/ modificao de
   * grupos de usurios. As aes previstas so para confirmar a operao
   * solicitada ou desistir da operao.
   * 
   * @return o painel de botes
   */
  protected JPanel makeButtonPanel() {
    JPanel buttonPanel = new JPanel();
    if (isNew) {
      buttonPanel.add(makeAddUserGroupButton());
      buttonPanel.add(makeCancelButton());
    }
    else {
      buttonPanel.add(makeChangeButton());
      buttonPanel.add(makeCloseButton());
    }
    return buttonPanel;
  }

  /**
   * Cria o boto de limpar campos.
   * 
   * @return o boto de limpar campos.
   */
  private JButton makeClearButton() {
    JButton clearButton = new JButton(LNG.get("IAS_CLEAR"));
    clearButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent ev) {
        clearFields();
      }
    });
    return clearButton;
  }

  /**
   * Cria o boto de incluso de grupo de usurio.
   * 
   * @return o boto de incluso do grupo de usuario.
   */
  protected JButton makeAddUserGroupButton() {
    JButton addUserGroupButton = new JButton(LNG.get("IAS_USERGROUP_INCLUDE"));
    addUserGroupButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent ev) {
        if (validateFields()) {
          createUserGroup();
        }
        else {
          StandardErrorDialogs.showErrorDialog(window, windowLabel,
            LNG.get("IAS_FILL_ALL_FIELDS"));
        }
      }
    });
    return addUserGroupButton;
  }

  /**
   * Cria o boto de modificao do grupo de usurio. Esse boto somente 
   * habilitado quando alguma modificao  feita nos campos.
   * 
   * @return o boto de modificao de dados do grupo de usurio.
   */
  protected JButton makeChangeButton() {
    modifyUserGroupButton = new JButton(LNG.get("IAS_USERGROUP_UPDATE"));
    modifyUserGroupButton.setEnabled(false);
    modifyUserGroupButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent ev) {
        if (validateFields()) {
          modifyUserGroup();
          window.close();
        }
        else {
          StandardErrorDialogs.showErrorDialog(window, windowLabel,
            LNG.get("IAS_FILL_ALL_FIELDS"));
          modifyUserGroupButton.setEnabled(false);
        }
      }
    });
    return modifyUserGroupButton;
  }

  /**
   * Cria o boto de cancelamento de operao.
   * 
   * @return o boto de cancelamento.
   */
  protected JButton makeCancelButton() {
    JButton cancelButton = new JButton(LNG.get("IAS_CLOSE"));
    cancelButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent ev) {
        window.close();
      }
    });
    return cancelButton;
  }

  /**
   * Cria o boto de fechamento da janela.
   * 
   * @return o boto de fechamento.
   */
  protected JButton makeCloseButton() {
    JButton closeButton = new JButton(LNG.get("IAS_CLOSE"));
    closeButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent ev) {
        // Se os campos forem alterados e no so vazios, apresenta a opo 
        // de salvar a alterao; caso contrrio, fecha a tela.
        if (hasChanged() && !isEmpty()) {
          Object[] options = { LNG.get("IAS_UPDATE"), LNG.get("IAS_CANCEL") };
          int selected =
            JOptionPane.showOptionDialog(window,
              LNG.get("IAS_USERGROUP_UPDATE_CONFIRMATION"), windowLabel,
              JOptionPane.DEFAULT_OPTION, JOptionPane.INFORMATION_MESSAGE,
              null, options, options[0]);
          if (options[selected] == LNG.get("IAS_UPDATE") && validateFields()) {
            modifyUserGroup();
          }
        }
        window.close();
      }
    });
    return closeButton;
  }

  /**
   * Verifica se o campos nome est preenchido (o campo descrio no 
   * obrigatrio).
   * 
   * @return true se o campos estiver preenchido ou false se estiver vazio.
   */
  protected boolean validateFields() {
    return (!name.getText().trim().equals(""));
  }

  /**
   * Limpa os campos de nome e descrio.
   */
  protected void clearFields() {
    name.setText("");
    description.setText("");
    name.requestFocus();
  }

  /**
   * Verifica se algum dos campos de preenchimento de dados do grupo de usurio
   * foi alterado.
   * 
   * @return true se algum campo foi editado ou false caso contrrio
   */
  public boolean hasChanged() {
    return (!name.getText().trim().equals(userGroup.getName()) || !description
      .getText().trim().equals(userGroup.getDesc()));
  }

  /**
   * Verifica se os campos de preenchimento de dados do grupo de usurio esto
   * sem preenchimento.
   * 
   * @return true se os campos estiverem vazios ou false caso contrario
   */
  private boolean isEmpty() {
    return ((name.getText().trim().equals("")) && description.getText().trim()
      .equals(""));
  }

  /**
   * Executa a ao de criar o grupo de usurio, usando os dados preenchidos nos
   * campos da janela.
   */
  public void createUserGroup() {
    String nameText = name.getText().trim();
    String descriptionText = description.getText().trim();
    UserGroupInfo userGroupInfo = new UserGroupInfo(nameText, descriptionText);
    userGroup =
      UserGroupProxy.createUserGroup(window, windowLabel,
        LNG.get("IAS_WAITING_USERGROUP_CREATION"), userGroupInfo);
    if (userGroup != null) {
      StandardDialogs.showInfoDialog(window, windowLabel, MessageFormat.format(
        LNG.get("IAS_USERGROUP_INCLUDED_WITH_SUCCESS"),
        new Object[] { userGroup.getName() }));
      userGroup = null;
      clearFields();
    }
  }

  /**
   * Executa a ao de modificar o grupo de usurio, usando os dados preenchidos
   * nos campos da janela.
   */
  public void modifyUserGroup() {
    String nameText = name.getText().trim();
    String descText = description.getText().trim();
    UserGroupInfo userGroupInfo = new UserGroupInfo(nameText, descText);
    UserGroup modifiedUserGroup =
      UserGroupProxy.modifyUserGroup(window, windowLabel,
        LNG.get("IAS_WAITING_USERGROUP_MODIFICATION"), userGroup.getId(),
        userGroupInfo);
    if (modifiedUserGroup != null) {
      this.userGroup = modifiedUserGroup;
      StandardDialogs.showInfoDialog(window, windowLabel, MessageFormat.format(
        LNG.get("IAS_USERGROUP_MODIFIED_WITH_SUCCESS"),
        new Object[] { userGroup.getName() }));
      modifyUserGroupButton.setEnabled(false);
    }
  }

  /**
   * Classe que implementa o listener que habilita ou no o boto de modificao
   * do grupo de usurio, dependendo se os dados dos campos estiverem ou no
   * preenchidos.
   */
  private class ChangeListener extends KeyAdapter {
    @Override
    public void keyReleased(KeyEvent kev) {
      if (hasChanged()) {
        modifyUserGroupButton.setEnabled(true);
      }
      else {
        modifyUserGroupButton.setEnabled(false);
      }
    }
  }

  /**
   * Cria uma janela para incluso de dados de um grupo de usurio novo.
   * 
   * @param owner Janela "pai".
   */
  public UserGroupInfoDialog(Window owner) {
    this(owner, null);
  }

  /**
   * Cria uma janela para modificao de dados de um grupo de usurio j
   * cadastrado no sistema, ou incluso de dados, caso o parmetro userGroup
   * seja null.
   * 
   * @param owner Janela "pai".
   * @param userGroup o UserGroup que vai ser modificado (ou null, se incluso)
   */
  public UserGroupInfoDialog(Window owner, UserGroup userGroup) {
    this.owner = owner;
    if (userGroup == null) {
      windowLabel = LNG.get("IAS_USERGROUP_INCLUSION_TITLE");
      isNew = true;
    }
    else {
      windowLabel = LNG.get("IAS_USERGROUP_UPDATE_TITLE");
      isNew = false;
      this.userGroup = userGroup;
    }
    display();
  }
}
