/*
 * $Id$
 */

package csbase.client.login;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Container;
import java.awt.Cursor;
import java.awt.FlowLayout;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Image;
import java.awt.Insets;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.net.URL;
import java.util.Locale;

import javax.swing.BorderFactory;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JPasswordField;
import javax.swing.JProgressBar;
import javax.swing.JTextField;
import javax.swing.UIManager;
import javax.swing.WindowConstants;

import csbase.exception.ConfigurationException;
import tecgraf.javautils.configurationmanager.Configuration;
import tecgraf.javautils.configurationmanager.ConfigurationManager;
import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.GUIUtils;
import tecgraf.javautils.gui.SwingThreadDispatcher;

/**
 * Representa um dilogo modal de login.  usado pelo cliente para verificar se
 * um usurio pode ou no acessar o sistema.
 *
 * @author Tecgraf/PUC-Rio
 */
public class LoginUI extends AbstractLoginUI {

  /**
   * O nome da propriedade que define o caminho para a imagem do dilogo de
   * login.
   */
  private static final String IMAGE_PATH_PROPERTY = "image.path";

  /**
   * O nome da propriedade que define o caminho para o cone do dilogo de
   * login.
   */
  private static final String ICON_PATH_PROPERTY = "icon.path";

  /**
   * Diretrio de imagens do CSBase.
   */
  private static final String ICON_DIRECTORY =
    "/csbase/client/resources/applicationimages/";

  /**
   * A configurao do dilogo.
   */
  final private Configuration configuration;

  /**
   * Barra de progresso
   */
  final private JProgressBar progressBar = createProgressBar(Color.black);

  /**
   * Painel de status
   */
  private JPanel statusPanel;

  /**
   * Label com a mensagem
   */
  private JLabel messageLabel;

  /**
   * Painel principal do dilogo
   */
  private JPanel dataPanel;

  /**
   * Label com icone do sistema
   */
  private JLabel image;

  /**
   * Painel de input
   */
  private JPanel inputPanel;

  /**
   * componentes extras para inserir no painel de input
   */
  private JComponent[][] extraInputComponents;

  /**
   * Dilogo propriamente dito.
   */
  final private JFrame frame = new JFrame();

  /**
   *
   */
  final private JButton loginButton = new JButton("");

  /**
   *
   */
  final private JButton cancelButton = new JButton("");

  /**
   * Campo texto onde o usurio digita o seu login.
   */
  final private JTextField loginTextField = new JTextField();

  /**
   * Campo texto onde o usurio digita sua senha.
   */
  final private JPasswordField passwordTextField = new JPasswordField();

  /**
   * A combo com os idiomas suportados pelo sistema.
   */
  final private JComboBox<LocaleComboBoxItem> languageComboBox =
    new JComboBox<>();

  /**
   * Cria o dilogo de login.
   *
   * @throws ConfigurationException Caso no seja possvel obter a configurao
   *         da classe.
   */
  public LoginUI() {
    frame.setResizable(false);
    setButtonsListeners();
    registerDefaultActions(frame);
    frame.setDefaultCloseOperation(WindowConstants.DO_NOTHING_ON_CLOSE);
    frame.addWindowListener(new WindowAdapter() {
      @Override
      public void windowClosing(final WindowEvent e) {
        cancel();
      }
    });
    try {
      configuration = ConfigurationManager.getInstance().getConfiguration(this
        .getClass());
    }
    catch (Exception e) {
      throw new IllegalStateException(e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void initializeUI(final Locale locale) {
    languageComboBox.setModel(new LocaleComboBoxModel());
    for (int i = 0; i < languageComboBox.getItemCount(); i++) {
      final LocaleComboBoxItem comboItem = languageComboBox.getItemAt(i);
      final Locale itemLocale = comboItem.getLocale();
      if (itemLocale.equals(locale)) {
        final LocaleComboBoxModel model = (LocaleComboBoxModel) languageComboBox
          .getModel();
        model.setSelectedItem(comboItem);
        break;
      }
    }
    mountDialog();

    this.frame.pack();
    GUIUtils.centerOnScreen(this.frame);
    this.frame.toFront();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void updateUIForLoginStarting() {
    showProgress();
    disableDialog();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected String getLogin() {
    return this.loginTextField.getText();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected String getPassword() {
    return new String(this.passwordTextField.getPassword());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Locale getSelectedLocale() {
    LocaleComboBoxItem selectedLocale =
      (LocaleComboBoxItem) this.languageComboBox.getSelectedItem();
    return selectedLocale.getLocale();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean validateLoginUI() {
    if (getLogin().isEmpty()) {
      showInvalidLoginMessage(LoginMessage.EMPTY_LOGIN);
      passwordTextField.setText("");
      this.loginTextField.requestFocusInWindow();
      return false;
    }

    if (getPassword().isEmpty()) {
      if (!this.loginTextField.isFocusOwner()) {
        showInvalidLoginMessage(LoginMessage.EMPTY_PASSWORD);
      }
      this.passwordTextField.requestFocusInWindow();
      return false;
    }

    return true;
  }

  /**
   * Faz restaurao do dilogo do elogin.
   *
   * @param message mensagem exibida
   * @param enableLogin indicativo de habilitao de boto de login.
   */
  private void restoreDialog(final LoginMessage message,
    final boolean enableLogin) {
    SwingThreadDispatcher.invokeLater(new Runnable() {
      @Override
      public void run() {
        passwordTextField.setText("");
        enableDialog();
        showInvalidLoginMessage(message);
        hideProgress();
        loginButton.setEnabled(enableLogin);
      }
    });
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void updateUIForInvalidLoginInfo() {
    restoreDialog(LoginMessage.INVALID_LOGIN_INFO, true);
    loginTextField.requestFocusInWindow();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void updateUIForInvalidClientVersion() {
    restoreDialog(LoginMessage.INVALID_CLIENT_VERSION, false);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void updateUIForLoginException() {
    restoreDialog(LoginMessage.LOGIN_EXCEPTION, true);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void preClientInitializationException() {
    adjustDialogStatus(true);
  }

  /**
   * Consulta o valor de frame
   *
   * @return o valor
   */
  protected final JFrame getFrame() {
    return frame;
  }

  /**
   * Consulta o valor de loginButton
   *
   * @return o valor
   */
  protected final JButton getLoginButton() {
    return loginButton;
  }

  /**
   * Consulta o valor de cancelButton
   *
   * @return o valor
   */
  protected final JButton getCancelButton() {
    return cancelButton;
  }

  /**
   * Consulta o valor de loginTextField
   *
   * @return o valor
   */
  protected final JTextField getLoginTextField() {
    return loginTextField;
  }

  /**
   * Consulta o valor de passwordTextField
   *
   * @return o valor
   */
  protected final JPasswordField getPasswordTextField() {
    return passwordTextField;
  }

  /**
   * Consulta o valor de languageComboBox
   *
   * @return o valor
   */
  protected final JComboBox<LocaleComboBoxItem> getLanguageComboBox() {
    return languageComboBox;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void setLogin(String login) {
    getLoginTextField().setText(login);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void setPassword(String password) {
    getPasswordTextField().setText(password);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void cancel() {
    setCanceled(true);
    disposeUI();
    unlock();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void disposeUI() {
    this.frame.dispose();
  }

  /**
   * Desabilita componentes do dilogo
   */
  private void disableDialog() {
    frame.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
    loginButton.setEnabled(false);
    cancelButton.setEnabled(false);
    adjustDialogStatus(false);
  }

  /**
   * Habilita os componentes do dilogo
   */
  private void enableDialog() {
    frame.setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
    adjustDialogStatus(true);
    loginButton.setEnabled(true);
    cancelButton.setEnabled(true);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void preClientInitialization() {
    showProgress();
  }

  /**
   * Cria o painel de botes.
   */
  private void setButtonsListeners() {
    this.loginButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(final ActionEvent ev) {
        login();
      }
    });

    this.cancelButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(final ActionEvent ev) {
        cancel();
      }
    });
  }

  /**
   * Obtm os dados informados pelo usurio.
   *
   * @return Os dados informados pelo usurio, ou null, caso o usurio tenha
   *         cancelado o dilogo.
   */
  @Override
  public InitialContext getLoginData() {
    if (isCanceled()) {
      return null;
    }
    final LocaleComboBoxItem item = (LocaleComboBoxItem) this.languageComboBox
      .getSelectedItem();
    final InitialContext loginData = new InitialContext(item.getLocale());
    return loginData;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void showUI() {
    this.frame.setVisible(true);
  }

  /**
   * Cria o cone do dilogo de login.
   *
   * @return A imagem a ser exibida como cone do dilogo, ou null, caso nenhuma
   *         tenha sido informada.
   */
  private Image createIconImage() {
    String iconPath = this.configuration.getOptionalProperty(
      ICON_PATH_PROPERTY);
    if (iconPath == null) {
      return null;
    }
    final URL iconPathUrl = this.getClass().getResource(iconPath);
    return Toolkit.getDefaultToolkit().getImage(iconPathUrl);
  }

  /**
   * Adiciona o <code>messageLabel</code> ao painel de status.
   */
  private void addMessageLabel() {
    GridBagConstraints gbc = new GridBagConstraints();
    gbc.weightx = 1;
    gbc.insets = new Insets(3, 3, 2, 2);
    gbc.anchor = GridBagConstraints.WEST;
    statusPanel.add(messageLabel, gbc);
  }

  /**
   * Cria o painel principal do dilogo. O painel principal do dilogo 
   * composto pelo painel de dados e pelo painel de botes.
   *
   * @return O painel principal do dilogo.
   */
  private JPanel createMainPanel() {
    JPanel mainPanel = new JPanel();
    mainPanel.setLayout(new GridBagLayout());
    dataPanel = createDataPanel();
    mainPanel.add(dataPanel, new GBC(1, 1));
    mainPanel.add(createButtonPanel(), new GBC(1, 2));
    return mainPanel;
  }

  /**
   * Cria o painel de botes.
   *
   * @return O painel de botes.
   */
  private JPanel createButtonPanel() {
    final JButton loginButton = getLoginButton();
    loginButton.setText("Entrar");

    final JButton cancelButton = getCancelButton();
    cancelButton.setText("Cancelar");

    final JComponent[] cmps = new JComponent[] { loginButton, cancelButton };
    GUIUtils.matchPreferredSizes(cmps);

    final JPanel buttonPanel = new JPanel(new FlowLayout());
    buttonPanel.add(loginButton);
    buttonPanel.add(cancelButton);
    return buttonPanel;
  }

  /**
   * Adiciona novos componentes no painel de entrada de dados. Os novos so
   * adicionados abaixo dos j existentes.
   *
   * @param extraComponents os componentes extras a serem adicionados.
   */
  public void addExtraInputComponents(final JComponent[][] extraComponents) {
    this.extraInputComponents = extraComponents;
  }

  /**
   * Cria o painel de dados.
   *
   * @return O painel de dados.
   */
  protected JPanel createDataPanel() {
    final JTextField loginTextField = getLoginTextField();
    final JLabel userLabel = new JLabel("Usurio:");

    final JPasswordField passwordTextField = getPasswordTextField();
    final JLabel passwordLabel = new JLabel("Senha:");

    final JComboBox<LocaleComboBoxItem> languageComboBox =
      getLanguageComboBox();
    final JLabel localeLabel = new JLabel("Idioma:");

    final JPanel panel = new JPanel(new GridBagLayout());

    image = this.createImageLabel();
    if (image != null) {
      panel.add(image, new GBC(0, 0).center().insets(5, 10, 5, 5));
    }

    JComponent[][] inputComponents = new JComponent[][] { { userLabel,
        loginTextField }, { passwordLabel, passwordTextField }, { localeLabel,
            languageComboBox } };

    if (extraInputComponents != null) {
      int c = inputComponents[0].length + extraInputComponents[0].length;
      int r = inputComponents.length + extraInputComponents.length;
      JComponent[][] jc = new JComponent[r][c];
      System.arraycopy(inputComponents, 0, jc, 0, inputComponents.length);
      System.arraycopy(extraInputComponents, 0, jc, inputComponents.length,
        extraInputComponents.length);
      inputPanel = GUIUtils.createBasicGridPanel(jc);
    }
    else {
      inputPanel = GUIUtils.createBasicGridPanel(inputComponents);
    }

    panel.add(inputPanel, new GBC(1, 0).center());
    return panel;
  }

  /**
   * Cria a imagem do dilogo.
   *
   * @return A imagem do dilogo, ou null, caso no tenha sido informada a
   *         imagem do dilogo.
   */
  private JLabel createImageLabel() {
    String imagePath = this.configuration.getOptionalProperty(
      IMAGE_PATH_PROPERTY);
    if (imagePath == null) {
      return null;
    }
    URL imagePathUrl = this.getClass().getResource(imagePath);
    return new JLabel(new ImageIcon(imagePathUrl));
  }

  /**
   * Exibe barra de progresso
   */
  protected void showProgress() {
    final String property = "progress.message";
    final String progressMessage = configuration.getOptionalProperty(property,
      "Aguarde...");

    progressBar.setString(progressMessage);
    if (messageLabel != null) {
      statusPanel.remove(messageLabel);
    }
    GridBagConstraints gbc = new GridBagConstraints();
    gbc.weightx = 1;
    gbc.insets = new Insets(2, 2, 2, 2);
    gbc.fill = GridBagConstraints.BOTH;
    statusPanel.add(progressBar, gbc);
    statusPanel.validate();
    statusPanel.repaint();
  }

  /**
   * Oculta barra de progresso
   */
  protected void hideProgress() {
    statusPanel.remove(progressBar);
    addMessageLabel();

    final JFrame frame = getFrame();
    frame.pack();

    statusPanel.validate();
    statusPanel.repaint();
  }

  /**
   * Cria uma barra de progresso
   *
   * @param c Cor do texto
   * @return Uma instncia da barra de progresso
   */
  private static JProgressBar createProgressBar(Color c) {
    Object selectionBackground = UIManager.get(
      "ProgressBar.selectionBackground");
    Object selectionForeground = UIManager.get(
      "ProgressBar.selectionForeground");
    UIManager.put("ProgressBar.selectionBackground",
      new javax.swing.plaf.ColorUIResource(c));
    UIManager.put("ProgressBar.selectionForeground",
      new javax.swing.plaf.ColorUIResource(c));
    final JProgressBar pb = new JProgressBar();
    UIManager.put("ProgressBar.selectionBackground", selectionBackground);
    UIManager.put("ProgressBar.selectionForeground", selectionForeground);
    pb.setIndeterminate(true);
    pb.setStringPainted(true);
    return pb;
  }

  /**
   * Habilita/Desabilita todos os components de um container
   *
   * @param container O container que contm os components
   * @param b true habilita e false desabilita
   */
  private void setEnableContainer(Container container, boolean b) {
    for (Component c : container.getComponents()) {
      if (c instanceof Container) {
        setEnableContainer((Container) c, b);
      }
      c.setEnabled(b);
    }
  }

  /**
   * Ajuste do dilogo em modo habilitado conforme indicativo
   * 
   * @param status indicativo de habilitado.
   */
  protected void adjustDialogStatus(final boolean status) {
    if (status) {
      setEnableContainer(dataPanel, true);
    }
    else {
      setEnableContainer(dataPanel, false);
      if (image != null) {
        image.setEnabled(true);
      }
    }
  }

  /**
   * Mostra no status bar o texto de falha de login.
   *
   * @param msg o tipo de mensagem a ser mostrada na barra de status
   */
  protected void showInvalidLoginMessage(final LoginMessage msg) {
    ImageIcon ERROR_ICON = new ImageIcon(LoginUI.class.getResource(
      ICON_DIRECTORY + "Error16.png"));
    messageLabel.setVerticalTextPosition(JLabel.TOP);
    messageLabel.setIcon(ERROR_ICON);
    messageLabel.setText(msg.getDefaultDescription());
  }

  /**
   * Mostra na status bar o texto defalt da mesma.
   */
  protected void showDefaultLoginMessage() {
    String property = "status.message";
    String statusMessage = configuration.getOptionalProperty(property);
    if (statusMessage == null) {
      return;
    }
    final ImageIcon INFO_ICON = new ImageIcon(LoginUI.class.getResource(
      ICON_DIRECTORY + "Information16.png"));
    messageLabel.setVerticalTextPosition(JLabel.TOP);
    messageLabel.setIcon(INFO_ICON);
    messageLabel.setText(statusMessage);
  }

  /**
   * Cria o painel com o contedo do dilogo. O contedo do dilogo  o seu
   * painel principal e a barra de status.
   */
  protected void mountDialog() {
    final JFrame frame = getFrame();
    final String title = getTitle();
    frame.setTitle(title);

    final Image icon = createIconImage();
    if (icon != null) {
      frame.setIconImage(icon);
    }

    final JTextField loginText = getLoginTextField();
    final JPasswordField pwdText = getPasswordTextField();
    final int N = 15;
    pwdText.setColumns(N);
    loginText.setColumns(N);

    final JPanel panel = new JPanel(new BorderLayout());
    final JPanel mainPanel = createMainPanel();
    panel.add(mainPanel, BorderLayout.CENTER);

    statusPanel = new JPanel(new GridBagLayout());
    statusPanel.setBorder(BorderFactory.createLoweredBevelBorder());
    panel.add(statusPanel, BorderLayout.SOUTH);
    messageLabel = new JLabel();
    showDefaultLoginMessage();
    if (messageLabel != null) {
      addMessageLabel();
    }

    frame.setContentPane(panel);
  }

  /**
   * Obtm o ttulo da janela da propriedade "title". Se no tiver sido
   * definido, usamos "CSBase".
   *
   * @return ttulo da janela
   */
  protected String getTitle() {
    return configuration.getOptionalProperty("title", "CSBase");
  }
}
