package csbase.client.preferences.util;

import java.awt.Font;
import java.awt.GridBagLayout;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.Stack;

import javax.swing.JCheckBox;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;

import csbase.client.desktop.DesktopFrame;
import csbase.client.desktop.DesktopPref;
import csbase.client.preferences.PreferenceCategory;
import csbase.client.preferences.PreferenceEditor;
import csbase.client.preferences.PreferenceValue;
import csbase.client.preferences.definition.PreferenceDefinition;
import csbase.client.preferences.definition.PreferencePolicy;
import csbase.client.preferences.editors.FavoriteAlgorithmsEditor;
import csbase.client.preferences.editors.FavoriteApplicationsEditor;
import csbase.client.preferences.editors.LocalDirectoryEditor;
import csbase.client.preferences.editors.PreferredAppEditor;
import csbase.client.preferences.types.PVBoolean;
import csbase.client.preferences.types.PVColor;
import csbase.client.preferences.types.PVInteger;
import csbase.client.preferences.types.PVTables;
import csbase.logic.applicationservice.ApplicationRegistry;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GBC;

/**
 * Conjunto de mtodos utilitrios para as preferncias.
 * 
 * @author Tecgraf
 */
public class PreferencesUtil {

  /** Fonte usada no painel que exibe as preferncias. */
  static Font font = new Font(Font.SANS_SERIF, Font.PLAIN, 13);

  /**
   * Cria um painel que organiza todas as preferncias de uma categoria e seus
   * editores.
   * 
   * @param pc categoria de preferncia.
   * @param showDefaultValues true se for para criar o painel com os valores
   *        default, false caso contrrio.
   * @return painel com as preferncias e seus editores.
   */
  public static JPanel createPreferencePanel(PreferenceCategory pc,
    boolean showDefaultValues) {
    JPanel preferencePanel = new JPanel(new GridBagLayout());
    if (pc.hasPreferences(PreferencePolicy.READ_ONLY,
      PreferencePolicy.READ_WRITE)) {
      int row = 0;
      for (JComponent component : createComponentList(pc, showDefaultValues)) {
        preferencePanel.add(component, new GBC(0, row++).horizontal().left(10)
          .right(10));
      }
      preferencePanel.add(new JLabel(), new GBC(0, row).both());
    }
    else {
      JLabel label = new JLabel(LNG.get("no.preferences"));
      label.setFont(font);

      preferencePanel.add(label, new GBC(0, 0).insets(10));
      preferencePanel.add(new JLabel(), new GBC(0, 1).both());
    }
    return preferencePanel;
  }

  /**
   * Dado um registro de aplicao, retorna o nome convencionado da enumerao
   * que define as preferncias da aplicao.
   * 
   * @param registry registro da aplicao.
   * @return nome da enumerao que define as preferncias de uma aplicao.
   */
  public static String getAppEnumName(ApplicationRegistry registry) {
    if (registry == null) {
      throw new IllegalArgumentException(
        "Registro de aplicao no pode ser nulo.");
    }
    return registry.getClassName() + "Pref";
  }

  /**
   * Busca a categoria filha que possui uma dada contante.
   * 
   * @param name nome da constante.
   * @param base categoria usada como base da busca.
   * @return categoria que possui a constante, null caso contrrio.
   */
  @SuppressWarnings("unchecked")
  public static PreferenceCategory deepSearch(PreferenceDefinition name,
    PreferenceCategory base) {

    PreferenceCategory result = base;

    Stack<Class<PreferenceDefinition>> stack =
      new Stack<Class<PreferenceDefinition>>();
    Class<?> currentCat = name.getClass();

    while (isPreferenceDefition(currentCat) && !base.getId().equals(currentCat
      .getName())) {
      stack.push((Class<PreferenceDefinition>) currentCat);
      currentCat = currentCat.getEnclosingClass();
    }

    while (!stack.empty()) {
      result = result.getCategory(stack.pop());
    }

    return result;
  }

  /**
   * Cria uma lista onde cada componente armazena uma preferncia (ou um
   * conjunto delas se forem do mesmo tipo). Com isso,  possvel organizar a
   * exibio das preferncias de uma maneira visualmente mais agradvel.
   * 
   * @param pc categoria de preferncias.
   * @param showDefaultValues true se for para criar o painel com os valores
   *        default, false caso contrrio.
   * @return lista com todos os paineis.
   */
  private static List<JComponent> createComponentList(PreferenceCategory pc,
    boolean showDefaultValues) {
    List<List<PreferenceValue<?>>> fold =
      new ArrayList<List<PreferenceValue<?>>>();

    PreferenceValue<?> lastValue = null;
    List<PreferenceValue<?>> lastList = null;

    for (PreferenceValue<?> pv : pc.getPreferences(PreferencePolicy.READ_ONLY,
      PreferencePolicy.READ_WRITE)) {
      if (isChanged(pv, lastValue)) {
        lastList = new LinkedList<PreferenceValue<?>>();
        fold.add(lastList);
      }
      lastList.add(pv);
      lastValue = pv;
    }

    List<JComponent> result = new LinkedList<JComponent>();
    for (List<PreferenceValue<?>> list : fold) {
      JPanel panel = new JPanel(new GridBagLayout());
      int row = 0;
      for (PreferenceValue<?> pv : list) {
        JLabel label = new JLabel(pv.getLabel());
        label.setToolTipText(pv.getDescription());
        label.setFont(font);

        JComponent component = null;

        if (pv.getEditor() != null) {
          PreferenceEditor<?> editor = pv.getEditor();

          component = editor.getComponent(showDefaultValues);
        }
        else {
          component = new JLabel();
        }

        if (pv.getName().equals(DesktopPref.SHOW_TRAY_ICON)) {
          DesktopFrame desktop = DesktopFrame.getInstance();

          if (desktop.isTrayIconSupported()) {
            JCheckBox check = (JCheckBox) component;
            check.setText(label.getText());
            check.setToolTipText(label.getToolTipText());
            check.setFont(font);

            panel.add(component, new GBC(0, row++).west().insets(3)
              .horizontal());
          }
        }
        else if (pv instanceof PVBoolean && component instanceof JCheckBox) {
          JCheckBox check = (JCheckBox) component;
          check.setText(label.getText());
          check.setToolTipText(label.getToolTipText());
          check.setFont(font);

          panel.add(component, new GBC(0, row++).west().insets(3).horizontal());
        }
        else if (pv instanceof PVBoolean || pv instanceof PVColor) {
          panel.add(component, new GBC(0, row).west().none().insets(3));
          panel.add(label, new GBC(1, row++).east().horizontal().insets(3));
        }
        else if (pv instanceof PVInteger) {
          panel.add(label, new GBC(0, row).west().none().insets(3));
          panel.add(component, new GBC(1, row).west().insets(3));
          panel.add(new JLabel(), new GBC(2, row++).horizontal().insets(3));
        }
        else if (pv instanceof PVTables || pv
          .getEditor() instanceof PreferredAppEditor || pv
            .getEditor() instanceof LocalDirectoryEditor || pv
              .getEditor() instanceof FavoriteApplicationsEditor || pv
                .getEditor() instanceof FavoriteAlgorithmsEditor) {
          panel.add(component, new GBC(0, row++).west().horizontal().insets(3));
        }
        else {
          panel.add(label, new GBC(0, row).west().none().insets(3));
          panel.add(component, new GBC(1, row++).horizontal().insets(3));
        }
      }
      result.add(panel);
    }

    return result;
  }

  /**
   * Retorna valor boleando que indica se o tipo do valor anterior  diferente
   * do atual seguindo algumas convenes. Consideramos que {@link PVBoolean} e
   * {@link PVColor} so diferentes de todos, e o restante dos valores so todos
   * do mesmo tipo.
   * 
   * @param pv valor atual.
   * @param lastValue valor anterior.
   * @return true se o tipo do valor anterior  diferente do atual, false caso
   *         contrrio.
   */
  private static boolean isChanged(PreferenceValue<?> pv,
    PreferenceValue<?> lastValue) {
    if (lastValue == null) {
      return true;
    }

    if (!lastValue.getClass().equals(pv.getClass())
      && (lastValue instanceof PVBoolean || pv instanceof PVBoolean)) {
      return true;
    }

    if (!lastValue.getClass().equals(pv.getClass())
      && (lastValue instanceof PVColor || pv instanceof PVColor)) {
      return true;
    }

    return false;
  }

  /**
   * Verifica se uma dada classe implementa a interface
   * {@link PreferenceDefinition}.
   * 
   * @param clazz classe a ser verificada.
   * @return true se a classe implementa {@link PreferenceDefinition}, false
   *         caso contrrio.
   */
  private static boolean isPreferenceDefition(Class<?> clazz) {
    if (clazz == null) {
      return false;
    }

    Class<?>[] interfaces = clazz.getInterfaces();
    for (Class<?> impl : interfaces) {
      if (impl.equals(PreferenceDefinition.class)) {
        return true;
      }
    }
    return false;
  }
}
