/*
 * $Id$
 */
package csbase.client.project;

import java.awt.Window;
import java.awt.dnd.DnDConstants;
import java.util.Comparator;
import java.util.Enumeration;
import java.util.EventObject;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Vector;

import javax.swing.Action;
import javax.swing.ActionMap;
import javax.swing.DefaultCellEditor;
import javax.swing.InputMap;
import javax.swing.JPopupMenu;
import javax.swing.JTree;
import javax.swing.KeyStroke;
import javax.swing.ToolTipManager;
import javax.swing.TransferHandler;
import javax.swing.UIManager;
import javax.swing.border.Border;
import javax.swing.event.CellEditorListener;
import javax.swing.event.ChangeEvent;
import javax.swing.event.TreeModelListener;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.text.Document;
import javax.swing.tree.DefaultTreeCellEditor;
import javax.swing.tree.DefaultTreeCellRenderer;
import javax.swing.tree.TreeCellEditor;
import javax.swing.tree.TreeNode;
import javax.swing.tree.TreePath;
import javax.swing.tree.TreeSelectionModel;

import csbase.client.desktop.RemoteTask;
import csbase.client.project.action.CommonFileCopyAction;
import csbase.client.project.action.CommonFileCutAction;
import csbase.client.project.action.CommonFileDeleteAction;
import csbase.client.project.action.CommonFilePasteAction;
import csbase.client.project.action.CommonFileRenameAction;
import csbase.client.project.action.ProjectTreeUpdateAction;
import csbase.client.project.tasks.GetChildFromPathTask;
import csbase.logic.ClientProjectFile;
import csbase.logic.CommonClientProject;
import csbase.logic.ProjectBasicInfo;
import csbase.logic.ProjectFileFilter;
import csbase.logic.filetypefinder.FileTypeFinder;
import tecgraf.javautils.core.lng.LNG;

/**
 * <p>
 * A classe modela uma rvore de diretrios de um projeto. A representao usa
 * uma {@link javax.swing.JTree} que pode ser recuperada com o mtodo
 * {@link #getTree()}.
 * </p>
 * <p>
 *  possvel definir quais arquivos sero visualizados atravs da rvore e
 * quais arquivos podero ser selecionados. Para fazer essa filtragem, deve-se
 * utilizar objetos do pacote csbase.logic.filters
 * </p>
 *
 * @author Tecgraf
 */
public class ProjectTree implements ProjectFileContainer {
  /**
   * O componente grfico da rvore
   */
  private JTree tree;

  /**
   * Janela sobre a qual a rvore est sendo exibida
   */
  private Window owner;

  /**
   * Ttulo
   */
  private String title;

  /**
   * Listeners interessados em mudanas na rvore
   */
  private Vector<ProjectTreeListener> projectTreeListenerList;

  /**
   * Menu de popup da raiz do projeto
   */
  private JPopupMenu popupRoot;

  /**
   * Menu de popup de diretrio
   */
  private JPopupMenu popupDir;

  /**
   * Menu de popup de dataset
   */
  private JPopupMenu popupDataset;

  /**
   * Menu de popup de seleo mltipla
   */
  private JPopupMenu popupMultSelection;

  /**
   * Menu de popup de arquivo
   */
  private JPopupMenu popupFile;

  /**
   * O projeto que est sendo representado pela rvore.
   */
  private CommonClientProject project;

  /**
   * Lista de ouvintes interessados em mudanas na seleo da rvore.
   */
  private List<ProjectTreeSelectionListener> treeSelectionListenerList;

  /**
   * O editor de ns. Utilizado para renomear os arquivos representados pelos
   * ns da rvore.
   */
  private ProjectTreeCellEditor editor;

  /**
   * Indica se a transferncia de arquivos est habilitada.
   */
  protected boolean transferEnabled;

  /**
   * Indica se a remoo de arquivos est habilitada.
   */
  protected boolean deleteEnabled;

  /**
   * Indica se a edio de nomes de arquivos est habilitada.
   */
  protected boolean renameEnabled;

  /**
   * Indica se a atualizao de diretrios e do projeto est habilitada.
   */
  protected boolean refreshEnabled;

  /**
   * Atribui um menu de pop-up para a raz da rvore.
   *
   * @param popupRoot O menu pop-up da raz.
   */
  public void setPopupRoot(JPopupMenu popupRoot) {
    this.popupRoot = popupRoot;
  }

  /**
   * Obtem o menu pop-up da raz
   *
   * @return O menu pop-up da raz.
   */
  public JPopupMenu getPopupRoot() {
    return this.popupRoot;
  }

  /**
   * Atribui um menu pop-up para arquivos.
   *
   * @param popupFile O menu pop-up de arquivos.
   */
  public void setPopupFile(JPopupMenu popupFile) {
    this.popupFile = popupFile;
  }

  /**
   * Obtem o menu pop-up de arquivos
   *
   * @return O menu pop-up de arquivos.
   */
  public JPopupMenu getPopupFile() {
    return this.popupFile;
  }

  /**
   * Atribui um menu pop-up para seleo mltipla.
   *
   * @param popupMultSelection O menu pop-up para seleo mltipla.
   */
  public void setPopupMultSelection(JPopupMenu popupMultSelection) {
    this.popupMultSelection = popupMultSelection;
  }

  /**
   * Obtem o menu pop-up de seleo mltipla.
   *
   * @return O menu pop-up de seleo mltipla.
   */
  public JPopupMenu getPopupMultSelection() {
    return this.popupMultSelection;
  }

  /**
   * Atribui um menu pop-up para dataset.
   *
   * @param popupDataset O menu pop-up para dataset.
   */
  public void setPopupDataset(JPopupMenu popupDataset) {
    this.popupDataset = popupDataset;
  }

  /**
   * Obtem o menu pop-up de dataset.
   *
   * @return O menu pop-up de dataset.
   */
  public JPopupMenu getPopupDataset() {
    return this.popupDataset;
  }

  /**
   * Atribui um menu pop-up para diretrios.
   *
   * @param popupDir O menu pop-up para diretrios.
   */
  public void setPopupDir(JPopupMenu popupDir) {
    this.popupDir = popupDir;
  }

  /**
   * Obtem o menu pop-up de diretrios.
   *
   * @return O menu pop-up de diretrios.
   */
  public JPopupMenu getPopupDir() {
    return this.popupDir;
  }

  /**
   * Obtm a janela onde essa rvore est sendo exibida.
   *
   * @return A janela pai dessa rvore.
   */
  @Override
  public Window getWindow() {
    return this.owner;
  }

  /**
   * Obtm o ttulo da rvore.
   *
   * @return O ttulo da rvore.
   */
  @Override
  public String getTitle() {
    return this.title;
  }

  /**
   * Obtm o componente de interface que apresenta a rvore de diretrios.
   *
   * @return O componente de interface de visualizao da rvore.
   */
  public JTree getTree() {
    return this.tree;
  }

  /**
   * Limpa a seleo da rvore.
   */
  @Override
  public void clearSelection() {
    this.tree.clearSelection();
  }

  /**
   * Adiciona um ouvinte que est interessado em mudanas na seleo da rvore.
   *
   * @param listener O ouvinte a ser adicionado.
   *
   * @return true, se conseguiu adicion-lo, false, caso contrrio.
   *
   * @throws IllegalArgumentException Caso seja recebido um ouvinte nulo.
   */
  public boolean addTreeSelectionListener(
    ProjectTreeSelectionListener listener) {
    if (listener == null) {
      throw new IllegalArgumentException(
        "No  permitido adicionar um ouvinte nulo (null).");
    }
    return this.treeSelectionListenerList.add(listener);
  }

  /**
   * Remove um ouvinte que estava interessado em mudanas na seleo da rvore.
   *
   * @param listener O ouvinte a ser removido.
   *
   * @return true, se conseguiu remov-lo, false, caso contrrio.
   */
  public boolean removeTreeSelectionListener(
    ProjectTreeSelectionListener listener) {
    return this.treeSelectionListenerList.remove(listener);
  }

  /**
   * Adiciona listener de modificao do modelo da rvore.
   *
   * @param listener listener
   */
  public void addTreeModelListener(TreeModelListener listener) {
    this.tree.getModel().addTreeModelListener(listener);
  }

  /**
   * Retira listener de modificao do modelo da rvore.
   *
   * @param listener listener
   */
  public void removeTreeModelListener(TreeModelListener listener) {
    this.tree.getModel().removeTreeModelListener(listener);
  }

  /**
   * Obtem os listeners interessados em mudanas na rvore
   *
   * @return .
   */
  public Vector<ProjectTreeListener> getListeners() {
    return projectTreeListenerList;
  }

  /**
   * Seleciona o caminho escolhido do diretrio dentro da rvore de projeto.
   * Deve-se fornecer null quando se deseja selecionar a raiz do projeto. Quando
   * se deseja selecionar um diretrio dentro do proejto, o caminho a ser usado
   * NO deve incluir o diretrio com o nome do projeto (raiz).
   *
   * @param selectedPath O caminho na rvore de projeto.
   */
  public void setSelectionPath(String[] selectedPath) {
    ClientProjectFile root = this.project.getRoot();
    ClientProjectFile file;
    if (selectedPath == null) {
      file = root;
    }
    else {
      file = GetChildFromPathTask.runTask(root, selectedPath);
    }
    ProjectTreeModel treeModel = (ProjectTreeModel) this.tree.getModel();
    TreePath treePath = treeModel.getTreePath(file);
    tree.expandPath(treePath.getParentPath());
    tree.setSelectionPath(treePath);
  }

  /**
   * Seleciona o caminho escolhido do diretrio dentro da rvore de projeto
   *
   * @param selectedPath O componente que representa o caminho na rvore de
   *        projeto.
   */
  public void setSelectionPath(TreePath selectedPath) {
    tree.setSelectionPath(selectedPath);
  }

  /**
   * Obtm o projeto da rvore.
   *
   * @return O projeto.
   */
  @Override
  public CommonClientProject getProject() {
    return this.project;
  }

  /**
   * Modifica a forma como os ns da rvore so ordenados. Existem dois tipos de
   * ordenao padro: por nome (feito pelo comparador
   * {@link ProjectTreeNodeNameComparator}) e por extenso (feito pelo
   * comparador {@link ProjectTreeNodeExtensionComparator}).
   *
   * @param comparator comparador a ser usado para ordenar os ns da rvore.
   */
  public void setComparator(Comparator<ProjectTreeNode> comparator) {
    ProjectTreeModel treeModel = (ProjectTreeModel) this.tree.getModel();
    treeModel.setComparator(comparator);
  }

  /**
   * Modifica o contedo da rvore que est sendo visualizada, a partir da raiz.
   *
   * @param newProject O novo projeto a ser apresentado pela rvore.
   */
  public void setProject(CommonClientProject newProject) {

    this.clearSelection();
    this.project = newProject;
    ProjectTreeModel treeModel = (ProjectTreeModel) this.tree.getModel();
    treeModel.setProject(this.project);
    setDefaultTreeExpansionState();

    /* Persiste o projeto no histrico dos ltimos projetos abertos */
    if (project != null) {
      ProjectBasicInfo info = new ProjectBasicInfo(project.getId(), project
        .getName(), project.getUserId());
      saveProjectAsRecent(info);
    }

    notifyProjectWasChanged(this.project);
    notifyProjectWasRemoved();
  }

  /**
   * Persiste Dados do projeto na lista de projetos abertos recentemente
   *
   * @param projectBasicInfo Dados do Projeto que foi aberto
   */
  private void saveProjectAsRecent(final ProjectBasicInfo projectBasicInfo) {
    final RecentProjectsManager manager = new RecentProjectsManager();
    RemoteTask<Void> task = new RemoteTask<Void>() {
      @Override
      protected void performTask() throws Exception {
        manager.saveProjectInfo(projectBasicInfo);
      }
    };
    task.execute(owner, title, LNG.get(
      "ProjectTree.task.save.project.as.recent.msg"));
    if (!task.getStatus()) {
      System.out.println("Erro na persistncia de Projetos Recentes. Projeto: "
        + projectBasicInfo.getProjectId());
    }
  }

  /**
   * Ajusta o modo default da rovre de arquivos para deixar a raiz aberta e
   * todos os diretrios (da raiz) incialmente fechados.
   */
  private void setDefaultTreeExpansionState() {
    final ProjectTreeNode root = (ProjectTreeNode) tree.getModel().getRoot();
    if (root == null) {
      return;
    }
    final TreePath rootPath = new TreePath(root);
    if (root.getChildCount() >= 0) {
      for (Enumeration<ProjectTreeNode> e = root.children(); e
        .hasMoreElements();) {
        final TreeNode node = e.nextElement();
        final TreePath nodePath = rootPath.pathByAddingChild(node);
        tree.collapsePath(nodePath);
      }
    }
    tree.expandPath(rootPath);
  }

  /**
   * Limpa a rvore que est sendo apresentada.
   */
  public void resetProject() {
    this.setProject(null);
  }

  /**
   * Fecha o projeto corrente.
   *
   * @throws IllegalStateException Caso no exista projeto aberto.
   */
  public void closeProject() {
    if (this.project == null) {
      throw new IllegalStateException(
        "No  possvel fechar o projeto, pois no existe projeto.");
    }
    notifyProjectWasClosed(this.project);
  }

  /**
   * Determina se a rvore possui seleo simples ou seleo mltipla.
   *
   * @param singleSelection true, se a rvore possui seleo simples ou false,
   *        se possui seleo mltipla.
   */
  public void setSingleSelectionMode(boolean singleSelection) {
    int mode = singleSelection ? TreeSelectionModel.SINGLE_TREE_SELECTION
      : TreeSelectionModel.DISCONTIGUOUS_TREE_SELECTION;
    tree.getSelectionModel().setSelectionMode(mode);
  }

  /**
   * Obtem o arquivo selecionado na rvore.
   *
   * @return O arquivo selecionado. Caso no exista arquivo selecionado ou no
   *         exista projeto aberto, retornar null.
   */
  @Override
  public ClientProjectFile getSelectedFile() {
    if (project == null) {
      return null;
    }
    TreePath path = tree.getSelectionPath();
    if (path == null) {
      return null;
    }
    ProjectTreeNode treeNode = (ProjectTreeNode) path.getLastPathComponent();
    ClientProjectFile file = treeNode.getClientProjectFile();
    return file;
  }

  /**
   * Obtm os arquivos atualmente selecionados na rvore.
   *
   * @return Os arquivos selecionados na rvore. Caso no existam arquivo
   *         selecionados ou no exista projeto aberto, retornar null.
   */
  @Override
  public ClientProjectFile[] getSelectedFiles() {
    if (project == null) {
      return null;
    }
    TreePath[] path = tree.getSelectionPaths();
    if (path == null) {
      return null;
    }
    if (path.length == 0) {
      return null;
    }
    ClientProjectFile[] file = new ClientProjectFile[path.length];
    for (int i = 0; i < path.length; i++) {
      ProjectTreeNode treeNode = (ProjectTreeNode) path[i]
        .getLastPathComponent();
      file[i] = treeNode.getClientProjectFile();
    }
    return file;
  }

  /**
   *
   */
  public void setOutOfDate() {
    getProject().setOutOfDate();
    repaint();
    notifyInfoWasModified();
  }

  /**
   * Habilita/desabilita a transferncia de arquivos por drag{@code &}drop e
   * clipboard.
   *
   * @param enable true, para habilitar a transferncia, ou false, caso
   *        contrrio.
   */
  @Override
  public void enableTransferAction(boolean enable) {
    if (enable) {
      tree.setTransferHandler(new ProjectTreeTransferHandler(owner, title));
      tree.setDragEnabled(true);
    }
    else {
      tree.setDragEnabled(false);
      tree.setTransferHandler(null);
    }
    transferEnabled = enable;
    enableTransferKeyBindings(enable);
  }

  /**
   * Habilita/desabilita o mapeamento de teclas de atalho para transferncia de
   * arquivos (Ctrl-X para Recortar, Ctrl-C para Copiar e Ctrl-V para Colar).
   *
   * @param enable true, para habilitar a transferncia, ou false, caso
   *        contrrio.
   */
  private void enableTransferKeyBindings(boolean enable) {
    KeyStroke cutKey = KeyStroke.getKeyStroke("ctrl X");
    KeyStroke copyKey = KeyStroke.getKeyStroke("ctrl C");
    KeyStroke pasteKey = KeyStroke.getKeyStroke("ctrl V");
    InputMap inputMap = tree.getInputMap();
    ActionMap actionMap = tree.getActionMap();
    if (enable) {
      Action cutAction = new CommonFileCutAction(this);
      Action copyAction = new CommonFileCopyAction(this);
      Action pasteAction = new CommonFilePasteAction(this);
      inputMap.put(cutKey, cutAction.getValue(Action.NAME));
      inputMap.put(copyKey, copyAction.getValue(Action.NAME));
      inputMap.put(pasteKey, pasteAction.getValue(Action.NAME));
      actionMap.put(cutAction.getValue(Action.NAME), cutAction);
      actionMap.put(copyAction.getValue(Action.NAME), copyAction);
      actionMap.put(pasteAction.getValue(Action.NAME), pasteAction);
    }
    else {
      String actionName = (String) inputMap.get(cutKey);
      actionMap.remove(actionName);
      inputMap.remove(cutKey);
      actionName = (String) inputMap.get(copyKey);
      actionMap.remove(actionName);
      inputMap.remove(copyKey);
      actionName = (String) inputMap.get(pasteKey);
      actionMap.remove(actionName);
      inputMap.remove(pasteKey);
    }
  }

  /**
   * Ativa listeners de seleo da rvore.
   *
   * @param selectedFiles arquivos selecionados
   */
  private void fireUpdateSelection(ClientProjectFile[] selectedFiles) {
    ProjectTreeSelectionEvent treeSelectionEvent =
      new ProjectTreeSelectionEvent(this, selectedFiles);
    Iterator<ProjectTreeSelectionListener> iterator =
      this.treeSelectionListenerList.iterator();
    while (iterator.hasNext()) {
      ProjectTreeSelectionListener listener = iterator.next();
      listener.update(treeSelectionEvent);
    }
  }

  /**
   * Habilita/desabilita edio nos ns da rvore. Usado para renomear os
   * arquivos de projeto.
   *
   * @param enable true, para habilitar a edio, ou false, caso contrrio.
   */
  @Override
  public void enableRenameAction(boolean enable) {
    if (enable) {
      tree.setEditable(true);
      tree.setCellEditor(this.editor);
    }
    else {
      tree.setEditable(false);
      tree.setCellEditor(null);
    }
    renameEnabled = enable;
  }

  /**
   * Inicia a edio de um nome de um arquivo ou diretrio da rvore.
   */
  @Override
  public void startRenamingAction() {
    tree.startEditingAtPath(tree.getSelectionPath());
  }

  /**
   * Recorta os arquivos selecionados para o clipboard.
   */
  @Override
  public void startCutAction() {
    TransferHandler handler = tree.getTransferHandler();
    handler.exportToClipboard(tree, null, DnDConstants.ACTION_MOVE);
  }

  /**
   * Copia os arquivos selecionados para o clipboard.
   */
  @Override
  public void startCopyAction() {
    TransferHandler handler = tree.getTransferHandler();
    handler.exportToClipboard(tree, null, DnDConstants.ACTION_COPY);
  }

  /**
   * Transfere os arquivos que esto no clipboard.
   */
  @Override
  public void startPasteAction() {
    TransferHandler handler = tree.getTransferHandler();
    handler.importData(tree, null);
  }

  /**
   * Habilita/desabilita a remoo de arquivos usando a tecla DELETE.
   *
   * @param enable se <code>true</code>, habilita; caso contrrio, desabilita.
   */
  @Override
  public void enableDeleteAction(boolean enable) {
    KeyStroke delKey = KeyStroke.getKeyStroke("DELETE");
    if (enable) {
      Action deleteAction = new CommonFileDeleteAction(this);
      String actionName = (String) deleteAction.getValue(Action.NAME);
      tree.getInputMap().put(delKey, actionName);
      tree.getActionMap().put(actionName, deleteAction);
    }
    else {
      String actionName = (String) tree.getInputMap().get(delKey);
      tree.getActionMap().remove(actionName);
      tree.getInputMap().remove(delKey);
    }
    deleteEnabled = enable;
  }

  /**
   * Habilita/desabilita a atualizao da rvore de projetos usando a tecla F5.
   *
   * @param enable se <code>true</code>, habilita; caso contrrio, desabilita.
   */
  @Override
  public void enableRefreshAction(boolean enable) {
    KeyStroke refreshKey = KeyStroke.getKeyStroke("F5");
    if (enable) {
      Action refreshAction = new ProjectTreeUpdateAction(this);
      String actionName = (String) refreshAction.getValue(Action.NAME);
      tree.getInputMap().put(refreshKey, actionName);
      tree.getActionMap().put(actionName, refreshAction);
    }
    else {
      String actionName = (String) tree.getInputMap().get(refreshKey);
      tree.getActionMap().remove(actionName);
      tree.getInputMap().remove(refreshKey);
    }
    refreshEnabled = enable;
  }

  /**
   * Limpa a rea de clipboard.
   */
  @Override
  public void clearClipboard() {
    ProjectTreeTransferHandler handler = (ProjectTreeTransferHandler) tree
      .getTransferHandler();
    handler.clearClipboard();
  }

  /**
   * Define o filtro de visualizao da rvore.
   *
   * @param filter filtro (pode ser <code>null</code>)
   */
  public void setVisualFilter(ProjectFileFilter filter) {
    ProjectTreeModel treeModel = (ProjectTreeModel) this.tree.getModel();
    treeModel.setFilter(filter);
  }

  /**
   * Ajusta filtro de seleo.
   *
   * @param filter filtro
   */
  public void setSelectionFilter(ProjectFileFilter filter) {
    ProjectTreeSelectionModel treeSelectionModel =
      (ProjectTreeSelectionModel) this.tree.getSelectionModel();
    treeSelectionModel.setFilter(filter);
  }

  /**
   * Cria um componente JTree para a rvore de arquivos do projeto. Cria o menu
   * de popup que acompanha esse componente.
   *
   * @param visualFilter .
   * @param selectionFilter .
   */
  private void makeTree(ProjectFileFilter visualFilter,
    ProjectFileFilter selectionFilter) {
    ProjectTreeModel model;
    if (visualFilter == null) {
      model = new ProjectTreeModel(this.project);
    }
    else {
      model = new ProjectTreeModel(this.project, visualFilter);
    }
    this.tree = new JTree(model);
    this.tree.setLargeModel(true);
    if (selectionFilter == null) {
      this.tree.setSelectionModel(new ProjectTreeSelectionModel());
    }
    else {
      this.tree.setSelectionModel(new ProjectTreeSelectionModel(
        selectionFilter));
    }
    /* Permite tooltip */
    ToolTipManager.sharedInstance().registerComponent(this.tree);

    /* Muda os icones e os tooltips dos ns */
    ProjectTreeRenderer renderer = new ProjectTreeRenderer();
    this.tree.setCellRenderer(renderer);
    this.editor = new ProjectTreeCellEditor(new CommonFileRenameAction(this),
      this.tree, renderer);
    this.tree.putClientProperty("JTree.lineStyle", "Angled");
    this.tree.setShowsRootHandles(true);
    /*
     * adicionamos um listener para mudanas do modelo
     */
    model.addTreeWillChangeStructureListener(
      new ProjectTreeStructureListenerImpl(tree));
    /*
     * agora adicionamos na rvore um listener para mudanas na seleo
     */
    this.tree.addTreeSelectionListener(new TreeSelectionListener() {
      @Override
      public void valueChanged(TreeSelectionEvent e) {
        List<ClientProjectFile> selectedFilesList =
          new LinkedList<ClientProjectFile>();
        TreePath[] treePathArray = e.getPaths();
        for (int i = 0; i < treePathArray.length; i++) {
          if (e.isAddedPath(i)) {
            ProjectTreeNode selectedNode = (ProjectTreeNode) treePathArray[i]
              .getLastPathComponent();
            final ClientProjectFile selectedFile = selectedNode
              .getClientProjectFile();
            selectedFilesList.add(selectedFile);
          }
        }
        ClientProjectFile[] selectedFiles = selectedFilesList.toArray(
          new ClientProjectFile[selectedFilesList.size()]);
        ProjectTree.this.fireUpdateSelection(selectedFiles);
      }
    });

    tree.addTreeWillExpandListener(new ProjectTreeWillExpandListener(
      (ProjectTreeModel) tree.getModel()));

    // Comportamento padro; os sistemas que desejarem podem desabilitar os
    // comportamentos conforme a necessidade chamando os mtodos abaixo com
    // parmetro "false".
    enableDeleteAction(true);
    enableRefreshAction(true);
    enableRenameAction(true);
    enableTransferAction(true);
  }

  /**
   * Adiciona o MouseListener da Tree
   */
  public void addMouseListener() {
    tree.addMouseListener(new ProjectTreeMouseAdapter(this) {
      @Override
      public void doubleClickAction(ClientProjectFile file) {
        if (ProjectTree.this.doubleClickIntercepted(file)) {
          return;
        }
        fireOpenAction(file);
      }
    });
  }

  /**
   * Mtodo de interceptao de arquivo acionado (duplo-clique) na rvore de
   * projetos, para que cada sistema possa eventualmente redefinir a ao para
   * outra coisa alm de simplesmente abrir o mesmo com uma aplicao
   * correspondente. Ao retornar true, nada mais ser feito aps a
   * interceptao.
   *
   * @param file arquivo da rvore de projeto
   * @return indicativo de interceptao.
   */
  protected boolean doubleClickIntercepted(ClientProjectFile file) {
    return false;
  }

  /**
   * Adiciona um listener de projeto. Sempre que o projeto da rvore for
   * modificado, os listeners so notificados.
   *
   * @param listener Um listener de seleo a ser includo.
   */
  public void addProjectTreeListener(ProjectTreeListener listener) {
    projectTreeListenerList.add(listener);
  }

  /**
   * Remove um listener de seleo da projeto.
   *
   * @param listener Um listener de seleo a ser removido.
   */
  public void removeProjectTreeListener(ProjectTreeListener listener) {
    projectTreeListenerList.remove(listener);
  }

  /**
   * Notifica os listeners que o projeto exibido foi trocado.
   *
   * @param proj O projeto corrente.
   */
  private void notifyProjectWasChanged(CommonClientProject proj) {
    ProjectTreeListener listener;
    for (Enumeration<ProjectTreeListener> elem = projectTreeListenerList
      .elements(); elem.hasMoreElements();) {
      listener = elem.nextElement();
      listener.projectChanged(proj);
    }
  }

  /**
   * Notifica os listeners sobre o fechamento de um projeto.
   *
   * @param proj o projeto fechado.
   */
  private void notifyProjectWasClosed(CommonClientProject proj) {
    ProjectTreeListener listener;
    for (Enumeration<ProjectTreeListener> elem = projectTreeListenerList
      .elements(); elem.hasMoreElements();) {
      listener = elem.nextElement();
      listener.projectClosed(proj);
    }
  }

  /**
   * Notifica os listeners sobre a remocao do projeto exibido.
   */
  private void notifyProjectWasRemoved() {
    ProjectTreeListener listener;
    for (Enumeration<ProjectTreeListener> elem = projectTreeListenerList
      .elements(); elem.hasMoreElements();) {
      listener = elem.nextElement();
      listener.projectRemoved();
    }
  }

  /**
   * Notifica os listeners sobre a mudana das informaes de um projeto.
   */
  private void notifyInfoWasModified() {
    ProjectTreeListener listener;
    for (Enumeration<ProjectTreeListener> elem = projectTreeListenerList
      .elements(); elem.hasMoreElements();) {
      listener = elem.nextElement();
      listener.projectInfoModified();
    }
  }

  /**
   * Indica se existem arquivos na rvore marcados para copy.
   *
   * @return true, caso existam arquivos na rvore marcados para copy, ou false,
   *         caso no existam.
   */
  public boolean hasFileToPaste() {
    ProjectTreeTransferHandler transferHandler =
      (ProjectTreeTransferHandler) tree.getTransferHandler();
    return (transferHandler != null && transferHandler
      .hasTransferableInClipboard());
  }

  /**
   * Consulta um path para o arquivo.
   *
   * @param file arquivo
   * @return path
   */
  private TreePath getPathToRoot(ClientProjectFile file) {
    ProjectTreeModel treeModel = (ProjectTreeModel) this.tree.getModel();
    return treeModel.getTreePath(file);
  }

  /**
   * Seleciona o arquivo ou diretrio especificado na rvore
   *
   * @param file arquivo a ser selecionado
   */
  public void setSelectedFile(ClientProjectFile file) {
    setSelectionPath(this.getPathToRoot(file));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int getSelectionCount() {
    return tree.getSelectionCount();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void repaint() {
    tree.repaint();
  }

  /**
   * Liberao da visualizao da rvore. Descadastra o observador de alteraes
   * da rvore do projeto.
   */
  public void release() {
    this.setProject(null);
  }

  /**
   * Responsvel pela edio dos ns da rvore (renomear).
   */
  private class ProjectTreeCellEditor extends DefaultTreeCellEditor {
    /**
     * {@inheritDoc}
     */
    @Override
    public boolean isCellEditable(EventObject event) {
      TreePath path = this.tree.getSelectionPath();
      if (path == null) {
        return false;
      }
      ProjectTreeNode treeNode = (ProjectTreeNode) path.getLastPathComponent();
      ClientProjectFile file = treeNode.getClientProjectFile();
      if (file.isUnderConstruction()) {
        return false;
      }
      if (file.getParent() == null) {
        return false;
      }
      return super.isCellEditable(event);
    }

    /**
     * Construtor.
     *
     * @param action Classe efetivamente responsvel por renomear o arquivo.
     * @param tree rvore  qual pertence o n sendo editado.
     * @param renderer Classe responsvel por desenhar os cones da rvore.
     */
    ProjectTreeCellEditor(CommonFileRenameAction action, JTree tree,
      DefaultTreeCellRenderer renderer) {
      this(action, tree, renderer, null);
    }

    /**
     * Construtor. Adiciona listener que vai renomear o arquivo no servidor aps
     * o usurio ter terminado a sua edio.
     *
     * @param action Classe efetivamente responsvel por renomear o arquivo.
     * @param tree rvore  qual pertence o n sendo editado.
     * @param renderer Classe responsvel por desenhar os cones da rvore.
     * @param editor Editor de ns
     */
    ProjectTreeCellEditor(final CommonFileRenameAction action, final JTree tree,
      DefaultTreeCellRenderer renderer, TreeCellEditor editor) {
      super(tree, renderer, editor);
      addCellEditorListener(new CellEditorListener() {
        /**
         * Dispara a ao de renomear o arquivo no servidor.
         */
        private void rename() {
          TreePath path = tree.getSelectionPath();
          if (path == null) {
            return;
          }
          ProjectTreeNode selectedNode = (ProjectTreeNode) path
            .getLastPathComponent();
          ClientProjectFile file = selectedNode.getClientProjectFile();
          String oldName = file.getName();
          String newName = (String) getCellEditorValue();
          if (oldName.equals(newName)) {
            return;
          }
          action.rename(file, newName);
        }

        /**
         * Disparado se o usurio selecionar outro arquivo sem pressionar a
         * tecla ENTER.
         *
         * @param event .
         */
        @Override
        public void editingCanceled(ChangeEvent event) {
          rename();
        }

        /**
         * Disparado se o usurio pressionar a tecla ENTER.
         *
         * @param event .
         */
        @Override
        public void editingStopped(ChangeEvent event) {
          rename();
        }
      });
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected TreeCellEditor createTreeCellEditor() {
      Border aBorder = UIManager.getBorder("Tree.editorBorder");
      /*
       * criamos um novo campo de texto onde o mtodo selectAll()  redefinido
       * para selecionar apenas at antes do ltimo '.'
       */
      DefaultTextField textField = new DefaultTextField(aBorder) {
        /**
         * {@inheritDoc}
         */
        @Override
        public void selectAll() {
          Document doc = getDocument();
          if (doc != null) {
            setCaretPosition(0);
            moveCaretPosition(getLastPositionBeforeDot());
          }
        }

        /**
         * Retorna a ltima posio a conter caracter antes da extenso do
         * arquivo.
         */
        private int getLastPositionBeforeDot() {
          String text = getText();
          int pos = text.lastIndexOf('.');
          if (pos == -1 || pos == 0) {
            /*
             * o nome no possui '.' ou este  o 1o caracter (arquivo oculto) e
             * no existe extenso; nestes casos, selecionamos tudo
             */
            return text.length();
          }
          return pos;
        }
      };

      return new DefaultCellEditor(textField);
    }
  }

  /**
   * Cria a rvore de projetos. A rvore estar vazia, pois no representar
   * nenhum projeto. Para configurar um projeto, utilize o mtodo
   * {@link #setProject(CommonClientProject)}.
   *
   * @param owner A janela pai onde a rvore ser exibida.
   * @param title O ttulo da janela pai.
   */
  protected ProjectTree(Window owner, String title) {
    this(owner, title, null, null, null);
  }

  /**
   * Cria a rvore de projetos que representar o projeto recebido.
   *
   * @param owner A janela pai onde a rvore ser exibida.
   * @param title O ttulo da janela pai.
   * @param project O projeto que ser representado.
   */
  protected ProjectTree(Window owner, String title,
    CommonClientProject project) {
    this(owner, title, project, null, null);
  }

  /**
   * Cria a rvore de projetos que representar o projeto recebido. S sero
   * apresentados na rvore os arquivos do projeto que so aceitos pelo filtro.
   *
   * @param owner A janela pai onde a rvore ser exibida.
   * @param title O ttulo da janela pai.
   * @param project O projeto que ser representado.
   * @param visualFilter O filtro de visualizao.
   */
  protected ProjectTree(Window owner, String title, CommonClientProject project,
    ProjectFileFilter visualFilter) {
    this(owner, title, project, visualFilter, null);
  }

  /**
   * Retorna o buscador (por inferncia) de tipo de arquivo que, por default, 
   * nulo ({@code null}); mas pode ser redefinido pelos sistemas para seus tipos
   * de arquivos.
   *
   * @return o buscador
   */
  public FileTypeFinder getFileTypeFinder() {
    return null;
  }

  /**
   * Cria a rvore de projetos que representar o projeto recebido. S sero
   * apresentados na rvore os arquivos do projeto que so aceitos pelo filtro
   * de visualizao. S podero ser selecionados os arquivos do projeto que
   * estejam sendo exibidos e que sejam aceitos pelo filtro de seleo.
   *
   * @param owner A janela pai onde a rvore ser exibida.
   * @param title O ttulo da janela pai.
   * @param project O projeto que ser representado.
   * @param visualFilter O filtro de visualizao.
   * @param selectionFilter O filtro de seleo.
   *
   * @throws IllegalArgumentException Caso a janela ou o ttulo da janela sejam
   *         nulos (null).
   */
  protected ProjectTree(Window owner, String title, CommonClientProject project,
    ProjectFileFilter visualFilter, ProjectFileFilter selectionFilter) {
    if (owner == null) {
      throw new IllegalArgumentException(
        "A janela pai da rvore no pode ser nula.");
    }
    if (title == null) {
      throw new IllegalArgumentException(
        "O ttulo da janela pai da rvore no pode ser nulo.");
    }
    this.owner = owner;
    this.title = title;
    this.project = project;
    this.treeSelectionListenerList = new LinkedList<>();
    this.projectTreeListenerList = new Vector<>();
    this.makeTree(visualFilter, selectionFilter);
  }

}
