/*
 * Created on Jun 17, 2004
 * 
 * $Id$
 */
package csbase.client.project.action;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import javax.swing.JOptionPane;

import tecgraf.javautils.core.lng.FormatUtils;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.StandardDialogs;
import csbase.client.desktop.RemoteTask;
import csbase.client.desktop.Task;
import csbase.client.externalresources.ExternalResources;
import csbase.client.externalresources.LocalFile;
import csbase.client.project.ExportTask;
import csbase.client.project.ProjectFileContainer;
import csbase.client.remote.srvproxies.NotificationProxy;
import csbase.client.util.StandardErrorDialogs;
import csbase.exception.CSBaseException;
import csbase.logic.ClientProjectFile;
import csbase.logic.User;

/**
 * Ao para exportao de arquivos, a partir do projeto no servidor para o
 * sistema de arquivos do cliente. Esta ao tenta primeiro acessar diretamente
 * o sistema de arquivos, utilizando o <code>JFileChooser</code>. Caso no
 * consiga (no caso dos jars no estarem assinados), utiliza a API JNLP.
 * 
 * @author Leonardo Barros
 */
public abstract class CommonFileExportAction extends CommonProjectAction {

  /**
   * Mensagem de erro para o caso em que se tentou exportar um diretrio via
   * JNLP.
   */
  protected static final String JNLP_CANT_EXPORT_DIR = "JNLP_CANT_EXPORT_DIR";

  /** Nome da ao, para ser exibida em botes ou menus */
  protected static final String ACTION_NAME = "PRJ_EXPORT_FILE";

  /** Mensagem para erro de sistema. No  possvel continuar. */
  protected static final String CRITICAL_ERROR =
    "PRJ_PROJECT_FILE_IMPORT_CRITICAL_ERROR";

  /** Mensagem de aviso para erro na exportao de um nico arquivo */
  protected static final String SINGLE_FILE_ERROR = "EXPORT_SINGLE_FILE_ERROR";

  /** Mensagem de aviso para arquivo de destino j existente */
  protected static final String FILE_EXISTS =
    "PRJ_PROJECT_FILE_EXPORT_FILE_EXISTS";

  /** Mensagem de aviso para arquivo de origem vazio. */
  protected static final String EMPTY_FILE =
    "PRJ_PROJECT_FILE_EXPORT_EMPTY_FILE";

  /** Mensagem de notificao do incio da exportao */
  protected static final String EXPORT_IN_PROGRESS_MSG =
    "PRJ_PROJECT_FILE_EXPORT_IN_PROGRESS";

  /** Mensagem para exportao efetuada com sucesso */
  protected static final String SUCCESS = "PRJ_PROJECT_FILE_EXPORT_SUCCESS";

  /** Mensagem para exportao via JNLP efetuada com sucesso */
  protected static final String SUCCESS_JNLP =
    "PRJ_PROJECT_FILE_EXPORT_SUCCESS_JNLP";

  /** Ttulo do dilogo de exportao */
  protected static final String TITLE = "PRJ_PROJECT_FILE_EXPORT_TITLE";

  /** Mensagem para atualizao dos dados do arquivo. */
  protected static final String UPDATE_FILE = "PRJ_PROJECT_FILE_UPDATE_FILE";

  /**
   * Cria a ao.
   * 
   * @param container objeto contendo arquivos de projeto.
   */
  public CommonFileExportAction(ProjectFileContainer container) {
    super(container);
  }

  /**
   * Mtodo de aviso de erros de sistema.
   * 
   * @param msg mensagem a ser exibida.
   */
  @Override
  protected void showError(String msg) {
    showError(LNG.get(TITLE), msg);
  }

  /**
   * Mtodo de aviso de erros de sistema.
   * 
   * @param msg mensagem a ser exibida.
   * @param ex exceo detectada.
   */
  protected void showError(String msg, Exception ex) {
    StandardErrorDialogs.showErrorDialog(getWindow(), LNG.get(TITLE), msg, ex);
  }

  /**
   * Obtm o volume total da transferncia, em Kilobytes.
   * 
   * @param sizeInBytes volume total da transferncia, em bytes.
   * 
   * @return volume total dos arquivos transferidos (em KBytes).
   */
  protected double getTotalSizeInKBytes(long sizeInBytes) {
    return sizeInBytes / 1024;
  }

  /**
   * Obtm o tempo decorrido, em segundos, desde o incio da execuo.
   * 
   * @param start tempo de incio da execuo (milissegundos desde 1970).
   * 
   * @return tempo decorrido desde o incio da execuo (em segundos).
   */
  protected double getElapsedTime(long start) {
    return (double) (System.currentTimeMillis() - start) / 1000;
  }

  /**
   * Obtm a taxa de transferncia, em Kbytes por segundo.
   * 
   * @param totalSize volume tranferido (em KBytes).
   * @param elapsedTime tempo decorrido (em segundos).
   * 
   * @return taxa de transferncia (em Kbytes/segundo).
   */
  protected double getTransferRate(double totalSize, double elapsedTime) {
    return totalSize / elapsedTime;
  }

  /**
   * Seleciona, dentre os arquivos selecionados, os mais externos, isto , que
   * possivelmente contenham outros selecionados, para evitar redundncias na
   * exportao, j que diretrios com todo seu contedo so exportados.
   * 
   * @param selectedFiles todos os arquivos selecionados.
   * 
   * @return arquivos selecionados mais externos.
   */
  protected Collection<ClientProjectFile> selectTopFiles(
    ClientProjectFile[] selectedFiles) {
    List<ClientProjectFile> topFiles = new ArrayList<ClientProjectFile>();
    NEXT_CANDIDATE: for (int i = 0; i < selectedFiles.length; i++) {
      TEST_PARENTHOOD: for (int j = 0; j < selectedFiles.length; j++) {
        if (i == j) {
          continue TEST_PARENTHOOD;
        }
        if (isDescendantOf(selectedFiles[i], selectedFiles[j])) {
          continue NEXT_CANDIDATE;
        }
      }
      topFiles.add(selectedFiles[i]);
    }
    return topFiles;
  }

  /**
   * Verifica se o primeiro arquivo  descendente do segundo, isto , se est
   * localizado debaixo do outro na rvore de diretrios.
   * 
   * @param descendant possvel descendente.
   * @param ancestor possvel ancestral.
   * 
   * @return <code>true</code> se o primeiro arquivo for descendente do segundo.
   */
  protected boolean isDescendantOf(ClientProjectFile descendant,
    ClientProjectFile ancestor) {
    // #TODO Verificar se este mtodo poderia migrar para ProjectFile
    String[] descPath = descendant.getPath();
    String[] ancPath = ancestor.getPath();
    if (descPath.length <= ancPath.length) {
      return false;
    }
    for (int i = 0; i < ancPath.length; i++) {
      if (!(descPath[i].equals(ancPath[i]))) {
        return false;
      }
    }
    return true;
  }

  /**
   * Exibe um dilogo para confirmar com o usurio se este deseja sobrescrever o
   * arquivo existente.
   * 
   * @param targetFile diretrio de destino.
   * 
   * @return <code>true</code> se o usurio confirmar, <code>false</code> caso
   *         contrrio.
   */
  protected boolean confirmOverwrite(File targetFile) {
    String msg =
      MessageFormat.format(LNG.get(FILE_EXISTS),
        new Object[] { targetFile.getAbsolutePath() });
    int answer =
      StandardDialogs.showYesNoDialog(getWindow(), LNG.get(TITLE), msg);
    return (answer == JOptionPane.YES_OPTION);
  }

  /**
   * Obtm o nome da ao.
   * 
   * @return nome da ao.
   */
  @Override
  public String getName() {
    return LNG.get(ACTION_NAME);
  }

  /**
   * Exibe uma mensagem avisando sobre a tentativa de exportar um arquivo vazio
   * e solicita ao usurio que confirme a transferncia.
   * 
   * @param sourceFile arquivo vazio.
   * 
   * @return <code>true</code> se o usurio confirmar que deseja continuar a
   *         transferncia, mesmo sabendo que o arquivo est vazio.
   */
  protected boolean confirmEmptyFileExport(ClientProjectFile sourceFile) {
    String msg =
      MessageFormat.format(LNG.get(EMPTY_FILE),
        new Object[] { sourceFile.getStringPath() });
    int answer =
      StandardDialogs.showYesNoDialog(getWindow(), LNG.get(TITLE), msg);
    return (answer == JOptionPane.YES_OPTION);
  }

  /**
   * Obtm o tamanho de um arquivo diretamente do servidor.
   * 
   * @param sourceFile - arquivo
   * @return tamanho do arquivo ou -1 se houve alguma falha na comunicao com o
   *         servidor
   */
  protected long getFileSize(final ClientProjectFile sourceFile) {
    /*
     * precisamos fazer um refresh no servidor para obter dados atuais
     */
    Task<Void> checkTask = new RemoteTask<Void>() {
      @Override
      protected void performTask() throws Exception {
        sourceFile.updateInfo();
      }
    };
    if (!checkTask.execute(getWindow(), LNG.get(TITLE), LNG.get(UPDATE_FILE))) {
      return -1L;
    }
    return sourceFile.size();
  }

  /**
   * Transfere um arquivo da rea de projetos para o disco local. Os paths de
   * origem e destino j foram devidamente configurados.
   * 
   * @param sourceFile - referncia no cliente para o arquivo a ser exportado
   * @param targetFile - arquivo de destino no disco local
   * @return flag indicando se a transferncia foi bem-sucedida
   */
  protected boolean transferFile(final ClientProjectFile sourceFile,
    final File targetFile) {
    long startTime = System.currentTimeMillis();
    ExportTask exportTask = new ExportTask(targetFile, sourceFile, getWindow());
    String msg =
      MessageFormat.format(LNG.get(EXPORT_IN_PROGRESS_MSG),
        new Object[] { targetFile.getName() });
    if (exportTask.execute(getWindow(), LNG.get(TITLE), msg, false, true)) {
      double transferSize = getTotalSizeInKBytes(exportTask.getTotalSize());
      double elapsedTime = getElapsedTime(startTime);
      double transferRate = getTransferRate(transferSize, elapsedTime);
      msg =
        MessageFormat.format(LNG.get(SUCCESS), sourceFile.getName(),
          FormatUtils.formatNumber(transferSize, 1),
          FormatUtils.formatNumber(elapsedTime, 1),
          FormatUtils.formatNumber(transferRate, 1));
      Object[] ids = new Object[] { User.getLoggedUser().getId() };
      NotificationProxy.notifyTo(ids, msg, false, false);
    }
    return true;
  }

  /**
   * Transfere um arquivo usando JNLP. Diretrios no podem ser transferidos
   * desta forma, e geram erro.
   * 
   * @param selectedFile - arquivo a ser transferido
   * @return flag indicando se a transferncia foi bem sucedida
   */
  protected boolean transferFileUsingJNLP(ClientProjectFile selectedFile) {
    if (!ExternalResources.getInstance().isEnabled()) {
      showError(LNG.get(CRITICAL_ERROR));
      return false;
    }
    if (selectedFile.isDirectory()) {
      /*
       * TODO no poderamos criar o dir no cliente e depois transferir os
       * arquivos um por vez?
       */
      showError(LNG.get(JNLP_CANT_EXPORT_DIR));
      return false;
    }
    InputStream projectFileInputStream = null;
    try {
      projectFileInputStream = selectedFile.getInputStream();
      LocalFile transferredFile =
        ExternalResources.getInstance().saveFileDialog(null, null,
          projectFileInputStream, selectedFile.getName());
      if (transferredFile != null) {
        double transferSize = getTotalSizeInKBytes(selectedFile.size());
        String msg =
          MessageFormat.format(LNG.get(SUCCESS_JNLP), selectedFile.getName(),
            FormatUtils.formatNumber(transferSize, 1));
        Object[] ids = new Object[] { User.getLoggedUser().getId() };
        NotificationProxy.notifyTo(ids, msg, false, false);
      }
      return true;
    }
    catch (CSBaseException ex3) {
      showError(LNG.get(SINGLE_FILE_ERROR), ex3);
    }
    catch (IOException ex4) {
      showError(LNG.get(SINGLE_FILE_ERROR), ex4);
    }
    finally {
      try {
        if (projectFileInputStream != null) {
          projectFileInputStream.close();
        }
      }
      catch (IOException ex5) {
        ex5.printStackTrace();
      }
    }
    return false;
  }
}
