/*
 * FileTypeChangeDialog.java
 * 
 * $Author$ $Revision$ - $Date: 2007-10-10 15:45:37 -0300
 * (Wed, 10 Oct 2007) $
 */
package csbase.client.project.action;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.FlowLayout;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextArea;
import javax.swing.JTextField;
import javax.swing.SwingConstants;

import csbase.client.desktop.DesktopComponentDialog;
import csbase.client.project.ClientProjectFileProperties;
import csbase.client.project.ProjectFileTypeComboBox;
import csbase.client.project.tasks.ChangeFileTypeTask;
import csbase.client.util.ClientUtilities;
import csbase.logic.ClientProjectFile;
import csbase.logic.CommonClientProject;
import csbase.logic.ProjectFileType;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GUIUtils;

/**
 * A classe <code>FileTypeChangeDialog</code> modela uma janela que permite ao
 * usurio modificar o tipo de um <code>ProjectFile</code>.
 */
public class FileTypeChangeDialog {

  /**
   * Dilogo principal.
   */
  private DesktopComponentDialog dialog = null;

  /**
   * Janela a partir do qual esse dilogo foi chamado.
   */
  private Window owner = null;

  /**
   * O boto default que fecha o dilogo.
   */
  private JButton defaultButton;

  /**
   * Projeto onde est o arquivo.
   */
  private CommonClientProject project;

  /**
   * O arquivo cujo tipo vai ser modificado.
   */
  private ClientProjectFile file;

  /**
   * Campo de tipos de arquivo.
   */
  private ProjectFileTypeComboBox fileTypeList;

  /**
   * Mostra a janela que permite ao usurio modificar o tipo do arquivo.
   * 
   * @param owner A janela a partir do qual esse dilogo foi chamado.
   * @param project O projeto no qual est o arquivo.
   * @param file O arquivo cujo tipo vai ser alterado.
   */
  public static void show(Window owner, CommonClientProject project,
    ClientProjectFile file) {
    new FileTypeChangeDialog(owner, project, file);
  }

  /**
   * Cria o dilogo.
   * 
   * @return .
   */
  private DesktopComponentDialog makeDialog() {
    /* Cria um dilogo no modal */
    DesktopComponentDialog d = new DesktopComponentDialog(owner, getTitle());
    d.setModal(false);
    /* Cria o painel principal do dilogo */
    JPanel mainPanel = new JPanel(new BorderLayout());
    mainPanel.add(makePropertiesPanel(d.getBackground()));
    mainPanel.add(makeButtonPanel(), BorderLayout.SOUTH);
    d.getContentPane().add(mainPanel);
    d.pack();
    d.center(owner);
    d.getRootPane().setDefaultButton(defaultButton);
    return d;
  }

  /**
   * Cria o painel com as propriedades do arquivo.
   * 
   * @param background A cor de fundo usada pelos componentes.
   * 
   * @return o painel.
   */
  private JPanel makePropertiesPanel(Color background) {
    ImageIcon icon = ClientProjectFileProperties.getImageIcon(this.file);
    ProjectFileType type = ProjectFileType.getFileType(file.getType());

    /* Campo com o nome do arquivo */
    JTextField nameTextField = new JTextField(file.getName());
    nameTextField.setEditable(false);

    /* Campo com a localizao */
    String fileLocalization = buildLocalization();
    JTextField pathTextField = new JTextField(fileLocalization);
    pathTextField.setEditable(false);

    /* Tipo atual do arquivo */
    JLabel typeLabel = new JLabel(icon, SwingConstants.LEFT);
    JTextArea typeTextArea = new JTextArea(" " + type.getDescription());
    typeTextArea.setBackground(background);
    typeTextArea.setEditable(false);
    JPanel typePanel = new JPanel(new FlowLayout(FlowLayout.LEADING, 0, 0));
    typePanel.add(typeLabel);
    typePanel.add(typeTextArea);

    /* Tipos de arquivos */
    ProjectFileTypeComboBox.Mode mode = file.isDirectory()
      ? ProjectFileTypeComboBox.Mode.DIRECTORY_ONLY
      : ProjectFileTypeComboBox.Mode.FILE_ONLY;
    fileTypeList = new ProjectFileTypeComboBox(mode, false);
    if (type != null) {
      fileTypeList.selectTypeCode(type.getCode());
    }
    fileTypeList.setRenderer(new ProjectFileTypeComboBoxRenderer());

    /* Monta o panel usando grid layout */
    JComponent[][] rows = { { new JLabel(LNG.get("PRJ_FILE_NAME")),
        nameTextField }, { new JLabel(LNG.get("PRJ_FILE_LOCALIZATION")),
            pathTextField }, { new JLabel(LNG.get("PRJ_FILE_TYPE")),
                typePanel }, { new JLabel(LNG.get("PRJ_FILE_NEW_TYPE")),
                    fileTypeList }, };
    return GUIUtils.createBasicGridPanel(rows);
  }

  /**
   * Cria o painel com os botes.
   * 
   * @return .
   */
  private JPanel makeButtonPanel() {
    JPanel panel = new JPanel();
    defaultButton = new JButton(LNG.get("PRJ_FILE_TYPE_CHANGE_CONFIRMATION"));
    panel.add(defaultButton);
    defaultButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        changeType();
      }
    });

    JButton cancelButton = new JButton(LNG.get("PRJ_CANCEL"));
    panel.add(cancelButton);
    cancelButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        dialog.close();
      }
    });

    JButton[] allButtons = { defaultButton, cancelButton };
    ClientUtilities.adjustEqualSizes(allButtons);
    return panel;
  }

  /**
   * Informa o ttulo que vai ser utilizado na janela de exibio.
   * 
   * @return O ttulo da janela.
   */
  protected String getTitle() {
    return LNG.get("PRJ_PROJECT_FILE_CHANGE_TYPE_TITLE");
  }

  /**
   * Consulta aos dados digitados pelo usurio.
   * 
   * @return Novo tipo de arquivo.
   */
  private ProjectFileType getNewType() {
    String code = fileTypeList.getSelectedTypeCode();
    return ProjectFileType.getFileType(code);
  }

  /**
   * Muda o tipo do arquivo.
   */
  private void changeType() {
    final ProjectFileType newType = getNewType();
    ChangeFileTypeTask.runTask(dialog, file, newType);
    dialog.close();
  }

  /**
   * Constri a representao de onde se encontra um determinado arquivo. A
   * String resultante possui a seguinte forma: [nome-do-projeto]:
   * caminho-dos-diretrios
   * 
   * @return A string que representa a localizao de um arquivo na rvore do
   *         projeto.
   */
  private String buildLocalization() {
    String path = "[" + project.getName() + "]" + ": ";
    String[] filePath = file.getPath();
    for (int i = 1; i < (filePath.length - 1); i++) {
      path += (filePath[i] + "/");
    }
    return path;
  }

  /**
   * Constri o dilogo.
   * 
   * @param owner A janela pai a partir da qual esse dilogo  chamado.
   * @param project O projeto no qual est o arquivo.
   * @param file O arquivo cujo tipo vai ser alterado.
   */
  private FileTypeChangeDialog(Window owner, CommonClientProject project,
    ClientProjectFile file) {
    this.owner = owner;
    this.project = project;
    this.file = file;
    dialog = makeDialog();
    dialog.setVisible(true);
  }
}
