/**
 * ProjectAction.java
 * 
 * $Id$
 */
package csbase.client.project.action;

import java.awt.Window;
import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.ClipboardOwner;
import java.awt.datatransfer.Transferable;
import java.awt.event.ActionEvent;

import javax.swing.AbstractAction;
import javax.swing.Action;

import tecgraf.javautils.core.lng.LNG;
import csbase.client.ClientServerManager;
import csbase.client.desktop.DesktopFrame;
import csbase.client.desktop.RemoteTask;
import csbase.client.project.ProjectTree;
import csbase.client.project.ProjectTreeAdapter;
import csbase.client.remote.ClientProjectObserver;
import csbase.logic.ClientProjectFile;
import csbase.logic.CommonClientProject;
import csbase.logic.Session;
import csbase.logic.UserProjectInfo;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.ProjectServiceInterface;

/**
 * A classe <code>ProjectAction</code> modela as aes que um usurio pode
 * efetuar sobre um <code>Project</code> ou sobre um <code>ProjectFile</code>.
 */
public abstract class ProjectAction extends AbstractAction implements
  ClipboardOwner {
  /** A instncia da rvore de projetos associada a essa ao. */
  private ProjectTree treeView;

  /**
   * Obtm a janela sobre a qual essa ao atua.
   * 
   * @return .
   */
  public Window getWindow() {
    return treeView.getWindow();
  }

  /**
   * Obtm o ttulo da janela sobre a qual essa ao atua.
   * 
   * @return .
   */
  public String getTitle() {
    return treeView.getTitle();
  }

  /**
   * Obtm a rvore de projeto sobre a qual essa ao atua.
   * 
   * @return .
   */
  public ProjectTree getProjectTree() {
    return treeView;
  }

  /**
   * Obtm o projeto sobre o qual essa ao atua.
   * 
   * @return .
   */
  public CommonClientProject getCurrentProject() {
    return treeView.getProject();
  }

  /**
   * Altera o projeto sobre o qual essa ao atua.
   * 
   * @param project .
   * 
   * @throws Exception em caso de falha.
   */
  public void setCurrentProject(CommonClientProject project) throws Exception {
    if (project == null) {
      return;
    }
    CommonClientProject currentProject = getCurrentProject();
    if (currentProject != null) {
      currentProject.uninstallObserver();
    }
    project.installObserver(new ClientProjectObserver(project));
    treeView.setProject(project);
    setProjectSessionProperty(project);
  }

  /**
   * Atualiza propriedade da sesso corrente com o projeto criado.
   * 
   * @param project O projeto recm criado.
   */
  private void setProjectSessionProperty(final CommonClientProject project) {
    final RemoteTask<Void> task = new RemoteTask<Void>() {
      @Override
      protected void performTask() throws Exception {
        final ClientServerManager clientServerManager =
          ClientServerManager.getInstance();
        final Session session = clientServerManager.getSession();
        final Object key = session.getKey();
        final Object userId = project.getUserId();
        final Object projectId = project.getId();
        final String projectName = project.getName();
        ClientRemoteLocator.server.setServerSessionProperty(key,
          ProjectServiceInterface.CURRENT_PROJECT_SESSION_PROPERTY,
          new UserProjectInfo(projectId, projectName, userId));
      }
    };

    task.execute(DesktopFrame.getInstance().getView(), getTitle(),
      LNG.get("ProjectAction.current.session.project.property"));
  }

  /**
   * Fecha o projeto sobre o qual essa ao atua.
   */
  public void closeProject() {
    treeView.closeProject();
    try {
      getCurrentProject().uninstallObserver();
    }
    catch (Exception e) {
      e.printStackTrace();
    }
  }

  /**
   * Obtm o arquivo sobre o qual essa ao atua.
   * 
   * @return .
   */
  public ClientProjectFile getSelectedFile() {
    return treeView.getSelectedFile();
  }

  /**
   * Obtm os arquivos sobre os quais essa ao atua.
   * 
   * @return .
   */
  public ClientProjectFile[] getSelectedFiles() {
    return treeView.getSelectedFiles();
  }

  /**
   * Limpa a rea de clipboard.
   */
  public void clearClipboard() {
    treeView.clearClipboard();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void lostOwnership(Clipboard clipboard, Transferable contents) {
  }

  /**
   * Habilita a ao se houver um projeto aberto na rvore.
   */
  public void enableIfProjectOpened() {
    treeView.addProjectTreeListener(new ProjectTreeAdapter() {
      @Override
      public void projectChanged(CommonClientProject project) {
        setEnabled(project != null);
      }
    });
  }

  /**
   * Habilita a selecao de projetos depois que remove algum.
   */
  public void enableIfProject() {
    treeView.addProjectTreeListener(new ProjectTreeAdapter() {
      @Override
      public void projectRemoved() {
        setEnabled(DesktopFrame.getInstance().userHasProject());
      }
    });
  }

  /**
   * Executa essa ao.
   * 
   * @param event .
   */
  @Override
  public abstract void actionPerformed(ActionEvent event);

  /**
   * Obtm o nome da ao.
   * 
   * @return .
   */
  public abstract String getName();

  /**
   * Constri uma ao.
   * 
   * @param treeView .
   */
  public ProjectAction(ProjectTree treeView) {
    this.treeView = treeView;
    putValue(Action.NAME, getName());
  }
}
