/*
 * NotificationProxy.java
 * 
 * @author Andr Luiz Clinio $Revision$ - $Date: 2009-07-29 16:35:57
 * -0300 (Wed, 29 Jul 2009) $
 */
package csbase.client.remote.srvproxies;

import java.rmi.RemoteException;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Observer;

import javax.swing.JFrame;

import tecgraf.javautils.core.lng.LNG;
import csbase.client.desktop.DesktopFrame;
import csbase.client.desktop.DesktopTask;
import csbase.client.kernel.ClientException;
import csbase.client.remote.NotificationObserver;
import csbase.client.util.StandardErrorDialogs;
import csbase.logic.User;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.NotificationServiceInterface;

/**
 * Classe que modela um proxy client/NotificationService.
 * 
 * @author Andr Luiz Clinio
 */
public class NotificationProxy {
  /** Hash de usurios */
  private static Hashtable<Object, NotificationObserver> userHash =
    new Hashtable<Object, NotificationObserver>();

  /**
   * Remove observer do usurio.
   * 
   * @param userId .
   * 
   * @throws RemoteException em caso de falha RMI.
   */
  static void killUserObserver(Object userId) throws RemoteException {
    NotificationObserver observer = userHash.get(userId);
    if (observer != null) {
      observer.decRefs();
      if (observer.hasNoRefs()) {
        userHash.remove(userId);
        observer.uninstall();
      }
    }
  }

  /**
   * Busca de um observer para um usurio.
   * 
   * @param userId .
   * 
   * @return .
   * 
   * @throws RemoteException em caso de falha RMI.
   */
  static NotificationObserver getUserObserver(Object userId)
    throws RemoteException {
    NotificationObserver observer = userHash.get(userId);
    if (observer == null) {
      observer = new NotificationObserver();
      userHash.put(userId, observer);
    }
    observer.incRefs();
    return observer;
  }

  /**
   * Mtodo de busca do servio
   * 
   * @return o servio de notificao.
   */
  private static NotificationServiceInterface getService() {
    return ClientRemoteLocator.notificationService;
  }

  /**
   * Mtodo de aviso de erro
   * 
   * @param msg mensagem a ser exibida.
   */
  private static void showError(String msg) {
    final DesktopFrame mainFrame = DesktopFrame.getInstance();
    if (mainFrame == null) {
      return;
    }
    JFrame jFrame = mainFrame.getDesktopFrame();
    if (jFrame == null) {
      return;
    }
    StandardErrorDialogs.showErrorDialog(jFrame,
      LNG.get("notification.check.title"), msg);
  }

  /**
   * Mtodo para auto-notifificao do usurio.
   * 
   * @param content contedo da mensagem
   * 
   * @return flag booleano de sucesso da operao
   */
  public static boolean autoNotify(String content) {
    User loggedUser = User.getLoggedUser();
    if (loggedUser != null) {
      Object[] ids = new Object[] { loggedUser.getId() };
      return notifyTo(ids, content, false, false);
    }
    return false;
  }

  /**
   * Envia uma notificao textual para uma lista de usurios. Se a lista de
   * usurios for nula, a mensagem  enviada para todos os usurios.
   * 
   * @param userIds Usurios destino.
   * @param content Contedo da notificao.
   * @param mustPopUp Indica se a mensagem deve ser exibida em um dilogo.
   * @param volFlag Indica se a mensagem  voltil.
   * @return flag booleano de sucesso da operao
   */
  public static boolean notifyTo(final Object[] userIds, final String content,
    final boolean mustPopUp, final boolean volFlag) {
    final DesktopFrame mainFrame = DesktopFrame.getInstance();
    final NotificationServiceInterface service = getService();
    if ((mainFrame == null) || (service == null)) {
      showError(LNG.get("notification.no.server.error"));
      return false;
    }
    DesktopTask task = new DesktopTask() {
      @Override
      public void performTask() throws Exception {
        service.notifyTo(userIds, content, mustPopUp, volFlag);
      }
    };
    task.start(mainFrame.getDesktopFrame(), LNG.get("notification.send.title"),
      LNG.get("notification.send.message"));
    /* Informa se ocorreu uma exceo */
    Exception exception = task.getException();
    if (exception != null) {
      showError(exception.getMessage());
      return false;
    }
    return true;
  }

  /**
   * Adio de observador remoto de notificaes.
   * 
   * @param observer observador remoto.
   * @param userId identificador do usurio.
   */
  public static void addUserObserver(Observer observer, Object userId) {
    try {
      NotificationObserver not = getUserObserver(userId);
      not.addObserver(observer);
      not.addUserObserver(userId);
    }
    catch (Exception e) {
      showError(e.getMessage());
    }
  }

  /**
   * Remoo de observador remoto de notificaes.
   * 
   * @param observer observador remoto.
   * @param userId identificador do usurio.
   */
  public static void deleteUserObserver(Observer observer, Object userId) {
    try {
      NotificationObserver not = getUserObserver(userId);
      not.deleteUserObserver(userId);
      not.deleteObserver(observer);
      killUserObserver(userId);
    }
    catch (Exception e) {
      showError(e.getMessage());
    }
  }

  /**
   * Remoo de todos os observadores.
   * 
   * @throws Exception em caso de falha.
   */
  public static void resetObservers() throws Exception {
    final NotificationServiceInterface service = getService();
    if (service == null) {
      throw new ClientException(LNG.get("notification.no.server.error"));
    }
    for (Enumeration e = userHash.keys(); e.hasMoreElements();) {
      Object uid = e.nextElement();
      NotificationObserver not = userHash.get(uid);
      not.deleteUserObserver(uid);
      not.deleteObservers();
      not.uninstall();
    }
    userHash = new Hashtable();
  }
}
