package csbase.client.util.csvpanel.columns;

/**
 * Gerador de valores sequenciais com prefixos e/ou sufixos.
 */
public class SequentialColumnGenerator implements ColumnGenerator<String> {

  /**
   * Valor inicial.
   */
  private int startValue;

  /**
   * Passo entre os valores.
   */
  private int step;

  /**
   * Prefixo do valor.
   */
  private String prefix;

  /**
   * Sufixo do valor.
   */
  private String sufix;

  /**
   * Construtor.
   */
  public SequentialColumnGenerator() {
    this(null, null);
  }

  /**
   * Construtor.
   * 
   * @param startValue Valor inicial.
   * @param step Passo entre os valores.
   */
  public SequentialColumnGenerator(Integer startValue, Integer step) {
    this(startValue, step, null, null);
  }

  /**
   * Construtor.
   * 
   * @param startValue Valor inicial.
   * @param step Passo entre os valores.
   * @param prefix Prefixo do valor.
   * @param suffix Sufixo do valor.
   */
  public SequentialColumnGenerator(Integer startValue, Integer step,
    String prefix, String suffix) {
    setStartValue(startValue);
    setStep(step);
    setPrefix(prefix);
    setSufix(suffix);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getValueAt(int rowIndex) {
    StringBuilder builder = new StringBuilder();
    builder.append(prefix);
    int value = startValue + (rowIndex * step);
    builder.append(value);
    builder.append(sufix);
    return builder.toString();
  }

  /**
   * Obtm o valor inicial.
   * 
   * @return o valor inicial.
   */
  public int getStartValue() {
    return startValue;
  }

  /**
   * Obtm o passo entre os valores.
   * 
   * @return o passo.
   */
  public int getStep() {
    return step;
  }

  /**
   * Obtm o prefixo do valor.
   * 
   * @return o prefixo.
   */
  public String getPrefix() {
    return prefix;
  }

  /**
   * Obtm o sufixo do valor.
   * 
   * @return o sufixo.
   */
  public String getSufix() {
    return sufix;
  }

  /**
   * Atribui o valor inicial. Se o valor passado for nulo, atribui o valor
   * padro.
   * 
   * @param startValue o valor inicial.
   */
  public void setStartValue(Integer startValue) {
    if (startValue == null) {
      this.startValue = 1;
    }
    else {
      this.startValue = startValue;
    }
  }

  /**
   * Atribui o passo entre os valores. Se o passo for nulo, atribui o valor
   * padro.
   * 
   * @param step o passo.
   */
  public void setStep(Integer step) {
    if (step == null) {
      this.step = 1;
    }
    else {
      this.step = step;
    }
  }

  /**
   * Atribui o prefixo do valor.
   * 
   * @param prefix o prefixo.
   */
  public void setPrefix(String prefix) {
    if (prefix == null) {
      this.prefix = "";
    }
    else {
      this.prefix = prefix;
    }
  }

  /**
   * Atribui o o sufixo do valor.
   * 
   * @param sufix o sufixo.
   */
  public void setSufix(String sufix) {
    if (sufix == null) {
      this.sufix = "";
    }
    else {
      this.sufix = sufix;
    }
  }
}
