package csbase.client.util.csvpanel.columns;

import java.awt.BorderLayout;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.text.MessageFormat;

import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GUIUtils;
import tecgraf.javautils.gui.StandardDialogs;
import csbase.client.desktop.DesktopComponentDialog;

/**
 * O dilogo de configurao do gerador sequencial de valores
 * {@link SequentialColumnGenerator}.
 */
public class SequentialColumnGeneratorDialog extends DesktopComponentDialog {

  /**
   * Rtulo de prefixo dos valores.
   */
  private JLabel prefixLabel;

  /**
   * Campo de prefixo dos valores.
   */
  private JTextField prefixField;

  /**
   * Rtulo de sufixo dos valores.
   */
  private JLabel sufixLabel;

  /**
   * Campo de sufixo dos valores.
   */
  private JTextField sufixField;

  /**
   * Rtulo do passo entre valores a serem gerados.
   */
  private JLabel stepLabel;

  /**
   * Campo com o passo entre valores a serem gerados.
   */
  private JTextField stepField;

  /**
   * Rtulo do valor inicial a ser gerado.
   */
  private JLabel startValueLabel;

  /**
   * Campo com o valor inicial a ser gerado.
   */
  private JTextField startValueField;

  /**
   * O gerador a ser configurado.
   */
  private SequentialColumnGenerator generator;

  /**
   * Indica se a operao de configuranao do gerador foi confirmada.
   */
  boolean confirmed;

  /**
   * Construtor.
   * 
   * @param owner a janela-me do dilogo.
   */
  public SequentialColumnGeneratorDialog(Window owner) {
    super(owner, getString("title"));
    this.confirmed = false;
    setDefaultCloseOperation(DISPOSE_ON_CLOSE);
    generator = new SequentialColumnGenerator();
    createUI();
  }

  /**
   * Monta os componentes grficos do dilogo.
   */
  private void createUI() {
    JPanel mainPanel = createMainPanel();
    JPanel buttonPanel = createButtonPanel();
    add(mainPanel, BorderLayout.CENTER);
    add(buttonPanel, BorderLayout.SOUTH);
    center();
    pack();
  }

  /**
   * Cria o painel principal do dilogo.
   * 
   * @return o painel.
   */
  private JPanel createMainPanel() {
    prefixLabel = new JLabel(getString("prefix.label"));
    prefixField = new JTextField(generator.getPrefix());
    sufixLabel = new JLabel(getString("sufix.label"));
    sufixField = new JTextField(generator.getSufix());
    stepLabel = new JLabel(getString("step.label"));
    stepField = new JTextField(String.valueOf(generator.getStep()));
    startValueLabel = new JLabel(getString("startValue.label"));
    startValueField = new JTextField(String.valueOf(generator.getStartValue()));

    final JComponent[][] rows =
      new JComponent[][] { { prefixLabel, prefixField },
          { sufixLabel, sufixField }, { stepLabel, stepField },
          { startValueLabel, startValueField } };
    JPanel mainPanel = GUIUtils.createBasicGridPanel(rows);
    return mainPanel;
  }

  /**
   * Cria o painel de botes do dilogo.
   * 
   * @return o painel.
   */
  private JPanel createButtonPanel() {
    JButton okButton = new JButton(getString("ok.button"));
    okButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        if (applyValues()) {
          confirmed = true;
          setVisible(false);
        }
      }
    });
    JButton cancelButton = new JButton(getString("cancel.button"));
    cancelButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        confirmed = false;
        setVisible(false);
      }
    });

    JPanel buttonPanel = new JPanel();
    buttonPanel.add(okButton, BorderLayout.LINE_START);
    buttonPanel.add(cancelButton, BorderLayout.LINE_END);
    return buttonPanel;
  }

  /**
   * Aplica e valida os valores escolhidos pelo usurio.
   * 
   * @return verdadeiro se os valores foram aplicados corretamente ou falso,
   *         caso contrrio.
   */
  private boolean applyValues() {
    String startValueText = startValueField.getText();
    if (!startValueText.trim().isEmpty()) {
      try {
        Integer startValue = Integer.valueOf(startValueText);
        generator.setStartValue(startValue);
      }
      catch (NumberFormatException e) {
        StandardDialogs.showErrorDialog(this, getTitle(),
          getString("error", startValueText, startValueLabel.getText()));
        return false;
      }
    }
    String stepText = stepField.getText();
    if (!stepText.trim().isEmpty()) {
      try {
        Integer stepValue = Integer.valueOf(stepText);
        generator.setStep(stepValue);
      }
      catch (NumberFormatException e) {
        StandardDialogs.showErrorDialog(this, getTitle(),
          getString("error", stepText, stepLabel.getText()));
        return false;
      }
    }
    generator.setPrefix(prefixField.getText());
    generator.setSufix(sufixField.getText());
    return true;
  }

  /**
   * Obtm a string localizada da chave de internacionalizao especificada.
   * 
   * @param key a chave de internacionalizao.
   * @return a string localizada.
   */
  private static String getString(String key) {
    return LNG.get(SequentialColumnGeneratorDialog.class.getName() + "." + key);
  }

  /**
   * Obtm a string localizada da chave de internacionalizao especificada.
   * 
   * @param key a chave de internacionalizao.
   * @param args lista de parmetros para formatao da string.
   * @return a string localizada.
   */
  private static String getString(String key, Object... args) {
    return MessageFormat.format(getString(key), args);
  }

  /**
   * Obtm o gerados de coluna configurado pelo usurio.
   * 
   * @return o gerador de coluna ou falso, caso no seja confirmada a operao.
   */
  public ColumnGenerator<?> getColumnGenerator() {
    if (!confirmed) {
      return null;
    }
    return generator;
  }

}
