package csbase.client.util.gui;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

import javax.swing.BorderFactory;
import javax.swing.JCheckBox;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.border.Border;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GBC;

public class AlgorithmFilterParamPanel extends JPanel {

  /**
   * A borda de erro.
   */
  private static final Border ERROR_BORDER =
    BorderFactory.createLineBorder(Color.RED, 2);

  /**
   * A borda padro do campo de texto.
   */
  private Border defaultTextFieldBorder;

  /**
   * Checkbox para indicar se deve ou no ignorar maisculas/minsculas.
   */
  private JCheckBox ignoreCaseBox;

  /**
   * O campo de texto.
   */
  private JTextField textField;

  /**
   * Checkbox para indicar uso ou no de expresso regular.
   */
  private JCheckBox useRegexBox;

  /**
   * Cheack box para habilitar algoritimos simples
   */
  private JCheckBox showSimpleBox;

  /**
   * Cheack box para habilitar algoritimos compostos (fluxos)
   */
  private JCheckBox showFlowBox;

  /**
   * O listener
   */
  private List<AlgorithmFilterParamPanelListener> listeners = new ArrayList<>();

  /**
   * Construtor
   */
  public AlgorithmFilterParamPanel() {

    textField = new JTextField();
    ignoreCaseBox = new JCheckBox(getString("ignoreCaseCheckBox.label"));
    useRegexBox = new JCheckBox(getString("regexCheckBox.label"));
    showSimpleBox = new JCheckBox(getString("simpleCheckBox.label"));
    showFlowBox = new JCheckBox(getString("fluxCheckBox.label"));

    textField.getDocument().addDocumentListener(new DocumentListener() {
      @Override
      public void removeUpdate(DocumentEvent e) {
        updateState();
      }

      @Override
      public void insertUpdate(DocumentEvent e) {
        updateState();
      }

      @Override
      public void changedUpdate(DocumentEvent e) {
        updateState();
      }
    });
    textField.addFocusListener(new FocusListener() {

      @Override
      public void focusLost(FocusEvent e) {

      }

      @Override
      public void focusGained(FocusEvent e) {
        String text = textField.getText();
        if (text.length() != 0) {
          textField.select(0, text.length());
        }
      }

    });
    defaultTextFieldBorder = textField.getBorder();

    ignoreCaseBox.setSelected(true);
    ignoreCaseBox.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        updateState();
      }
    });

    useRegexBox.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        updateState();
        if (useRegexBox.isSelected()) {
          ignoreCaseBox.setEnabled(false);
        }
        else {
          ignoreCaseBox.setEnabled(true);
        }
      }
    });

    showSimpleBox.setSelected(true);
    showSimpleBox.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        updateState();
      }
    });

    showFlowBox.setSelected(true);
    showFlowBox.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        updateState();
      }

    });

    JPanel flowOrSimplePanel = new JPanel(new GridBagLayout());
    GBC gbc = new GBC(0, 0).northwest().none().insets(0, 0, 0, 2);
    flowOrSimplePanel.add(showSimpleBox, gbc);
    flowOrSimplePanel.add(showFlowBox, gbc.gridx(1).insets(0, 0, 0, 0));

    JPanel searchPanel = new JPanel(new GridBagLayout());
    gbc = new GBC(1, 1).northwest().none().insets(2, 2, 2, 2).weights(0, 0);
    searchPanel.add(ignoreCaseBox, gbc);
    searchPanel.add(useRegexBox, gbc.gridy(2));
    searchPanel.add(textField, gbc.gridy(3).weightx(1).horizontal());
    searchPanel.add(flowOrSimplePanel, gbc.gridy(4).weightx(0).none());
    searchPanel.setBorder(BorderFactory.createEtchedBorder());
    setLayout(new BorderLayout());
    add(searchPanel, BorderLayout.NORTH);
  }

  /**
   * Adiciona um listener
   *
   * @param listener O listener
   */
  public void addListener(AlgorithmFilterParamPanelListener listener) {
    listeners.add(listener);
  }

  /**
   * Remove um listener
   *
   * @param listener O listener
   */
  public void removeListener(AlgorithmFilterParamPanelListener listener) {
    listeners.remove(listener);
  }

  /**
   * Chama os listeners
   */
  public void updateState() {
    if (useRegexBox.isSelected()) {
      if (getRegexPattern() == null) {
        textField.setBorder(ERROR_BORDER);
        return;
      }
    }
    textField.setBorder(defaultTextFieldBorder);

    for (AlgorithmFilterParamPanelListener listener : listeners) {
      listener.panelChanged(ignoreCaseBox.isSelected(), useRegexBox.isSelected(),
        showSimpleBox.isSelected(), showFlowBox.isSelected(),
        textField.getText());
    }
  }

  /**
   * Obtm o padro de expresso regular a ser utilizado na busca.
   *
   * @return .
   */
  public Pattern getRegexPattern() {
    String text = textField.getText();
    try {
      return Pattern.compile(text);
    }
    catch (PatternSyntaxException e) {
      return null;
    }
  }

  /**
   * Obtm o texto de uma propriedade de internacionalizao
   *
   * @param tag propriedade desejada
   * @return o texto de internacionalizao
   */
  private String getString(final String tag, final Object... args) {
    final String prefix = this.getClass().getSimpleName();
    final String fmt = LNG.get(prefix + "." + tag);
    final String text = String.format(fmt, args);
    return text;
  }

  /**
   * Obtm o campo de texto usado para filtragem.
   *
   * @return o campo de texto.
   */
  public JTextField getTextField() {
    return textField;
  }
}
