/*
 * $Id: LogViewerAction.java 69058 2007-10-15 17:34:47Z clinio $
 */
package csbase.client.util.gui.log.actions.core;

import java.awt.Window;
import java.awt.event.ActionEvent;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.ImageIcon;
import javax.swing.JComponent;

import csbase.client.util.StandardErrorDialogs;
import csbase.client.util.gui.log.LogPanel;

/**
 * @author Tecgraf/PUC-Rio
 */
public abstract class AbstractLogAction extends AbstractAction {

  /**
   * Tag usado para buscar o nome da ao.
   */
  final static protected String TAG_NAME = "name";

  /**
   * Tag usado para buscar o mnemnico da ao.
   */
  final static protected String TAG_MNEMONIC = "mnemonic";

  /**
   * Tag usado para buscar o tooltip da ao.
   */
  final static protected String TAG_TOOLTIP = "tooltip";

  /**
   * Painel de log associado a ao.
   */
  private final LogPanel logPanel;

  /**
   * Construtor
   * 
   * @param logPanel o painel de log associado a ao.
   * @param icon a imagem da ao.
   */
  protected AbstractLogAction(final LogPanel logPanel, final ImageIcon icon) {
    this(logPanel);
    putValue(Action.SMALL_ICON, icon);
  }

  /**
   * Construtor
   * 
   * @param logPanel o painel de log associado a ao.
   */
  protected AbstractLogAction(final LogPanel logPanel) {
    super();

    this.logPanel = logPanel;

    final String name = getName();
    if (name != null) {
      putValue(Action.NAME, name);
    }

    final Integer mnemonic = getMnemonic();
    if (mnemonic != null) {
      putValue(Action.MNEMONIC_KEY, mnemonic);
    }

    final String tooltip = getToolTipText();
    if (tooltip != null) {
      putValue(Action.SHORT_DESCRIPTION, tooltip);
    }
  }

  /**
   * Consulta o painel de log associado  ao.
   * 
   * @return o painel
   */
  protected final LogPanel getLogPanel() {
    return logPanel;
  }

  /**
   * {@inheritDoc} Repasse para outro mtodo com encapsulamento de excees.
   * 
   * @see #actionDone(JComponent)
   */
  @Override
  public void actionPerformed(ActionEvent ae) {
    try {
      final JComponent cmp = (JComponent) ae.getSource();
      actionDone(cmp);
    }
    catch (Exception e) {
      final Window window = getLogPanel().getOwner();
      final String title = getLogPanel().getTitle();

      StandardErrorDialogs.showErrorDialog(window, title, e);
    }
  }

  /**
   * Retorna um texto de internacionalizao.
   * 
   * @param key a chave de internacionalizao
   * @param args objetos de formatao
   * @return o texto
   */
  protected final String getString(String key, Object... args) {
    final String className = getClass().getSimpleName();
    String formattedKey = String.format("%s.%s", className, key);
    return LogPanel.getString(formattedKey, args);
  }

  /**
   * Mtodo de consulta ao texto de <i>tooltip</i> da ao montado no padro "
   * <code>NomeDaClasse.NAME_SUFIX</code>"
   * 
   * @return o texto (nome) da ao.
   * @see #getString(String, Object[])
   */
  public final String getName() {
    return getString(TAG_NAME);
  }

  /**
   * Mtodo de consulta ao texto de <i>tooltip</i> da ao montado no padro "
   * <code>NomeDaClasse.tooltip</code>"
   * 
   * @return o texto de tooltip
   * @see #getString(String, Object[])
   */
  public final String getToolTipText() {
    return getString(TAG_TOOLTIP);
  }

  /**
   * Consulta ao mnemnico da ao.
   * 
   * @return o mnemnico
   * @see #getString(String, Object[])
   */
  private final Integer getMnemonic() {
    final String mn = getString(TAG_MNEMONIC);
    if (mn == null || "".equals(mn)) {
      return null;
    }

    return Integer.valueOf(mn.charAt(0));
  }

  /**
   * Mtodo de ao a ser definido dentro do Planref.
   * 
   * @param component o componente que ativou a ao.
   * 
   * @throws Exception se houver falha na ao.
   */
  public abstract void actionDone(JComponent component) throws Exception;

}
