/*
 * Detalhes da ltima alterao:
 * 
 * $Id$
 */
package tecgraf.ftc_1_2.common.logic;

import java.util.HashMap;
import java.util.Map;

import tecgraf.ftc_1_2.client.RemoteFileChannel;

/**
 * Representa uma operao que pode ser realizada num {@link RemoteFileChannel}.
 * 
 * @author Tecgraf/PUC-Rio
 */
public enum Operation {
  /**
   * Operao de abertura de um arquivo somente para leitura.
   */
  OPEN_READ_ONLY((byte) 0),
  /**
   * Operao de abertura de um arquivo para leitura e para gravao.
   */
  OPEN_READ_WRITE((byte) 1),
  /**
   * Operao de fechamento de um arquivo.
   */
  CLOSE((byte) 2),
  /**
   * Operao para truncar o arquivo para um novo tamanho.
   */
  SET_SIZE((byte) 3),
  /**
   * Operao de obteno da posio atual do arquivo.
   */
  GET_POSITION((byte) 4),
  /**
   * Operao para definir a nova posio do arquivo.
   */
  SET_POSITION((byte) 5),
  /**
   * Operao de obteno do tamanho do arquivo.
   */
  GET_SIZE((byte) 6),
  /**
   * Operao para leitura de dados.
   */
  READ((byte) 7),
  /**
   * Operao para escrita de dados.
   */
  WRITE((byte) 8),

  /**
   * Operao para manter a conexao ativa.
   */
  KEEP_ALIVE((byte) 9);

  /**
   * O cdigo da operao.
   */
  private byte code;

  /**
   * Cria uma operao com um cdigo associado.
   * 
   * @param code O cdigo da operao.
   */
  private Operation(byte code) {
    this.code = code;
  }

  /**
   * Obtm o cdigo da operao.
   * 
   * @return O cdigo da operao.
   */
  public byte getCode() {
    return this.code;
  }

  /**
   * Mapa esttico relacionando os cdigos s operaes. Usado por
   * {@link #valueOf(byte)}.
   */
  private static final Map<Byte, Operation> codesMap =
    new HashMap<Byte, Operation>();

  static {
    /*
     * inicializao do mapa
     */
    for (Operation operation : Operation.values()) {
      codesMap.put(operation.code, operation);
    }
  }

  /**
   * Obtm uma operao a partir de seu cdigo.
   * 
   * @param code O cdigo.
   * 
   * @return A operao, ou {@code null}, caso o cdigo no corresponda a
   *         nenhuma operao.
   */
  public static Operation valueOf(byte code) {
    return codesMap.get(Byte.valueOf(code));
  }
}
