/*
 * $Id: FileServer.java 88595 2009-02-18 21:33:43Z vfusco $
 */
package tecgraf.ftc_1_2.server;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.ServerSocket;
import java.nio.channels.SelectionKey;
import java.nio.channels.Selector;
import java.nio.channels.ServerSocketChannel;
import java.nio.channels.SocketChannel;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.logging.FileHandler;
import java.util.logging.Handler;
import java.util.logging.Level;
import java.util.logging.Logger;

import tecgraf.ftc_1_2.common.exception.InvalidArraySize;
import tecgraf.ftc_1_2.common.logic.ErrorCode;
import tecgraf.ftc_1_2.server.states.CloseState;
import tecgraf.ftc_1_2.server.states.State;

/**
 * Representa o servidor de arquivos.
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class FileServer {

  /**
   * O dono do servidor de arquivos.
   */
  private FileProvider fileProvider;
  /**
   * Indica se o tratamento de requisies deve ser interrompido.
   */
  private volatile boolean wasStopped;
  /**
   * Indica se o servidor ja foi inicializado.
   */
  protected boolean initialized = false;
  /**
   * Gerenciador de conexes. Usado para selecionar as conexes ativas.
   */
  private Selector selector;
  /**
   * Mapeia uma chave de acesso s informaes sobre a requisio de um arquivo.
   */
  private Map<AccessKey, FileChannelRequestInfo> channels;

  /**
   * Objeto com as configuraes utilizadas pelo servidor.
   */
  private FileServerConfig config = null;

  /**
   * Socket do servidor
   */
  private ServerSocketChannel serverChannel = null;

  /**
   * SelectionKey do servidor
   */
  SelectionKey serverKey = null;

  /**
   * Tempo que indica a ultima verificao de conexes expiradas.
   */
  private long lastTimeoutCheck = 0;

  /**
   * Objeto o qual as excecoes capturadas pelo servidor serao repassadas.
   */
  private FileServerExceptionHandler exceptionHandler = null;

  /**
   * Objeto responsvel por registrar as atividades do servidor.
   */
  private final static Logger logger = Logger.getLogger("tecgraf.ftc");

  /**
   * Varivel que indica se estamos usando java 7
   */
  public final static boolean PLATAFORM_HAS_TRANSFERTO_BUG;

  /* Class static initializer */
  static {
    if (System.getProperty("os.name").contains("Linux")
      && System.getProperty("sun.arch.data.model").contains("32")
      && (!System.getProperty("java.version").contains("1.7.0"))) {
      PLATAFORM_HAS_TRANSFERTO_BUG = true;
    }
    else {
      PLATAFORM_HAS_TRANSFERTO_BUG = false;
    }
  };

  /**
   * Cria um servidor de arquivos.
   * 
   * @param fileProvider O objeto que prove os arquivos.
   * 
   * @throws IOException Caso ocorra algum erro na criao do servidor.
   */
  public FileServer(FileProvider fileProvider) throws IOException {

    this.fileProvider = fileProvider;
    this.config = new FileServerConfigImpl();
    this.channels = new HashMap<AccessKey, FileChannelRequestInfo>();

    logger.setUseParentHandlers(false);
    for (Handler handler : logger.getHandlers()) {
      logger.removeHandler(handler);
    }

  }

  /**
   * @return O objeto que prove os arquivos
   */
  public FileProvider getFileProvider() {
    return fileProvider;
  }

  /**
   * Metodo que faz a inicializao do servidor.
   * 
   * @return false se houver algum erro durante a inicializao
   */
  public boolean serverSetup() {
    if (!initialized) {

      try {
        logger.setLevel(config.getLoglevel());
        Handler logOutputFile =
          new FileHandler(config.getOutputLogFilename(), true);
        logOutputFile.setFormatter(new LogFormatter());
        logger.addHandler(logOutputFile);

        serverChannel = ServerSocketChannel.open();
        serverChannel.configureBlocking(false);
        ServerSocket serverSocket = serverChannel.socket();
        serverSocket.bind(new InetSocketAddress(config.getHostName(), config
          .getPort()));

        this.selector = Selector.open();
        serverKey =
          serverChannel.register(this.selector, SelectionKey.OP_ACCEPT);
      }
      catch (Exception e) {
        this.exceptionRaised(e);
        return false;
      }
      initialized = true;
    }

    return initialized;
  }

  /**
   * Inicia o tratamento de eventos.
   * 
   */
  public void dispatch() {

    // Setup server Socket and selector
    if (!serverSetup())
      return;

    while (!this.wasStopped) {
      try {
        int selectedKeyCount = this.selector.select(config.getSelectTimeout());

        checkTimedOutConnections();

        if (selectedKeyCount == 0) {
          continue;
        }
      }
      catch (IOException e) {
        this.exceptionRaised(e);
        continue;
      }

      Iterator<SelectionKey> selectedKeys =
        this.selector.selectedKeys().iterator();

      while (selectedKeys.hasNext()) {
        SelectionKey key = selectedKeys.next();
        selectedKeys.remove();

        if (key.isValid()) {
          if (serverKey.equals(key)) {
            if (key.isAcceptable())
              this.accept(key);
          }
          else {

            //if (logger.isLoggable(Level.FINEST))
            //  printKeyAddress(key);

            this.read(key);

            if (key.isValid() && key.isWritable()) {
              this.write(key);
            }
          }
        }
      }
    }
    shutdownConnections();
  }

  /**
   * Apenas um metodo auxiliar para logar o endereo do cliente.
   * 
   * @param key
   */
  private void printKeyAddress(SelectionKey key) {
    Session session = (Session) key.attachment();
    if (session == null)
      return;
    logger.finest("Address: "
      + session.getChannel().socket().getRemoteSocketAddress());
  }

  /**
   * Procura conexoes inativas por um determinado tempo.
   */
  private void checkTimedOutConnections() {
    long currentTime = System.currentTimeMillis();
    if (currentTime - lastTimeoutCheck < config.getSelectTimeout())
      return;

    lastTimeoutCheck = currentTime;

    Set<SelectionKey> keys = selector.keys();

    for (SelectionKey key : keys) {
      Session session = (Session) key.attachment();

      // Ignorando a chave do servidor
      if (session == null)
        continue;

      long test = currentTime - session.getLastActivity();

      if (test > config.getClientTimeout()) {

        logger.finer("Conexao fechada por inatividade.");
        stopConnection(key, session, ChannelClosedReason.CHANNEL_TIMEOUT);
      }
    }

    Set<Entry<AccessKey, FileChannelRequestInfo>> channelSet =
      channels.entrySet();
    Iterator<Entry<AccessKey, FileChannelRequestInfo>> iter =
      channelSet.iterator();
    while (iter.hasNext()) {
      Entry<AccessKey, FileChannelRequestInfo> entry = iter.next();
      FileChannelRequestInfo requestInfo = entry.getValue();

      long test = currentTime - requestInfo.getCreationTime();
      if (test > config.getChannelRequestTimeout()) {
        logger.finer("Canal removido porque no foi consumido dentro do tempo");
        iter.remove();
      }

    }

  }

  /**
   * Fecha todas as conexoes
   */
  private void shutdownConnections() {
    logger.finer("Fechando conexoes.");

    for (SelectionKey key : selector.keys()) {
      Session session = (Session) key.attachment();
      stopConnection(key, session, ChannelClosedReason.SERVER_SHUTDOWN);
    }
    serverChannel = null;
  }

  /**
   * Solicita a interrupo do tratamento de requisies.
   */
  public void stop() {
    this.wasStopped = true;
  }

  /**
   * Cria uma descrio de canal de arquivo.
   * 
   * @param requester O requisitante.
   * @param fileId O identificador do arquivo.
   * 
   * @return A descrio de canal de arquivo.
   * @throws InvalidArraySize Quando chave passada  invalida
   * @throws MaxChannelRequestsException Quando o limite de canais nao
   *         consumidos  atingido
   */
  public FileChannelAccessInfo createFileChannelInfo(Object requester,
    byte[] fileId) throws InvalidArraySize, MaxChannelRequestsException {
    return createFileChannelInfo(requester, fileId, null);
  }

  /**
   * Cria uma descrio de canal de arquivo.
   * 
   * @param requester O requisitante.
   * @param fileId O identificador do arquivo.
   * @param accessKey Chave de acesso ao arquivo.
   * 
   * @return A descrio de canal de arquivo.
   * @throws InvalidArraySize Quando chave passada  invalida
   * @throws MaxChannelRequestsException Quando o limite de canais nao
   *         consumidos  atingido
   */
  public FileChannelAccessInfo createFileChannelInfo(Object requester,
    byte[] fileId, byte[] accessKey) throws InvalidArraySize,
    MaxChannelRequestsException {
    return createFileChannelInfo(requester, fileId, accessKey, true);
  }

  /**
   * Cria uma descrio de canal de arquivo.
   * 
   * @param requester O requisitante.
   * @param fileId O identificador do arquivo.
   * @param accessKey Chave de acesso ao arquivo.
   * @param useTransferTo Indica se o metodo FileChannel.transferTo pode ser
   *        utilizado.
   * 
   * @return A descrio de canal de arquivo.
   * @throws InvalidArraySize Quando chave passada  invalida
   * @throws MaxChannelRequestsException Quando o limite de canais nao
   *         consumidos  atingido
   */
  public FileChannelAccessInfo createFileChannelInfo(Object requester,
    byte[] fileId, byte[] accessKey, boolean useTransferTo)
    throws InvalidArraySize, MaxChannelRequestsException {

    if (channels.size() + 1 > config.getMaxChannelRequests())
      throw new MaxChannelRequestsException("Limite maximo de canais atingido");

    AccessKey key =
      (accessKey != null) ? new AccessKey(accessKey) : new AccessKey();
    FileChannelRequestInfo fileChannelIinfo =
      new FileChannelRequestInfo(requester, fileId);

    if (logger.isLoggable(Level.FINEST))
      logger.finest("Criando novo FileChannelInfo: " + Arrays.toString(fileId));

    fileChannelIinfo.useTransferTo(useTransferTo);

    this.channels.put(key, fileChannelIinfo);
    return new FileChannelAccessInfo(config.getHostName(), serverChannel
      .socket().getLocalPort(), key.getBytes(), fileId);
  }

  /**
   * Obtm as informaes sobre a requisio de um arquivo, a partir de uma
   * chave de acesso.
   * 
   * @param accessKey A chave de acesso.
   * 
   * @return As informaes sobre a requisio de um arquivo.
   */
  public FileChannelRequestInfo getFileChannelInfo(AccessKey accessKey) {
    // Se estiver em modo de teste, no remover a chave
    if (config.isTestMode()) {
      return this.channels.get(accessKey);
    }
    return this.channels.remove(accessKey);
  }

  /**
   * Invocado quando uma exceo  lanada no servidor.
   * 
   * @param e A exceo lanada.
   * @param fileId Identificador do arquivo envolvido no momento do erro
   */
  public void exceptionRaised(Exception e, byte[] fileId) {
    if (exceptionHandler != null)
      this.exceptionHandler.exceptionRaised(e, fileId);
  }

  /**
   * Invocado quando uma exceo  lanada no servidor.
   * 
   * @param e A exceo lanada.
   */
  public void exceptionRaised(Exception e) {
    if (exceptionHandler != null)
      this.exceptionHandler.exceptionRaised(e);
  }

  /**
   * Trata uma tentativa de conexo a um {@link ServerSocketChannel}
   * 
   * @param key A chave selecionada.
   */
  private void accept(SelectionKey key) {
    ServerSocketChannel serverChannel = (ServerSocketChannel) key.channel();
    SocketChannel socketChannel = null;

    try {
      while (true) {
        socketChannel = serverChannel.accept();

        if (socketChannel == null)
          break;

        // configurando o socket como nao bloqueante.
        socketChannel.configureBlocking(false);

        // Criando a sessao desse cliente
        Session session = new Session(socketChannel, this);
        session.markLastActivity();

        // Resgitrando a sesso.
        socketChannel.register(this.selector, SelectionKey.OP_READ
          | SelectionKey.OP_WRITE, session);

        int clients = selector.keys().size() - 1;

        // Verificando se podemos atender mais um cliente
        if (clients > config.getMaxClients()) {

          if (logger.isLoggable(Level.FINER))
            logger.finer("Nmero mximo de clientes atingido.");

          session
            .setCurrentState(new CloseState(ErrorCode.MAX_CLIENTS_REACHED));
          break;
        }

        if (logger.isLoggable(Level.FINE)) {
          logger.fine("Cliente conectado:"
            + socketChannel.socket().getRemoteSocketAddress());
          logger.fine("Numero de clientes:" + clients);
        }

        if (!config.acceptMaxPossible())
          break;

      }
      return;

    }
    catch (IOException e) {
      this.exceptionRaised(e);
    }
    catch (OutOfMemoryError e) {
      if (logger.isLoggable(Level.WARNING))
        logger.warning("No h recursos suficientes. A Conexo sera fechada.");
    }

    // Se chegamos aqui  porque uma exceo foi lanada.
    // Vamos fechar o socket do cliente.
    try {
      if (socketChannel != null)
        socketChannel.close();
    }
    catch (IOException ioe) {
      if (logger.isLoggable(Level.WARNING))
        logger.warning("Erro ao tentar fechar a conexo.");
    }

    return;

  }

  /**
   * Trata uma tentativa de leitura em um {@link SocketChannel}
   * 
   * @param key A chave selecionada.
   */
  private void read(SelectionKey key) {
    Session session = (Session) key.attachment();
    try {
      State currentState = session.getCurrentState();
      if ((currentState == null) || (!currentState.read(session))) {
        this.stopConnection(key, session, ChannelClosedReason.CHANNEL_ERROR);
      }
    }
    catch (Exception e) {
      this.exceptionRaised(e, session.getFileChannelInfo().getFileId());
      this.stopConnection(key, session, ChannelClosedReason.CHANNEL_ERROR);
    }
  }

  /**
   * Trata uma tentativa de escrita em um {@link SocketChannel}
   * 
   * @param key A chave selecionada.
   */
  private void write(SelectionKey key) {
    Session session = (Session) key.attachment();
    try {
      State currentState = session.getCurrentState();
      if ((currentState == null) || (!currentState.write(session))) {
        this.stopConnection(key, session, ChannelClosedReason.CHANNEL_ERROR);
      }

    }
    catch (Exception e) {
      this.exceptionRaised(e, session.getFileChannelInfo().getFileId());
      this.stopConnection(key, session, ChannelClosedReason.CHANNEL_ERROR);
    }
  }

  /**
   * Finaliza uma conexo com um cliente.
   * 
   * @param key A chave de seleo do canal do cliente.
   * @param session A sesso do cliente.
   * @param reason Razo pela qual o canal foi fechado.
   */
  private void stopConnection(SelectionKey key, Session session,
    ChannelClosedReason reason) {
    if (session != null) {
      session.close(reason);
    }
    else {
      try {
        key.channel().close();
      }
      catch (IOException e) {
        logger.finer("Erro ao fechar conexo.");
      }
    }
    key.attach(null);
    key.cancel();
  }

  /**
   * Retorna Objeto com as configuraes utilizadas pelo servidor
   * 
   * @return config
   */
  public FileServerConfig getConfig() {
    return config;
  }

  /**
   * @param config Objeto com as configuraes utilizadas pelo servidor
   */
  public void setConfig(FileServerConfig config) {
    this.config = config;
  }

  /**
   * @return O exception handler cadastrado
   */
  public FileServerExceptionHandler getExceptionHandler() {
    return exceptionHandler;
  }

  /**
   * Cadastra um exception handler para receber as excecoes do servidor.
   * 
   * @param exceptionHandler
   */
  public void setExceptionHandler(FileServerExceptionHandler exceptionHandler) {
    this.exceptionHandler = exceptionHandler;
  }
}
