package tecgraf.ftc_1_2.server.states;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.SocketChannel;
import java.util.logging.Level;
import java.util.logging.Logger;

import tecgraf.ftc_1_2.common.logic.ErrorCode;
import tecgraf.ftc_1_2.common.logic.PrimitiveTypeSize;
import tecgraf.ftc_1_2.common.logic.ProtocolVersion;
import tecgraf.ftc_1_2.server.Session;

/**
 * Classe que implementa o estado de handshake inicial. Nesse handshake 
 * determinado qual verso do protocolo sera utilizada.
 * 
 * @author Tecgraf
 */
public class VersionHandshakeState implements State {

  /**
   * Representa os estados internos desta operao.
   * 
   * @author Tecgraf/PUC-Rio
   */
  private enum InternalState {
    /**
     * O estado inicial.
     */
    INITIAL,
    /**
     * Estado que criado para suportar clientes antigos que fala protocolo 1.1
     */
    FTC_1_1_PROTOCOL_CHECKED,
    /**
     * Estado que indica que o identificador de protocolo foi recebido.
     */
    PROTOCOL_ID_RECEIVED,
    /**
     * Estado que indica que o numero da verso ja foi recebido.
     */
    VERSION_RECEIVED,
    /**
     * Estado que indica que o cdigo de erro j foi enviado.
     */
    ERROR_CODE_SENT;
  }

  /**
   * O estado atual da operao.
   */
  private InternalState currentState;

  /**
   * O cdigo de erro.
   */
  private ErrorCode errorCode;

  /**
   * Objeto responsvel por registrar as atividades do servidor.
   */
  private final static Logger logger = Logger.getLogger("tecgraf.ftc");

  /**
   * Cria o estado que lida com o handshake inicial.
   */
  public VersionHandshakeState() {
    this.currentState = InternalState.INITIAL;

    if (logger.isLoggable(Level.FINER))
      logger.finer("Estado que lida com o handshake inicial.");
  }

  /**
   * {@inheritDoc}
   */
  public boolean read(Session session) {
    ByteBuffer buffer = session.getBuffer();
    SocketChannel channel = session.getChannel();
    switch (this.currentState) {
      case INITIAL:
        buffer.limit(PrimitiveTypeSize.BYTE.getSize());
        try {
          if (channel.read(buffer) > 0)
            session.markLastActivity();
        }
        catch (IOException e) {
          if (logger.isLoggable(Level.FINER))
            logger.finer("Erro ao ler do canal.");
          return false;
        }
        if (buffer.hasRemaining()) {
          return true;
        }

        // Workaround para suportar clientes com verso 1.1
        if (buffer.get(0) != 0) {
          session
            .setCurrentState(new tecgraf.ftc_1_2.server.states.v1_1.ValidateKeyState());
          return true;
        }
        this.currentState = InternalState.FTC_1_1_PROTOCOL_CHECKED;

      case FTC_1_1_PROTOCOL_CHECKED:

        buffer.limit(PrimitiveTypeSize.INTEGER.getSize());
        try {
          if (channel.read(buffer) > 0)
            session.markLastActivity();
        }
        catch (IOException e) {
          if (logger.isLoggable(Level.FINER))
            logger.finer("Erro ao ler do canal.");
          return false;
        }
        if (buffer.hasRemaining()) {
          return true;
        }

        buffer.flip();
        int protocolId = buffer.getInt();
        buffer.clear();

        if (protocolId != ProtocolVersion.PROTOCOL_IDENTIFICATION) {
          if (logger.isLoggable(Level.FINER)) {
            logger.finer("Cliente enviou uma mensagem invalida.");
          }
          this.currentState = InternalState.VERSION_RECEIVED;
          this.errorCode = ErrorCode.FAILURE;
          return true;
        }

        this.currentState = InternalState.PROTOCOL_ID_RECEIVED;

      case PROTOCOL_ID_RECEIVED:
        buffer.limit(PrimitiveTypeSize.INTEGER.getSize());
        try {
          if (channel.read(buffer) > 0)
            session.markLastActivity();
        }
        catch (IOException e) {
          if (logger.isLoggable(Level.FINER))
            logger.finer("Erro ao ler do canal.");
          return false;
        }
        if (buffer.hasRemaining()) {
          return true;
        }

        buffer.flip();
        buffer.get();//padding
        byte major_version = buffer.get();
        byte minor_version = buffer.get();
        byte patch_version = buffer.get();
        buffer.clear();

        this.currentState = InternalState.VERSION_RECEIVED;

        if (logger.isLoggable(Level.FINER)) {
          logger.fine("Cliente conectado utilizando versao: " + major_version
            + "." + minor_version + "." + patch_version);
        }

        if ((major_version == ProtocolVersion.MAJOR_VERSION)
          && (minor_version == ProtocolVersion.MINOR_VERSION)) {
          this.errorCode = ErrorCode.OK;
        }
        else {
          if (logger.isLoggable(Level.FINER)) {
            logger.fine("Versao nao suportada por este servidor");
          }
          this.errorCode = ErrorCode.INVALID_VERSION;
        }

      default:
        return true;
    }
  }

  /**
   * {@inheritDoc}
   */
  public boolean write(Session session) {
    ByteBuffer buffer = session.getBuffer();
    SocketChannel channel = session.getChannel();
    switch (this.currentState) {
      case VERSION_RECEIVED:
        buffer.limit(PrimitiveTypeSize.BYTE.getSize());
        buffer.put(this.errorCode.getCode());
        buffer.flip();
        try {
          if (channel.write(buffer) > 0)
            session.markLastActivity();
        }
        catch (IOException e) {
          if (logger.isLoggable(Level.FINER))
            logger.finer("Erro ao ler do canal.");
          return false;
        }
        finally {
          buffer.clear();
        }
        this.currentState = InternalState.ERROR_CODE_SENT;

        if (logger.isLoggable(Level.FINER))
          logger.finer("Cdigo " + this.errorCode + " enviado.");

        if (errorCode == ErrorCode.OK)
          session.setCurrentState(new ValidateKeyState());
        else
          return false;

      default:
        return true;
    }
  }

}
