/*
 * $Id: WriteState.java 88595 2009-02-18 21:33:43Z vfusco $
 */
package tecgraf.ftc_1_2.server.states.v1_1;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.SocketChannel;
import java.util.logging.Level;
import java.util.logging.Logger;

import tecgraf.ftc_1_2.common.logic.ErrorCode;
import tecgraf.ftc_1_2.common.logic.PrimitiveTypeSize;
import tecgraf.ftc_1_2.server.FileChannelRequestInfo;
import tecgraf.ftc_1_2.server.Session;
import tecgraf.ftc_1_2.server.states.State;
import tecgraf.ftc_1_2.utils.IOUtils;

/**
 * Operao para escrita de dados a partir de uma determinada posio.
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class WriteState implements State {
  /**
   * Representa os estados internos desta operao.
   * 
   * @author Tecgraf/PUC-Rio
   */
  private enum InternalState {
    /**
     * O estado inicial.
     */
    INITIAL,
    /**
     * Estado que indica se a possibilidade de escrtita j foi verificada.
     */
    CHECKED,
    /**
     * Estado que indica que a posio a partir da qual os dados sero escritos
     * j foi lida.
     */
    POSITION_READ,
    /**
     * Estado que indica que a quantidade de bytes que sero escritos j foi
     * lida.
     */
    BYTE_COUNT_READ,
    /**
     * Estado que indica que todos os bytes solicitados j foram recebidos.
     */
    BYTES_RECEIVED;
  }

  /**
   * O estado atual da operao.
   */
  private InternalState currentState;

  /**
   * A posio a partir da qual os dados sero escritos no arquivo.
   */
  private long position;
  /**
   * Quantidade de bytes que sero escritos no arquivo.
   */
  private long count;
  /**
   * A quantidade de bytes recebidos do cliente.
   */
  private long bytesReceived;

  /**
   * Objeto responsvel por registrar as atividades do servidor.
   */
  private final static Logger logger = Logger.getLogger("tecgraf.ftc");

  /**
   * Cria operao para escrita de dados a partir de uma determinada posio.
   */
  public WriteState() {
    this.currentState = InternalState.INITIAL;

    if (logger.isLoggable(Level.FINER))
      logger.finer("Estado de escrita.");
  }

  /**
   * {@inheritDoc}
   */
  public boolean read(Session session) {
    ByteBuffer buffer = session.getBuffer();
    SocketChannel channel = session.getChannel();
    FileChannel fileChannel = session.getFileChannel();
    switch (this.currentState) {
      case INITIAL:
        buffer.limit(PrimitiveTypeSize.LONG.getSize());
        try {
          if (channel.read(buffer) > 0)
            session.markLastActivity();
        }
        catch (IOException e) {
          e.printStackTrace();
          return false;
        }
        if (buffer.hasRemaining()) {
          return true;
        }
        buffer.flip();
        this.position = buffer.getLong();
        buffer.clear();
        if (this.position < 0) {
          try {
            this.position = fileChannel.position();
          }
          catch (IOException e) {
            if (logger.isLoggable(Level.FINER))
              logger.finer("Erro ao ler posio do arquivo.");
            return false;
          }
        }
        this.currentState = InternalState.POSITION_READ;
        return true;
      case POSITION_READ:
        buffer.limit(PrimitiveTypeSize.LONG.getSize());
        try {
          if (channel.read(buffer) > 0)
            session.markLastActivity();
        }
        catch (IOException e) {
          e.printStackTrace();
          return false;
        }
        if (buffer.hasRemaining()) {
          return true;
        }
        buffer.flip();
        this.count = buffer.getLong();
        buffer.clear();
        this.currentState = InternalState.BYTE_COUNT_READ;
        return true;
      case CHECKED:
        FileChannelRequestInfo fileInfo = session.getFileChannelInfo();

        long bytesRead;
        try {

          if (fileInfo.useTransferTo()) {
            if (logger.isLoggable(Level.FINEST)) {
              logger.finest("Using TransferFrom");
              logger.finest("position=" + this.position);
              logger.finest("count=" + this.count);
              logger.finest("bytesReceived=" + this.bytesReceived);
            }

            bytesRead =
              fileChannel.transferFrom(channel, this.position
                + this.bytesReceived, this.count - this.bytesReceived);
          }
          else {
            bytesRead =
              IOUtils.transferFromNonBlock(fileChannel, this.position
                + this.bytesReceived, this.count - this.bytesReceived, channel,
                buffer);
          }

          if (bytesRead > 0) {
            session.markLastActivity();
            fileChannel.position(this.position + bytesRead);
          }
        }
        catch (IOException e) {
          session.getFileServer().exceptionRaised(e,
            session.getFileChannelInfo().getFileId());
          return false;
        }

        this.bytesReceived += bytesRead;

        if (logger.isLoggable(Level.FINEST)) {
          logger.finest("Recebidos " + bytesRead);
          logger.finest("Total " + bytesReceived);
        }

        if (this.bytesReceived == this.count) {
          this.currentState = InternalState.BYTES_RECEIVED;
          session.setCurrentState(new GetOperationState());
        }
      default:
        return true;
    }
  }

  /**
   * {@inheritDoc}
   */
  public boolean write(Session session) {
    ByteBuffer buffer = session.getBuffer();
    SocketChannel channel = session.getChannel();
    FileChannel fileChannel = session.getFileChannel();
    switch (this.currentState) {
      case BYTE_COUNT_READ:
        ErrorCode errorCode;
        FileChannelRequestInfo fileChannelInfo = session.getFileChannelInfo();
        if (session
          .getFileServer()
          .getFileProvider()
          .isLocked(fileChannelInfo.getRequester(), fileChannelInfo.getFileId())) {
          errorCode = ErrorCode.FILE_LOCKED;
        }
        else {
          errorCode = ErrorCode.OK;
        }
        buffer.limit(PrimitiveTypeSize.BYTE.getSize());
        buffer.put(errorCode.getCode());
        buffer.flip();
        try {
          if (channel.write(buffer) > 0)
            session.markLastActivity();
        }
        catch (IOException e) {
          session.getFileServer().exceptionRaised(e,
            session.getFileChannelInfo().getFileId());
          return false;
        }
        finally {
          buffer.clear();
        }

        if (logger.isLoggable(Level.FINER))
          logger.finer("Cdigo " + errorCode + " enviado.");

        if (errorCode.equals(ErrorCode.OK)) {
          this.currentState = InternalState.CHECKED;
          try {
            // workaround para usar o transferFrom quando tamanho do arquivo
            // for menor que a posicao de escrita pedida
            if (fileChannelInfo.useTransferTo()) {
              ByteBuffer src = ByteBuffer.wrap(new byte[] { 0 });
              long currentPosition = fileChannel.position();
              fileChannel.position(this.position + this.count - 1);
              fileChannel.write(src);
              fileChannel.position(currentPosition);
            }
          }
          catch (IOException e) {
            session.getFileServer().exceptionRaised(e,
              session.getFileChannelInfo().getFileId());
            return false;
          }
        }
        else {
          session.setCurrentState(new GetOperationState());
        }
      default:
        return true;
    }
  }
}
