package tecgraf.ftc_1_4.client;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.InetSocketAddress;
import java.nio.ByteBuffer;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.SocketChannel;
import java.nio.channels.WritableByteChannel;
import java.util.Formatter;

import tecgraf.ftc_1_4.common.exception.DataChannelException;
import tecgraf.ftc_1_4.common.exception.FailureException;
import tecgraf.ftc_1_4.common.exception.FileLockedException;
import tecgraf.ftc_1_4.common.exception.InvalidProtocolVersionException;
import tecgraf.ftc_1_4.common.exception.MaxClientsReachedException;
import tecgraf.ftc_1_4.common.exception.PermissionException;
import tecgraf.ftc_1_4.common.exception.UnexpectedProtocolMessage;
import tecgraf.ftc_1_4.common.logic.ErrorCode;
import tecgraf.ftc_1_4.common.logic.Operation;
import tecgraf.ftc_1_4.common.logic.PrimitiveTypeSize;
import tecgraf.ftc_1_4.common.logic.ProtocolVersion;
import tecgraf.ftc_1_4.common.logic.ResultMessage;
import tecgraf.ftc_1_4.utils.ByteBufferUtils;

import static tecgraf.ftc_1_4.server.ErrorMessages.hexString;

/**
 * Representa uma implementao de um canal de arquivo remoto.
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class RemoteDataChannel implements IRemoteDataChannel {

  /** Tamanho maximo do identificador */
  static public final int MAX_IDENTIFIER_SIZE = 65535;

  /** Tamanho maximo da chave */
  static public final int MAX_KEY_SIZE = 255;

  /**
   * Tamanho minimo para o buffer. Baseado no maior pacote de dados possivel no
   * protocolo
   */
  static public final int MIN_BUFFER_SIZE = MAX_IDENTIFIER_SIZE;

  private static final String REMOTE_DATA_CHANNEL_CLOSED = "Remote file data channel closed";
  private static final String FAILED_TO_CLOSE_REMOTE_DATA_CHANNEL = "Failed to close remote file data channel";
  private static final String READ_ONLY_DATA_CHANNEL = "Remote data channel is read-only";
  private static final String NO_PERMISSION_ON_DATA_CHANNEL = "Remote data channel access denied (reason: %s)";
  private static final String DATA_CHANNEL_ALREADY_OPENED = "Remote data channel is already opened";
  private static final String DATA_CHANNEL_ALREADY_LOCKED = "Remote data channel is already locked";
  private static final String UNABLE_TO_RESOLVE_REMOTE_ADDRESS = "Unable to resolve remote address %s";

  private static final String INVALID_ERROR_CODE = "Invalid error code %d returned by server";
  private static final String INVALID_ACCESS_KEY_ERROR_CODE = "Access denied due to invalid access key %s (reason: %s)";
  private static final String SERVER_FAILURE_OPERATION_ON_DATA_CHANNEL = "Server failure %s (reason: %s)";
  private static final String SERVER_REPORTED_INVALID_VERSION = "Server reported invalid protocol version %s";

  private static final String EARLIER_EOS = "earlier end of stream";
  private static final String ILLEGAL_REMOTE_POSITION = "Remote position must be greater or equal to -1: ";
  private static final String ILLEGAL_NEGATIVE_NUMBER = "%s must be a non negative number: %d";
  private static final String ILLEGAL_NULL_VALUE = "%s must be not null";
  /**
   * A mquina do servidor de arquivos.
   */
  private String host;
  /**
   * A porta do servidor de arquivos.
   */
  private int port;
  /**
   * A chave de acesso ao arquivo.  usada para verificar se o usurio possui
   * permisso de acesso ao arquivo.
   */
  private byte[] key;
  /**
   * Indica se  permitida a escrita no arquivo.
   */
  private boolean writable;
  /**
   * Canal de comunicao com o servidor de arquivos.
   */
  private SocketChannel channel;
  /**
   * <i>Buffer</i> utilizado na comunicao com o servidor de arquivos.
   */
  private ByteBuffer buffer;

  /**
   * <i>bufferSize</i> determina o tamanho do buffer utilizado.
   */
  private int bufferSize = 1024 * 1024;

  /**
   * Variavel que armazena a mensagem de retorno da ultima operacao;
   */
  private ResultMessage lastResultMessage;

  /**
   * Variavel que representa as operacoes suportas por um canal remoto
   */
  private short operations = 0;

  /**
   * Cria um canal de dados remoto. O canal so  efetivamente aberto aps a
   * chamada do metodo open.
   * 
   * @param writable Indica se a escrita  permitida no arquivo.
   * @param host A mquina do servidor de arquivos.
   * @param port A porta do servidor de arquivos.
   * @param key A chave de acesso ao arquivo com tamanho mximo padro {@link #MAX_KEY_SIZE}.
   */
  public RemoteDataChannel(boolean writable, String host, int port, byte[] key) {
    if (host == null || host.isEmpty()) {
      throw new IllegalArgumentException("Host must be a non empty string");
    }

    if (port < 0 || port > 0xFFFF) {
      throw new IllegalArgumentException("Port number out of range " + 0 + " and " + 0xFFFF);
    }

    if (key == null || key.length == 0 || key.length > MAX_KEY_SIZE) {
      throw new IllegalArgumentException("Access key must be a non empty array with maximum length " + MAX_KEY_SIZE);
    }

    this.host = host;
    this.port = port;
    this.key = key;
    this.writable = writable;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void open() throws PermissionException, FileNotFoundException,
      FailureException, MaxClientsReachedException,
      InvalidProtocolVersionException {

    if (isOpen()) {
      throw new FailureException(DATA_CHANNEL_ALREADY_OPENED);
    }

    try {
      InetSocketAddress address = new InetSocketAddress(this.host, this.port);
      if (address.isUnresolved()) {
        throw new FailureException(String.format(UNABLE_TO_RESOLVE_REMOTE_ADDRESS, address));
      }

      this.channel = SocketChannel.open(address);
      this.channel.socket().setTcpNoDelay(true);
    } catch (IOException e) {
      throw new FailureException(e);
    }

    this.buffer = ByteBuffer.allocate(this.bufferSize);

    protocolVersionHandshake();

    authenticate();

    ResultMessage retMesg = null;
    try {
      Operation operation = this.writable ? Operation.OPEN_READ_WRITE
          : Operation.OPEN_READ_ONLY;

      this.buffer.clear();
      ByteBufferUtils.writeByte(this.buffer, this.channel, operation.getCode());

      retMesg = readResultMessage();

      if (retMesg.success) {
        this.operations = ByteBufferUtils.readShort(this.buffer, this.channel);
      }
    } catch (IOException e) {
      this.release();
      throw new FailureException(e);
    } catch (UnexpectedProtocolMessage e) {
      this.release();
      throw new FailureException(e);
    }

    if (!retMesg.success) {
      this.release();

      if (retMesg.code.equals(ErrorCode.FILE_NOT_FOUND)) {
        throw new FileNotFoundException(retMesg.message);
      } else if (retMesg.code.equals(ErrorCode.NO_PERMISSION)) {
        throw new PermissionException(String.format(NO_PERMISSION_ON_DATA_CHANNEL, retMesg.message));
      } else if (retMesg.code.equals(ErrorCode.FAILURE)) {
        throw new FailureException(String.format(SERVER_FAILURE_OPERATION_ON_DATA_CHANNEL, "opening data channel", retMesg.message));
      }
      throw new IllegalStateException(String.format(INVALID_ERROR_CODE, retMesg.code));
    }
  }

  private String currentProtocolVersion() {
      StringBuilder builder = new StringBuilder();
      builder.append("Protocol ID:");
      builder.append(new Formatter().format("%X", ProtocolVersion.PROTOCOL_IDENTIFICATION));
      builder.append(ProtocolVersion.MAJOR_VERSION);
      builder.append(".");
      builder.append(ProtocolVersion.MINOR_VERSION);
      builder.append(".");
      builder.append(ProtocolVersion.PATCH_VERSION);
      return builder.toString();
  }

  /**
   * Metodo responsavel por realizar o handshake inicial de verso do protocolo.
   * 
   * @throws FailureException
   * @throws PermissionException
   * @throws MaxClientsReachedException
   * @throws InvalidProtocolVersionException
   * 
   */
  private void protocolVersionHandshake() throws FailureException,
      PermissionException, MaxClientsReachedException,
      InvalidProtocolVersionException {
    long idAndVersion = ProtocolVersion.PROTOCOL_IDENTIFICATION;
    idAndVersion <<= 32;
    int major = ProtocolVersion.MAJOR_VERSION;
    int minor = ProtocolVersion.MINOR_VERSION;
    int patch = ProtocolVersion.PATCH_VERSION;
    idAndVersion |= ((major << 16) | (minor << 8) | (patch));

    ResultMessage retMesg = null;
    try {
      this.buffer.clear();
      ByteBufferUtils.writeLong(this.buffer, this.channel, idAndVersion);
      retMesg = readResultMessage();
    } catch (IOException e) {
      this.release();
      throw new FailureException(e);
    } catch (UnexpectedProtocolMessage e) {
      this.release();
      throw new FailureException(e);
    }

    if (!retMesg.success) {
      this.release();

      if (retMesg.code.equals(ErrorCode.INVALID_VERSION)) {
        throw new InvalidProtocolVersionException(
          String.format(SERVER_REPORTED_INVALID_VERSION, currentProtocolVersion()));
      } else if (retMesg.code.equals(ErrorCode.MAX_CLIENTS_REACHED)) {
        throw new MaxClientsReachedException(
          String.format(NO_PERMISSION_ON_DATA_CHANNEL, "max clients reached"));
      } else if (retMesg.code.equals(ErrorCode.FAILURE)) {
        throw new FailureException(
          String.format(SERVER_FAILURE_OPERATION_ON_DATA_CHANNEL, "negotiating protocol version", retMesg.message));
      }
      throw new IllegalStateException(String.format(INVALID_ERROR_CODE, retMesg.code));
    }
  }

  /**
   * Metodo responsavel por realizar a autenticao
   * 
   * @throws FailureException
   * @throws PermissionException
   * @throws MaxClientsReachedException
   */
  private void authenticate() throws FailureException, PermissionException,
      MaxClientsReachedException {
    ResultMessage retMesg = null;
    try {
      this.buffer.clear();
      ByteBufferUtils.writeBytesByteSize(this.buffer, this.channel, this.key);
      retMesg = readResultMessage();
    } catch (IOException e) {
      this.release();
      throw new FailureException(e);
    } catch (UnexpectedProtocolMessage e) {
      this.release();
      throw new FailureException(e);
    }

    if (!retMesg.success) {
      this.release();

      if (retMesg.code.equals(ErrorCode.INVALID_KEY)) {
        throw new PermissionException(
          String.format(INVALID_ACCESS_KEY_ERROR_CODE, hexString(this.key), retMesg.message));
      } else if (retMesg.code.equals(ErrorCode.FAILURE)) {
        throw new FailureException(
          String.format(SERVER_FAILURE_OPERATION_ON_DATA_CHANNEL, "validating the access key", retMesg.message));
      }
      throw new IllegalStateException(String.format(INVALID_ERROR_CODE, retMesg.code));
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isOpen() {
    if (this.channel == null) {
      return false;
    }
    return this.channel.isOpen();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void close() throws IOException {
    checkIsOpen();

    ResultMessage retMesg = null;
    try {
      ByteBufferUtils.writeByte(this.buffer, this.channel,
          Operation.CLOSE.getCode());
      retMesg = readResultMessage();

      if (!retMesg.success) {
        throw new IOException(FAILED_TO_CLOSE_REMOTE_DATA_CHANNEL);
      }
    } catch (IOException e) {
      throw e;
    } catch (UnexpectedProtocolMessage e) {
      throw new IOException(e);
    } finally {
      this.release();
    }
  }

  /**
   * Metodo privado que le o codigo de retorno enviado pelo servidor
   * 
   * @return Codigo de retorno
   * @throws IOException
   * @throws UnexpectedProtocolMessage
   */
  private ResultMessage readResultMessage() throws IOException,
      UnexpectedProtocolMessage {

    this.lastResultMessage = new ResultMessage();

    // Lendo a resposta da operacao
    this.buffer.clear();
    this.buffer.limit(PrimitiveTypeSize.BYTE.getSize());
    do {
      if (this.channel.read(this.buffer) == -1) {
        throw new IOException(REMOTE_DATA_CHANNEL_CLOSED);
      }
    } while (this.buffer.hasRemaining());

    // Interpretando resultado
    this.buffer.flip();
    byte result = this.buffer.get();
    this.buffer.clear();

    if ((result != ResultMessage.FAILURE) && (result != ResultMessage.SUCCESS)) {
      throw new UnexpectedProtocolMessage(String.format(INVALID_ERROR_CODE, result));
    }

    this.lastResultMessage.success = (result == ResultMessage.SUCCESS);

    // Retornando ok se a operacao foi bem sucedida
    if (this.lastResultMessage.success) {
      return this.lastResultMessage;
    }

    // Se a operacao nao foi bem sucedida temos que ler o codigo de erro e a
    // mensagem associada

    // Lendo codigo e tamanho da mensagem
    this.buffer.limit(PrimitiveTypeSize.BYTE.getSize()
        + PrimitiveTypeSize.SHORT.getSize());
    do {
      if (this.channel.read(this.buffer) == -1) {
        throw new IOException(REMOTE_DATA_CHANNEL_CLOSED);
      }
    } while (this.buffer.hasRemaining());

    this.buffer.flip();
    byte code = this.buffer.get();
    int msgSize = this.buffer.getChar();
    this.buffer.clear();

    ErrorCode errorCode = ErrorCode.valueOf(code);
    this.lastResultMessage.code = (errorCode == null) ? ErrorCode.UNKNOWN_ERROR
        : errorCode;

    if (msgSize <= 0) {
      this.lastResultMessage.message = "";
      return this.lastResultMessage;
    }

    // Lendo mensagem de erro
    this.buffer.limit(msgSize);
    do {
      if (this.channel.read(this.buffer) == -1) {
        throw new IOException(REMOTE_DATA_CHANNEL_CLOSED);
      }
    } while (this.buffer.hasRemaining());

    this.buffer.flip();
    byte[] msgArray = new byte[msgSize];
    this.buffer.get(msgArray);
    this.buffer.clear();

    this.lastResultMessage.message = new String(msgArray);

    return this.lastResultMessage;
  }

  /**
   * Libera os recursos alocados pelo arquivo.
   */
  private void release() {
    try {
      this.channel.close();
    } catch (IOException e) {
      // No fazer nada mesmo
      // e.printStackTrace();
    } finally {
      this.buffer = null;
      this.channel = null;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setSize(long size) throws PermissionException, FailureException {
    checkIsOpen();

    if (size < 0) {
      throw new IllegalArgumentException(String.format(ILLEGAL_NEGATIVE_NUMBER, "Size", size));
    }

    if (!this.writable) {
      throw new PermissionException(READ_ONLY_DATA_CHANNEL);
    }

    this.buffer.put(Operation.SET_SIZE.getCode());
    try {
      ByteBufferUtils.writeLong(this.buffer, this.channel,
          PrimitiveTypeSize.BYTE.getSize(), size);
      ResultMessage retMesg = readResultMessage();
      if (!retMesg.success) {
        switch (retMesg.code) {
          case UNSUPPORTED_OPERATION:
            throw new UnsupportedOperationException();
          case READ_ONLY:
            throw new PermissionException(READ_ONLY_DATA_CHANNEL);
          default:
            throw new FailureException(
              String.format(SERVER_FAILURE_OPERATION_ON_DATA_CHANNEL, "setting size", retMesg.message));
        }
      }
    } catch (IOException e) {
      this.release();
      throw new FailureException(e);
    } catch (UnexpectedProtocolMessage e) {
      this.release();
      throw new FailureException(e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public long getPosition() throws FailureException {
    checkIsOpen();

    try {
      ByteBufferUtils.writeByte(this.buffer, this.channel,
          Operation.GET_POSITION.getCode());

      ResultMessage retMesg = readResultMessage();

      if (!retMesg.success) {
        switch (retMesg.code) {
          case UNSUPPORTED_OPERATION:
            throw new UnsupportedOperationException();
          default:
            throw new FailureException(
              String.format(SERVER_FAILURE_OPERATION_ON_DATA_CHANNEL, "getting position", retMesg.message));
        }
      }

      long pos = ByteBufferUtils.readLong(this.buffer, this.channel);

      return pos;
    } catch (IOException e) {
      this.release();
      throw new FailureException(e);
    } catch (UnexpectedProtocolMessage e) {
      this.release();
      throw new FailureException(e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setPosition(long position) throws FailureException {
    checkIsOpen();

    if (position < 0) {
      throw new IllegalArgumentException(ILLEGAL_REMOTE_POSITION + position);
    }

    try {
      this.buffer.put(Operation.SET_POSITION.getCode());
      ByteBufferUtils.writeLong(this.buffer, this.channel,
          PrimitiveTypeSize.BYTE.getSize(), position);
      ResultMessage retMesg = readResultMessage();
      if (!retMesg.success) {
        switch (retMesg.code) {
          case UNSUPPORTED_OPERATION:
            throw new UnsupportedOperationException();
          default:
            throw new FailureException(
              String.format(SERVER_FAILURE_OPERATION_ON_DATA_CHANNEL, "setting position", retMesg.message));
        }
      }
    } catch (IOException e) {
      this.release();
      throw new FailureException(e);
    } catch (UnexpectedProtocolMessage e) {
      this.release();
      throw new FailureException(e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public long getSize() throws FailureException {
    checkIsOpen();

    try {
      ByteBufferUtils.writeByte(this.buffer, this.channel,
          Operation.GET_SIZE.getCode());

      ResultMessage retMesg = readResultMessage();

      if (!retMesg.success) {
        switch (retMesg.code) {
          case UNSUPPORTED_OPERATION:
            throw new UnsupportedOperationException();
          default:
            throw new FailureException(
              String.format(SERVER_FAILURE_OPERATION_ON_DATA_CHANNEL, "getting size", retMesg.message));
        }
      }

      long size = ByteBufferUtils.readLong(this.buffer, this.channel);

      return size;
    } catch (IOException e) {
      this.release();
      throw new FailureException(e);
    } catch (UnexpectedProtocolMessage e) {
      this.release();
      throw new FailureException(e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int read(ByteBuffer dest) throws FailureException {
    return this.read(dest, -1);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int read(ByteBuffer dest, long remotePosition) throws FailureException {
    checkIsOpen();

    if (dest == null) {
      throw new IllegalArgumentException(String.format(ILLEGAL_NULL_VALUE, "Destination byte buffer"));
    }

    if (remotePosition < -1) {
      throw new IllegalArgumentException(ILLEGAL_REMOTE_POSITION + remotePosition);
    }

    this.buffer.put(Operation.READ.getCode());
    this.buffer.putLong(remotePosition);
    this.buffer.putLong(dest.remaining());
    this.buffer.flip();
    try {
      this.channel.write(this.buffer);
    } catch (IOException e) {
      throw new FailureException(e);
    } finally {
      this.buffer.clear();
    }

    int originalLimit = dest.limit();
    int totalBytesRead = 0;
    while (dest.hasRemaining()) {
      try {
        ResultMessage retMesg = readResultMessage();

        if (!retMesg.success) {
          switch (retMesg.code) {
            case END_OF_FILE:

              if (totalBytesRead == 0) {
                return -1;
              }
              return totalBytesRead;

            case UNSUPPORTED_OPERATION:
              throw new UnsupportedOperationException();
            default:
              throw new FailureException(
                String.format(SERVER_FAILURE_OPERATION_ON_DATA_CHANNEL, "reading data", retMesg.message));
          }
        }

        int preChunkSize = ByteBufferUtils.readInt(this.buffer, this.channel);
        long chunkSize = (preChunkSize & 0xFFFFFFFFL);

        int missingBytes = originalLimit - dest.position();
        int limit = chunkSize < missingBytes ? (int) chunkSize : missingBytes;
        dest.limit(dest.position() + limit);

        int bytesRead = 0;
        int readCount = 0;
        while (bytesRead < chunkSize) {
          readCount = this.channel.read(dest);
          if (readCount < 0) {
            throw new FailureException(
              String.format(SERVER_FAILURE_OPERATION_ON_DATA_CHANNEL, "reading data", EARLIER_EOS));
          }
          bytesRead += readCount;
        }
        dest.limit(originalLimit);
        totalBytesRead += bytesRead;
      } catch (IOException e) {
        this.release();
        throw new FailureException(e);
      } catch (UnexpectedProtocolMessage e) {
        this.release();
        throw new FailureException(e);
      }
    }

    return totalBytesRead;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int write(ByteBuffer source) throws PermissionException,
      FailureException, FileLockedException {
    return this.write(source, -1);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int write(ByteBuffer source, long remotePosition)
      throws PermissionException, FailureException, FileLockedException {
    checkIsOpen();

    if (source == null) {
      throw new IllegalArgumentException(String.format(ILLEGAL_NULL_VALUE, "Source byte buffer"));
    }

    if (!this.writable) {
      throw new PermissionException(READ_ONLY_DATA_CHANNEL);
    }

    if (remotePosition < -1) {
      throw new IllegalArgumentException(ILLEGAL_REMOTE_POSITION + remotePosition);
    }

    this.buffer.put(Operation.WRITE.getCode());
    this.buffer.putLong(remotePosition);

    ResultMessage retMesg = null;
    try {
      ByteBufferUtils
          .writeLong(this.buffer, this.channel, (PrimitiveTypeSize.BYTE
              .getSize() + PrimitiveTypeSize.LONG.getSize()), source
              .remaining());
      retMesg = readResultMessage();
    } catch (IOException e) {
      this.release();
      throw new FailureException(e);
    } catch (UnexpectedProtocolMessage e) {
      this.release();
      throw new FailureException(e);
    }

    if (!retMesg.success) {
      switch (retMesg.code) {
        case UNSUPPORTED_OPERATION:
          throw new UnsupportedOperationException();
        case FILE_LOCKED:
          throw new FileLockedException(DATA_CHANNEL_ALREADY_LOCKED);
        case READ_ONLY:
          throw new PermissionException(READ_ONLY_DATA_CHANNEL);
        default:
          throw new FailureException(
            String.format(SERVER_FAILURE_OPERATION_ON_DATA_CHANNEL, "writing data", retMesg.message));
      }
    }

    try {
      return this.channel.write(source);
    } catch (IOException e) {
      throw new FailureException(e);
    }
  }

  /**
   * {@inheritDoc}
   * 
   * @throws IOException
   */
  @Override
  public long transferTo(long remotePosition, long count,
      WritableByteChannel dest) throws FailureException, IOException {
    checkIsOpen();

    if (dest == null) {
      throw new IllegalArgumentException(String.format(ILLEGAL_NULL_VALUE, "Destination byte channel"));
    }

    if (remotePosition < -1) {
      throw new IllegalArgumentException(ILLEGAL_REMOTE_POSITION + remotePosition);
    }

    if (count < 0) {
      throw new IllegalArgumentException(String.format(ILLEGAL_NEGATIVE_NUMBER, "Count", count));
    }

    this.buffer.put(Operation.READ.getCode());
    this.buffer.putLong(remotePosition);
    this.buffer.putLong(count);
    this.buffer.flip();
    try {
      this.channel.write(this.buffer);
    } catch (IOException e) {
      this.release();
      throw e;
    }

    this.buffer.clear();

    long bytesWrittenTotal = 0;
    long currentChunkSize = 0;
    int chunkReadBytes = 0;
    int bytesRead = 0;

    while (bytesWrittenTotal < count) {
      try {
        ResultMessage retMesg = readResultMessage();

        if (!retMesg.success) {

          switch (retMesg.code) {
            case END_OF_FILE:

              if (bytesWrittenTotal == 0) {
                return -1;
              }
              return bytesWrittenTotal;

            case UNSUPPORTED_OPERATION:
              throw new UnsupportedOperationException();
            default:
              throw new FailureException(
                String.format(SERVER_FAILURE_OPERATION_ON_DATA_CHANNEL, "reading data", retMesg.message));
          }
        }

        int preChunkSize = ByteBufferUtils.readInt(this.buffer, this.channel);
        currentChunkSize = (preChunkSize & 0xFFFFFFFFl);

      } catch (IOException e) {
        this.release();
        throw new FailureException(e);
      } catch (UnexpectedProtocolMessage e) {
        this.release();
        throw new FailureException(e);
      }

      chunkReadBytes = 0;
      while (chunkReadBytes < currentChunkSize) {
        this.buffer.clear();
        this.buffer.limit((int) (currentChunkSize - chunkReadBytes));
        try {
          bytesRead = this.channel.read(this.buffer);
          if (bytesRead < 0) {
            throw new FailureException(
              String.format(SERVER_FAILURE_OPERATION_ON_DATA_CHANNEL, "reading data", EARLIER_EOS));
          }

          this.buffer.flip();
          dest.write(this.buffer);

        } catch (IOException e) {
          this.buffer.clear();
          throw new FailureException(e);
        }
        bytesWrittenTotal += bytesRead;
        chunkReadBytes += bytesRead;
      }
      this.buffer.clear();
    }
    this.buffer.clear();

    return bytesWrittenTotal;
  }

  /**
   * {@inheritDoc}
   * 
   * @throws IOException
   */
  @Override
  public long transferFrom(ReadableByteChannel source, long remotePosition,
      long count) throws IOException, DataChannelException {
    checkIsOpen();

    if (!this.writable) {
      throw new PermissionException(READ_ONLY_DATA_CHANNEL);
    }

    if (source == null) {
      throw new IllegalArgumentException(String.format(ILLEGAL_NULL_VALUE, "Source byte channel"));
    }

    if (remotePosition < -1) {
      throw new IllegalArgumentException(ILLEGAL_REMOTE_POSITION + remotePosition);
    }

    if (count < 0) {
      throw new IllegalArgumentException(String.format(ILLEGAL_NEGATIVE_NUMBER, "Count", count));
    }

    this.buffer.put(Operation.WRITE.getCode());
    this.buffer.putLong(remotePosition);
    ResultMessage retMesg = null;
    try {
      ByteBufferUtils
          .writeLong(this.buffer, this.channel, (PrimitiveTypeSize.BYTE
              .getSize() + PrimitiveTypeSize.LONG.getSize()), count);
      retMesg = readResultMessage();
    } catch (IOException e) {
      this.buffer.clear();
      throw e;
    }

    if (!retMesg.success) {
      switch (retMesg.code) {
        case FILE_LOCKED:
          throw new FileLockedException(DATA_CHANNEL_ALREADY_LOCKED);
        case READ_ONLY:
          throw new PermissionException(READ_ONLY_DATA_CHANNEL);
        default:
          throw new FailureException(
            String.format(SERVER_FAILURE_OPERATION_ON_DATA_CHANNEL, "writing data", retMesg.message));
      }
    }

    long bytesReadTotal = 0;
    while (bytesReadTotal < count) {
      this.buffer.clear();
      if (this.buffer.limit() > count - bytesReadTotal) {
        this.buffer.limit((int) (count - bytesReadTotal));
      }
      try {
        int bytesRead = source.read(this.buffer);
        if (bytesRead == -1) {
          break;
        }
        bytesReadTotal += bytesRead;
        this.buffer.flip();

        while (this.buffer.hasRemaining()) {
          if (this.channel.write(this.buffer) < 0) {
            this.buffer.clear();
            throw new FailureException(
              String.format(SERVER_FAILURE_OPERATION_ON_DATA_CHANNEL, "writing data", EARLIER_EOS));
          }
        }
      } catch (IOException e) {
        this.buffer.clear();
        throw e;
      }

    }
    this.buffer.clear();
    return bytesReadTotal;
  }

  /**
   * {@inheritDoc}
   * 
   * @throws FailureException
   */
  @Override
  public void keepAlive() throws FailureException {
    checkIsOpen();

    try {
      ByteBufferUtils.writeByte(this.buffer, this.channel,
          Operation.KEEP_ALIVE.getCode());
      ResultMessage retMesg = readResultMessage();

      if (!retMesg.success) {
        throw new FailureException(
          String.format(SERVER_FAILURE_OPERATION_ON_DATA_CHANNEL, "sending keep-alive", retMesg.message));
      }
    } catch (IOException e) {
      this.release();
      throw new FailureException(e);
    } catch (UnexpectedProtocolMessage e) {
      this.release();
      throw new FailureException(e);
    }
  }

  /**
   * @param bufferSize O tamanho do buffer em bytes.
   */
  public void setBufferSize(int bufferSize) {
    if (bufferSize > MIN_BUFFER_SIZE) {
      this.bufferSize = bufferSize;
    }
  }

  /**
   * @return O tamanho do ByteBuffer Utilizado
   */
  public int getBufferSize() {
    return this.bufferSize;
  }

  @Override
  public short supportedOperations() {
    return this.operations;
  }

  @Override
  public long remaining() throws IOException, DataChannelException {
    return getSize() - getPosition();
  }

  @Override
  public long skip(long bytes) throws IOException, DataChannelException {
    setPosition(getPosition() + bytes);
    return bytes;
  }

  public ResultMessage getLastResultMessage() {
    return this.lastResultMessage;
  }

  /**
   * Mtodo auxiliar para verificar se o canal esta aberto.
   */
  private void checkIsOpen() {
    if (!isOpen()) {
      throw new IllegalStateException(REMOTE_DATA_CHANNEL_CLOSED);
    }
  }
}
