package tecgraf.ftc_1_4.common;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.WritableByteChannel;

import tecgraf.ftc_1_4.common.exception.DataChannelException;

/**
 * Classe que adapta um FileChannel para ser utilizado como um IDataChannel
 * 
 * @author Tecgraf
 */
public class FileDataChannel implements IDataChannel {

  /** FileChannel que sera utilizado como IDataChannel */
  private FileChannel channel;

  /** Operacoes suportadas por esta implementacao de IDataChannel */
  private final short supportedOps = IDataChannel.OP_GET_SIZE
    | IDataChannel.OP_SET_SIZE | IDataChannel.OP_GET_POSITION
    | IDataChannel.OP_SET_POSITION | IDataChannel.OP_READ
    | IDataChannel.OP_WRITE | IDataChannel.OP_TRANSFER_FROM
    | IDataChannel.OP_TRANSFER_TO;

  /**
   * @param channel
   */
  public FileDataChannel(FileChannel channel) {

    if (channel == null) {
      throw new NullPointerException();
    }

    this.channel = channel;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public short supportedOperations() {
    return supportedOps;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isOpen() {
    return getChannel().isOpen();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void close() throws IOException {
    getChannel().close();
  }

  @Override
  public long remaining() throws IOException, DataChannelException {
    long remaining = channel.size() - channel.position();

    if (remaining == 0) {
      return -1;
    }

    return remaining;
  }

  @Override
  public long skip(long bytes) throws IOException, DataChannelException {
    if (bytes < 0) {
      throw new IllegalArgumentException("Bytes to skip must be a non negative number: " + bytes);
    }
    long originalPosition = channel.position();

    long newPosition = channel.position() + bytes;
    if (newPosition > channel.size()) {
      newPosition = channel.size();
    }

    channel.position(newPosition);

    return newPosition - originalPosition;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public long getSize() throws IOException {
    return getChannel().size();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setSize(long size) throws IOException {
    if (size > getChannel().size()) {
      ByteBuffer src = ByteBuffer.wrap(new byte[] { 0 });
      long currentPosition = getChannel().position();
      getChannel().position(size - 1);
      getChannel().write(src);
      getChannel().position(currentPosition);
    }
    else {
      getChannel().truncate(size);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public long getPosition() throws IOException {
    return getChannel().position();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setPosition(long position) throws IOException {
    getChannel().position(position);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int read(ByteBuffer target) throws IOException {
    return getChannel().read(target);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int read(ByteBuffer target, long position) throws IOException {
    return getChannel().read(target, position);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int write(ByteBuffer source) throws IOException {
    return getChannel().write(source);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int write(ByteBuffer source, long position) throws IOException {
    return getChannel().write(source, position);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public long transferTo(long position, long count, WritableByteChannel output)
    throws IOException {
    long currentPos = channel.position();
    long readBytes = channel.transferTo(position, count, output);
    if (readBytes > 0) {
      channel.position(currentPos + readBytes);
    }
    return readBytes;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public long transferFrom(ReadableByteChannel source, long position, long count)
    throws IOException {

    // workaround para usar o transferFrom quando tamanho do arquivo
    // for menor que a posicao de escrita pedida
    if (channel.size() < position) {
      setSize(position + count - 1);
    }

    long currentPos = channel.position();
    long readBytes = channel.transferFrom(source, position, count);
    if (readBytes > 0) {
      channel.position(currentPos + readBytes);
    }
    return readBytes;
  }

  /**
   * Retorna o FileChannel encapsulado
   * 
   * @return FileChannel
   */
  public FileChannel getChannel() {
    return channel;
  }

}
