package tecgraf.ftc_1_4.common;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.Channel;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.WritableByteChannel;

import tecgraf.ftc_1_4.common.exception.DataChannelException;

/**
 * Representa um canal de acesso a dados.
 * 
 * @author Tecgraf
 */
public interface IDataChannel extends Channel {

  /** Constante que representa que o metodo getSize  suportado */
  short OP_GET_SIZE = 0x01;

  /** Constante que representa que o metodo setSize  suportado */
  short OP_SET_SIZE = 0x02;

  /** Constante que representa que o metodo getPosition  suportado */
  short OP_GET_POSITION = 0x04;

  /** Constante que representa que o metodo setPosition  suportado */
  short OP_SET_POSITION = 0x08;

  /** Constante que representa que os metodos de read so suportados */
  short OP_READ = 0x10;

  /** Constante que representa que os metodos de write so suportados */
  short OP_WRITE = 0x20;

  /** Constante que representa que o metodo transferFrom  suportado */
  short OP_TRANSFER_FROM = 0x40;

  /** Constante que representa que o metodo transferTo  suportado */
  short OP_TRANSFER_TO = 0x80;

  /**
   * Retornas os metodos/operaes suportados pela implementao do canal de
   * dados. Todas as implementaes devem definir as operaes suportadas. Se as
   * operaes no forem conhecidas, a constante ALL_OPS deve ser utilizada.
   * 
   * Implementao obrigatoria.
   * 
   * @return operacoes suportadas
   */
  short supportedOperations();

  /**
   * Verifica se o canal est aberto.
   * 
   * Implementao obrigatoria.
   * 
   * @return {@code true} caso o canal esteja aberto, ou {@code false}, caso
   *         contrrio.
   */
  @Override
  boolean isOpen();

  /**
   * Solicita o fechamento do canal.
   * 
   * Implementao obrigatoria.
   * 
   * @throws IOException
   *           Caso ocorra alguma falha no procedimento.
   * 
   */
  @Override
  void close() throws IOException;

  /**
   * Retorna a quantidade de bytes no canal a partir da posio atual.
   * 
   * O valor -1 significa end-of-stream.
   * 
   * O valor 0 significa que atualmente no h dados, mas pode vir a ter.
   * Utilizado em live-streams
   * 
   * Implementao obrigatoria.
   * 
   * @return quantidades de bytes existentes no canal.
   * 
   * @throws IOException
   *           Caso ocorra alguma falha no procedimento.
   * @throws DataChannelException
   *           Caso ocorra alguma situao excepcional no canal.
   */
  long remaining() throws IOException, DataChannelException;

  /**
   * Metodo que pula/ignora uma quantidade de bytes a partir da posicao atual.
   * 
   * Implementao obrigatoria.
   * 
   * @param bytes
   *          quantidade de bytes a serem pulados. Quantidade de bytes a serem
   *          pulados deve ser maior ou igual a zero.
   * 
   * @return Quantidade de bytes pulados
   * 
   * @throws IOException
   *           Caso ocorra alguma falha no procedimento.
   * @throws DataChannelException
   *           Caso ocorra alguma situao excepcional no canal.
   */
  long skip(long bytes) throws IOException, DataChannelException;

  /**
   * Obtm a quantidade de bytes no canal a partir da posio inicial.
   * 
   * Implementao opcional. No caso da operao de no ser suportada deve ser
   * lanada {@link UnsupportedOperationException}.
   * 
   * @return O tamanho do arquivo ou {@literal -1}, caso ocorra algum erro.
   * 
   * @throws IOException
   *           Caso ocorra alguma falha no procedimento.
   * @throws DataChannelException
   *           Caso ocorra alguma situao excepcional no canal.
   * @throws UnsupportedOperationException
   *           Caso a operao na seja suportada nesse canal.
   */
  long getSize() throws IOException, DataChannelException,
      UnsupportedOperationException;

  /**
   * Define a quantidade de bytes para o canal.
   * 
   * Implementao opcional. No caso da operao no ser suportada deve ser
   * lanada {@link UnsupportedOperationException}.
   * 
   * @param size
   *          O novo tamanho.
   * 
   * @throws IOException
   *           Caso ocorra alguma falha no procedimento.
   * @throws DataChannelException
   *           Caso ocorra alguma situao excepcional no canal.
   * @throws UnsupportedOperationException
   *           Caso a operao na seja suportada nesse canal.
   */
  void setSize(long size) throws IOException, DataChannelException,
      UnsupportedOperationException;

  /**
   * Obtm a posio atual do canal.
   * 
   * Implementao opcional. No caso da operao no ser suportada deve ser
   * lanada {@link UnsupportedOperationException}.
   * 
   * @return A posio atual do arquivo ou {@literal -1}, caso ocorra algum
   *         erro.
   * @throws IOException
   *           Caso ocorra alguma falha no procedimento.
   * @throws DataChannelException
   *           Caso ocorra alguma situao excepcional no canal.
   * @throws UnsupportedOperationException
   *           Caso a operao na seja suportada nesse canal.
   */
  long getPosition() throws IOException, DataChannelException,
      UnsupportedOperationException;

  /**
   * Define a nova posio do arquivo.
   * 
   * Implementao opcional. No caso da operao no ser suportada deve ser
   * lanada {@link UnsupportedOperationException}.
   * 
   * @param position
   *          A nova posio.
   * 
   * @throws IOException
   *           Caso ocorra alguma falha no procedimento.
   * @throws DataChannelException
   *           Caso ocorra alguma situao excepcional no canal.
   * @throws UnsupportedOperationException
   *           Caso a operao na seja suportada nesse canal.
   */
  void setPosition(long position) throws IOException,
      DataChannelException, UnsupportedOperationException;

  /**
   * L uma sequncia de bytes do arquivo.
   * 
   * Implementao opcional. No caso da operao no ser suportada deve ser
   * lanada {@link UnsupportedOperationException}.
   * 
   * @param target
   *          O buffer.
   * 
   * @return A quantidade de bytes lidos.
   * 
   * @throws IOException
   *           Caso ocorra alguma falha I/O no procedimento.
   * @throws DataChannelException
   *           Caso ocorra alguma situao excepcional no canal.
   * @throws UnsupportedOperationException
   *           Caso a operao na seja suportada nesse canal.
   */
  int read(ByteBuffer target) throws IOException, DataChannelException,
      UnsupportedOperationException;

  /**
   * L uma sequncia de bytes do arquivo.
   * 
   * Implementao opcional. No caso da operao no ser suportada deve ser
   * lanada {@link UnsupportedOperationException}.
   * 
   * @param target
   *          O buffer.
   * @param position
   *          A posio do arquivo a partir da qual a leitura vai ser iniciada.
   * 
   * @return A quantidade de bytes lidos.
   * 
   * @throws IOException
   *           Caso ocorra alguma falha no procedimento.
   * @throws DataChannelException
   *           Caso ocorra alguma situao excepcional no canal.
   * @throws UnsupportedOperationException
   *           Caso a operao na seja suportada nesse canal.
   */
  int read(ByteBuffer target, long position) throws IOException,
      DataChannelException, UnsupportedOperationException;

  /**
   * Escreve uma sequncia de bytes no arquivo.
   * 
   * Implementao opcional. No caso da operao no ser suportada deve ser
   * lanada {@link UnsupportedOperationException}.
   * 
   * @param source
   *          O buffer.
   * 
   * @return A quantidade de bytes escritos.
   * 
   * @throws IOException
   *           Caso ocorra alguma falha no procedimento.
   * @throws DataChannelException
   *           Caso ocorra alguma situao excepcional no canal.
   * @throws UnsupportedOperationException
   *           Caso a operao na seja suportada nesse canal.
   */
  int write(ByteBuffer source) throws IOException, DataChannelException,
      UnsupportedOperationException;

  /**
   * Escreve uma sequncia de bytes no arquivo.
   * 
   * Implementao opcional. No caso da operao no ser suportada deve ser
   * lanada {@link UnsupportedOperationException}.
   * 
   * @param source
   *          O buffer.
   * @param position
   *          A posio do arquivo a partir da qual a escrita vai ser iniciada.
   * 
   * @return A quantidade de bytes escritos.
   * 
   * @throws IOException
   *           Caso ocorra alguma falha no procedimento.
   * @throws DataChannelException
   *           Caso ocorra alguma situao excepcional no canal.
   * @throws UnsupportedOperationException
   *           Caso a operao na seja suportada nesse canal.
   */
  int write(ByteBuffer source, long position) throws IOException,
      DataChannelException, UnsupportedOperationException;

  /**
   * Transfere os dados do arquivo para um fluxo de sada.
   * 
   * Implementao opcional. No caso da operao no ser suportada deve ser
   * lanada {@link UnsupportedOperationException}.
   * 
   * @param output
   *          O fluxo de sada que receber os dados do arquivo.
   * @param position
   *          A posio inicial a partir daqual o arquivo ser lido.
   * @param count
   *          A quantidade de bytes que sero transferidos.
   * 
   * @return A quantidade de bytes efetivamente transferidos.
   * 
   * @throws IOException
   *           Caso ocorra alguma falha no procedimento.
   * @throws DataChannelException
   *           Caso ocorra alguma situao excepcional no canal.
   * @throws UnsupportedOperationException
   *           Caso a operao na seja suportada nesse canal.
   */
  long transferTo(long position, long count, WritableByteChannel output)
      throws IOException, DataChannelException, UnsupportedOperationException;

  /**
   * Transfere os dados de um fluxo de entrada para o arquivo.
   * 
   * Implementao opcional. No caso da operao no ser suportada deve ser
   * lanada {@link UnsupportedOperationException}.
   * 
   * @param source
   *          O fluxo de entrada.
   * @param position
   *          A posio inicial a partir daqual o arquivo ser escrito.
   * @param count
   *          A quantidade de bytes que sero transferidos.
   * 
   * @return A quantidade de bytes efetivamente transferidos.
   * 
   * @throws IOException
   *           Caso ocorra alguma falha no procedimento.
   * @throws DataChannelException
   *           Caso ocorra alguma situao excepcional no canal.
   * @throws UnsupportedOperationException
   *           Caso a operao na seja suportada nesse canal.
   */
  long transferFrom(ReadableByteChannel source, long position, long count)
      throws IOException, DataChannelException, UnsupportedOperationException;

}