package tecgraf.ftc_1_4.server;

import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.Arrays;

import tecgraf.ftc_1_4.common.exception.InvalidArraySize;

import static tecgraf.ftc_1_4.server.ErrorMessages.EXCEED_MAXIMUM_LENGTH;

/**
 * Representa uma chave de acesso a um canal de dados.
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class AccessKey {
  /**
   * O nome do algoritmo de gerao de nmeros pseudo-aleatrios.
   */
  private static final String PRNG_ALGORITHM = "SHA1PRNG";
  /**
   * O tamanho da chave de acesso.
   */
  public static final int DEFAULT_KEY_SIZE = 16;

  /**
   * O tamanho maximo da chave de acesso.
   */
  public static final int MAX_KEY_SIZE = 255;

  /**
   * A chave propriamente dita.
   */
  private byte[] key;

  /**
   * Cria uma chave de acesso a um canal de dados.
   */
  public AccessKey() {
    SecureRandom secureRandom = null;
    try {
      secureRandom = SecureRandom.getInstance(PRNG_ALGORITHM);
    }
    catch (NoSuchAlgorithmException e) {
      secureRandom = new SecureRandom();
    }
    this.key = new byte[DEFAULT_KEY_SIZE];
    secureRandom.nextBytes(this.key);
  }

  /**
   * Cria uma chave de acesso a um canal de dados a partir de um conjunto de
   * bytes.
   * 
   * @param bytes O conjunto de bytes, que deve possuir tamanho
   *        {@value #DEFAULT_KEY_SIZE}.
   * @throws InvalidArraySize caso o conjunto de bytes for maior que {@value #MAX_KEY_SIZE}
   */
  public AccessKey(byte[] bytes) throws InvalidArraySize {
    if (bytes.length > MAX_KEY_SIZE)
      throw new InvalidArraySize(String.format(
        EXCEED_MAXIMUM_LENGTH, "access key", bytes.length, MAX_KEY_SIZE));
    this.key = new byte[bytes.length];
    System.arraycopy(bytes, 0, this.key, 0, bytes.length);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean equals(Object obj) {
    if (obj == null) {
      return false;
    }
    if (!this.getClass().equals(obj.getClass())) {
      return false;
    }
    AccessKey accessKey = (AccessKey) obj;
    return Arrays.equals(this.key, accessKey.key);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int hashCode() {
    return Arrays.hashCode(this.key);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    return ErrorMessages.hexString(this.key);
  }

  /**
   * Obtm os bytes que compem a chave de acesso.
   * 
   * @return Os bytes que compem a chave de acesso.
   */
  byte[] getBytes() {
    byte[] bytes = new byte[key.length];
    System.arraycopy(this.key, 0, bytes, 0, key.length);
    return bytes;
  }
}
