package tecgraf.ftc_1_4.server;

import java.io.FileNotFoundException;
import java.io.RandomAccessFile;
import java.io.UnsupportedEncodingException;
import java.util.logging.Logger;

import tecgraf.ftc_1_4.common.FileDataChannel;
import tecgraf.ftc_1_4.common.IDataChannel;
import tecgraf.ftc_1_4.common.exception.FailureException;
import tecgraf.ftc_1_4.common.exception.PermissionException;
import tecgraf.ftc_1_4.utils.Utils;

import static tecgraf.ftc_1_4.server.ErrorMessages.FILE_CHANNEL_WAS_CLOSED_FILE_ID;

/**
 * Classe que implementa um provedor de arquivos (FileDataChannel) para o
 * servidor de arquivos.
 * 
 * @author Tecgraf
 */
public class DefaultDataProvider implements DataChannelProvider {

  /**
   * Objeto responsvel por registrar as atividades do servidor.
   */
  private final static Logger logger = Logger.getLogger("tecgraf.ftc");

  /**
   * {@inheritDoc}
   */
  @Override
  public IDataChannel createFileChannel(Object requester, byte[] fileId,
    boolean readOnly) throws PermissionException, FailureException {
    String absolutePath;
    try {
      absolutePath = new String(fileId, Utils.CHARSET_ENCODING);
    }
    catch (UnsupportedEncodingException e) {
      throw new FailureException(e);
    }
    String mode = "r";
    if (!readOnly) {
      mode += "w";
    }
    RandomAccessFile raf;
    try {
      raf = new RandomAccessFile(absolutePath, mode);
    }
    catch (FileNotFoundException e) {
      return null;
    }
    return new FileDataChannel(raf.getChannel());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isLocked(Object requester, byte[] fileId) {
    return false;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void fileChannelClosed(Object requester, byte[] fileId,
    ChannelClosedReason reason) {
    String absolutePath;
    try {
      absolutePath = new String(fileId, Utils.CHARSET_ENCODING);
    }
    catch (UnsupportedEncodingException e) {
      absolutePath = new String(fileId);
      e.printStackTrace();
    }
    System.err.println(String.format(FILE_CHANNEL_WAS_CLOSED_FILE_ID,
      requester != null ? requester : "<none>", absolutePath, reason));
  }

}
