package tecgraf.ftc_1_4.server;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.ServerSocket;
import java.net.SocketAddress;
import java.nio.channels.SelectionKey;
import java.nio.channels.Selector;
import java.nio.channels.ServerSocketChannel;
import java.nio.channels.SocketChannel;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

import tecgraf.ftc_1_4.common.exception.InvalidArraySize;
import tecgraf.ftc_1_4.server.states.State;

import static tecgraf.ftc_1_4.server.ErrorMessages.CLIENT_CONNECTED_SUCCESSFULLY;
import static tecgraf.ftc_1_4.server.ErrorMessages.CONNECTION_CLOSED_DUE_TO_CHANNEL_ERROR;
import static tecgraf.ftc_1_4.server.ErrorMessages.CONNECTION_CLOSED_DUE_TO_CLIENT_TIMEOUT;
import static tecgraf.ftc_1_4.server.ErrorMessages.DISCARDING_CLIENT_AFTER_FAILURE;
import static tecgraf.ftc_1_4.server.ErrorMessages.EXCEED_MAXIMUM_LENGTH;
import static tecgraf.ftc_1_4.server.ErrorMessages.FAILED_ACCEPTING_CONNECTION;
import static tecgraf.ftc_1_4.server.ErrorMessages.FAILED_DISCARDING_CONNECTION_AFTER_FAILURE;
import static tecgraf.ftc_1_4.server.ErrorMessages.FAILED_TO_CLOSE_THE_SERVER_SOCKET_SELECTOR;
import static tecgraf.ftc_1_4.server.ErrorMessages.FAILED_TO_REGISTER_SERVER_SOCKET_SELECTOR;
import static tecgraf.ftc_1_4.server.ErrorMessages.FAILED_TO_SELECT_FOR_IO_OPERATIONS;
import static tecgraf.ftc_1_4.server.ErrorMessages.NO_RESOURCES_AVAILABLE;
import static tecgraf.ftc_1_4.server.ErrorMessages.REQUEST_DISCARDED;
import static tecgraf.ftc_1_4.server.ErrorMessages.REQUEST_DISCARDED_DUE_TO_TIMEOUT;
import static tecgraf.ftc_1_4.server.ErrorMessages.REQUEST_REGISTERED;
import static tecgraf.ftc_1_4.server.ErrorMessages.SERVER_REACHED_MAXIMUM_OF_CLIENTS;
import static tecgraf.ftc_1_4.server.ErrorMessages.SERVER_SHUTTING_DOWN_COMPLETED;
import static tecgraf.ftc_1_4.server.ErrorMessages.SERVER_SHUTTING_DOWN_STARTED;
import static tecgraf.ftc_1_4.server.ErrorMessages.SERVER_STARTUP_COMPLETED;
import static tecgraf.ftc_1_4.server.ErrorMessages.UNEXPECTED_EXCEPTION_DURING_STATE_READ_OPERATION;
import static tecgraf.ftc_1_4.server.ErrorMessages.UNEXPECTED_EXCEPTION_DURING_STATE_READ_OPERATION_FILE_ID;
import static tecgraf.ftc_1_4.server.ErrorMessages.UNEXPECTED_EXCEPTION_DURING_STATE_WRITE_OPERATION;
import static tecgraf.ftc_1_4.server.ErrorMessages.UNEXPECTED_EXCEPTION_DURING_STATE_WRITE_OPERATION_FILE_ID;
import static tecgraf.ftc_1_4.server.ErrorMessages.hexString;

/**
 * Representa o servidor de arquivos.
 *
 * @author Tecgraf/PUC-Rio
 */
public final class FileServer {

  /**
   * O tamanho mximo que o array do identificador do arquivo pode assumir
   */
  public static final int FILEID_MAX_LENGTH = 255;
  /**
   * O dono do servidor de arquivos.
   */
  private DataChannelProvider fileProvider;
  /**
   * Indica se o tratamento de requisies deve ser interrompido.
   */
  private volatile boolean wasStopped;
  /**
   * Indica se o servidor ja foi inicializado.
   */
  protected boolean initialized = false;
  /**
   * Gerenciador de conexes. Usado para selecionar as conexes ativas.
   */
  private Selector selector;
  /**
   * Mapeia uma chave de acesso s informaes sobre a requisio de um arquivo.
   */
  private Map<AccessKey, FileChannelRequestInfo> channels;

  /**
   * Objeto com as configuraes utilizadas pelo servidor.
   */
  private FileServerConfig config = null;

  /**
   * Socket do servidor
   */
  private ServerSocketChannel serverChannel = null;

  /**
   * SelectionKey do servidor
   */
  SelectionKey serverKey = null;

  /**
   * Tempo que indica a ultima verificao de conexes expiradas.
   */
  private long lastTimeoutCheck = 0;

  /**
   * Objeto o qual as excecoes capturadas pelo servidor serao repassadas.
   */
  private FileServerExceptionHandler exceptionHandler = null;

  /**
   * Objeto responsvel por registrar as atividades do servidor.
   */
  private final static Logger logger = Logger.getLogger("tecgraf.ftc");

  /**
   * Varivel que indica se estamos usando java 7
   */
  public final static boolean PLATAFORM_HAS_TRANSFERTO_BUG;

  static {
    if (System.getProperty("os.name").contains("Linux")
        && System.getProperty("sun.arch.data.model").contains("32")
        && (!System.getProperty("java.version").contains("1.7.0"))) {
      PLATAFORM_HAS_TRANSFERTO_BUG = true;
    } else {
      PLATAFORM_HAS_TRANSFERTO_BUG = false;
    }
  }

  /**
   * Cria um servidor de arquivos.
   *
   * @param fileProvider O objeto que prove os arquivos.
   *
   * @throws IOException Caso ocorra algum erro na criao do servidor.
   */
  public FileServer(DataChannelProvider fileProvider) throws IOException {
    this.fileProvider = fileProvider;
    this.config = new FileServerConfigImpl();
    this.channels = new HashMap<AccessKey, FileChannelRequestInfo>();
  }

  /**
   * @return O objeto que prove os arquivos
   */
  public DataChannelProvider getDataProvider() {
    return this.fileProvider;
  }

  /**
   * Metodo que faz a inicializao do servidor.
   *
   * @return false se houver algum erro durante a inicializao
   */
  public boolean serverSetup() {
    if (!this.initialized) {

      try {
        this.serverChannel = ServerSocketChannel.open();
        this.serverChannel.configureBlocking(false);
        ServerSocket serverSocket = this.serverChannel.socket();
        serverSocket.bind(new InetSocketAddress(this.config.getHostName(),
            this.config.getPort()));

        this.selector = Selector.open();
        this.serverKey = this.serverChannel.register(this.selector,
            SelectionKey.OP_ACCEPT);
      } catch (Exception e) {
        this.serverKey = null;
        this.selector = null;
        this.serverChannel = null;
        if (logger.isLoggable(Level.SEVERE)) {
          logger.log(Level.SEVERE, FAILED_TO_REGISTER_SERVER_SOCKET_SELECTOR, e);
        }
        this.exceptionRaised(e);
        return false;
      }
      if (logger.isLoggable(Level.INFO)) {
        logger.info(String.format(SERVER_STARTUP_COMPLETED,
          this.config.getHostName(), this.serverChannel.socket().getLocalPort()));
      }
      this.initialized = true;
    }

    return this.initialized;
  }

  /**
   * Inicia o tratamento de eventos.
   *
   */
  public void dispatch() {
    if (!serverSetup()) {
      return;
    }

    while (!this.wasStopped) {
      select();
    }
    shutdownConnections();
  }

  private void select() {
    try {
      int selectedKeyCount = this.selector.select(this.config
          .getSelectTimeout());

      checkTimedOutConnections();

      if (selectedKeyCount == 0) {
        return;
      }
    } catch (IOException e) {
      if (logger.isLoggable(Level.WARNING)) {
        logger.log(Level.WARNING, FAILED_TO_SELECT_FOR_IO_OPERATIONS, e);
      }
      this.exceptionRaised(e);
      return;
    }

    Iterator<SelectionKey> selectedKeys = this.selector.selectedKeys()
        .iterator();

    while (selectedKeys.hasNext()) {
      SelectionKey key = selectedKeys.next();
      selectedKeys.remove();

      if (key.isValid()) {
        if (this.serverKey.equals(key)) {
          if (key.isAcceptable()) {
            this.accept(key);
          }
        } else {
          this.read(key);

          if (key.isValid() && key.isWritable()) {
            this.write(key);
          }

          Session session = (Session) key.attachment();
          if (key.isValid() && session.isValid()) {
            int ops = SelectionKey.OP_READ;
            if (session.isWriting()) {
              ops |= SelectionKey.OP_WRITE;
            }
            key.interestOps(ops);
          }
        }
      }
    }
  }

  /**
   * Procura conexoes inativas por um determinado tempo.
   */
  private synchronized void checkTimedOutConnections() {
    long currentTime = System.currentTimeMillis();
    if (currentTime - this.lastTimeoutCheck < this.config.getSelectTimeout()) {
      return;
    }

    this.lastTimeoutCheck = currentTime;

    Set<SelectionKey> keys = this.selector.keys();

    for (SelectionKey key : keys) {
      Session session = (Session) key.attachment();

      // Ignorando a chave do servidor
      if (session == null) {
        continue;
      }

      long test = currentTime - session.getLastActivity();

      if (test > this.config.getClientTimeout()) {
        if (logger.isLoggable(Level.WARNING)) {
          logger.warning(String.format(CONNECTION_CLOSED_DUE_TO_CLIENT_TIMEOUT,
            session.getChannel().socket().getRemoteSocketAddress(), this.config.getClientTimeout()));
        }
        stopConnection(key, session, ChannelClosedReason.CHANNEL_TIMEOUT);
      }
    }

    Set<Entry<AccessKey, FileChannelRequestInfo>> channelSet = this.channels
        .entrySet();
    Iterator<Entry<AccessKey, FileChannelRequestInfo>> iter = channelSet
        .iterator();
    while (iter.hasNext()) {
      Entry<AccessKey, FileChannelRequestInfo> entry = iter.next();
      FileChannelRequestInfo requestInfo = entry.getValue();

      long test = currentTime - requestInfo.getCreationTime();
      if (test > this.config.getChannelRequestTimeout()) {
        if (logger.isLoggable(Level.WARNING)) {
          logger.warning(String.format( REQUEST_DISCARDED_DUE_TO_TIMEOUT,
            entry.getKey(), this.config.getChannelRequestTimeout()));
        }
        iter.remove();
      }

    }

  }

  /**
   * Fecha todas as conexoes
   */
  private void shutdownConnections() {
    if (logger.isLoggable(Level.INFO)) {
      logger.info(SERVER_SHUTTING_DOWN_STARTED);
    }
    for (SelectionKey key : this.selector.keys()) {
      Session session = (Session) key.attachment();
      stopConnection(key, session, ChannelClosedReason.SERVER_SHUTDOWN);
    }
    try {
      this.selector.close();
    } catch (IOException e) {
      if (logger.isLoggable(Level.WARNING)) {
        logger.log(Level.WARNING, FAILED_TO_CLOSE_THE_SERVER_SOCKET_SELECTOR, e);
      }
    }
    this.serverChannel = null;
    if (logger.isLoggable(Level.INFO)) {
      logger.info(SERVER_SHUTTING_DOWN_COMPLETED);
    }
  }

  /**
   * Solicita a interrupo do tratamento de requisies.
   */
  public void stop() {
    this.wasStopped = true;
  }

  /**
   * Cria uma descrio de canal de arquivo.
   *
   * @param requester O requisitante.
   * @param fileId O identificador do arquivo.
   *
   * @return A descrio de canal de arquivo.
   * @throws InvalidArraySize Quando chave passada  invalida
   * @throws MaxChannelRequestsException Quando o limite de canais nao
   *           consumidos  atingido
   */
  public FileChannelAccessInfo createFileChannelInfo(Object requester,
      byte[] fileId) throws InvalidArraySize, MaxChannelRequestsException {
    return createFileChannelInfo(requester, fileId, null);
  }

  /**
   * Cria uma descrio de canal de arquivo.
   *
   * @param requester O requisitante.
   * @param fileId O identificador do arquivo.
   * @param accessKey Chave de acesso ao arquivo.
   *
   * @return A descrio de canal de arquivo.
   * @throws InvalidArraySize Quando chave passada  invalida
   * @throws MaxChannelRequestsException Quando o limite de canais nao
   *           consumidos  atingido
   */
  public FileChannelAccessInfo createFileChannelInfo(Object requester,
      byte[] fileId, byte[] accessKey) throws InvalidArraySize,
      MaxChannelRequestsException {
    return createFileChannelInfo(requester, fileId, accessKey, true);
  }

  /**
   * Cria uma descrio de canal de arquivo.
   *
   * @param requester O requisitante.
   * @param fileId O identificador do arquivo.
   * @param accessKey Chave de acesso ao arquivo.
   * @param useTransferTo Indica se o metodo FileChannel.transferTo pode ser
   *          utilizado.
   *
   * @return A descrio de canal de arquivo.
   * @throws InvalidArraySize Quando chave passada  invalida
   * @throws MaxChannelRequestsException Quando o limite de canais nao
   *           consumidos  atingido
   */
  public synchronized FileChannelAccessInfo createFileChannelInfo(
      Object requester, byte[] fileId, byte[] accessKey, boolean useTransferTo)
      throws InvalidArraySize, MaxChannelRequestsException {

    if (fileId == null) {
      throw new IllegalArgumentException("fileId cannot be null");
    }

    if (fileId.length > FILEID_MAX_LENGTH) {
      throw new InvalidArraySize(String.format(EXCEED_MAXIMUM_LENGTH,"file id", fileId.length, FILEID_MAX_LENGTH));
    }

    if (this.channels.size() + 1 > this.config.getMaxChannelRequests()) {
      throw new MaxChannelRequestsException("reached the maximum of " +
        this.config.getMaxChannelRequests() + " pending channel requests");
    }

    AccessKey key = (accessKey != null) ? new AccessKey(accessKey)
        : new AccessKey();
    FileChannelRequestInfo fileChannelIinfo = new FileChannelRequestInfo(
        requester, fileId);

    fileChannelIinfo.useTransferTo(useTransferTo);

    this.channels.put(key, fileChannelIinfo);

    if (logger.isLoggable(Level.INFO)) {
      logger.info(String.format(REQUEST_REGISTERED, fileChannelIinfo.getCreationTime(),
        key, hexString(fileId)));
    }
    return new FileChannelAccessInfo(this.config.getHostName(),
        this.serverChannel.socket().getLocalPort(), key.getBytes(), fileId);
  }

  /**
   * Remove um canal registrado anteriormente
   *
   * @param key Chave do canal
   * @return {@link FileChannelRequestInfo} criado anteriormente.
   */
  public synchronized FileChannelRequestInfo removeFileChannelInfo(AccessKey key) {
    FileChannelRequestInfo result = this.channels.remove(key);
    if (result != null) {
      if (logger.isLoggable(Level.INFO)) {
        logger.info(String.format(REQUEST_DISCARDED, System.currentTimeMillis(), key, hexString(result.getFileId())));
      }
    }
    return result;
  }

  /**
   * Obtm as informaes sobre a requisio de um arquivo, a partir de uma
   * chave de acesso.
   *
   * @param accessKey A chave de acesso.
   *
   * @return As informaes sobre a requisio de um arquivo.
   */
  public synchronized FileChannelRequestInfo getFileChannelInfo(AccessKey accessKey) {
    // TODO: por qu esse mtodo no  protected ?
    // Se estiver em modo de teste, no remover a chave
    if (this.config.isTestMode()) {
      return this.channels.get(accessKey);
    }
    return this.channels.remove(accessKey);
  }

  /**
   * Invocado quando uma exceo  lanada no servidor.
   *
   * @param e A exceo lanada.
   * @param fileId Identificador do arquivo envolvido no momento do erro
   */
  public void exceptionRaised(Exception e, byte[] fileId) {
    if (this.exceptionHandler != null) {
      this.exceptionHandler.exceptionRaised(e, fileId);
    }
  }

  /**
   * Invocado quando uma exceo  lanada no servidor.
   *
   * @param e A exceo lanada.
   */
  public void exceptionRaised(Exception e) {
    if (this.exceptionHandler != null) {
      this.exceptionHandler.exceptionRaised(e);
    }
  }

  /**
   * Trata uma tentativa de conexo a um {@link ServerSocketChannel}
   *
   * @param key A chave selecionada.
   */
  private void accept(SelectionKey key) {
    ServerSocketChannel serverChannel = (ServerSocketChannel) key.channel();
    SocketChannel socketChannel = null;

    try {
      while (true) {
        socketChannel = serverChannel.accept();

        if (socketChannel == null) {
          break;
        }

        socketChannel.configureBlocking(false);

        Session session = new Session(socketChannel, this);
        session.markLastActivity();

        socketChannel.register(this.selector, SelectionKey.OP_READ, session);

        SocketAddress clientAddress = socketChannel.socket().getRemoteSocketAddress();

        int clients = this.selector.keys().size() - 1;

        if (clients > this.config.getMaxClients()) {
          if (logger.isLoggable(Level.WARNING)) {
            logger.warning(String.format(SERVER_REACHED_MAXIMUM_OF_CLIENTS, clientAddress, this.config.getMaxClients()));
          }
          session.setMaxClientsReached(true);
          break;
        }

        if (logger.isLoggable(Level.INFO)) {
          logger.info(String.format(CLIENT_CONNECTED_SUCCESSFULLY, clientAddress, clients, this.config.getMaxClients()));
        }

        if (!this.config.acceptMaxPossible()) {
          break;
        }

      }
      return;
    } catch (IOException e) {
      if (logger.isLoggable(Level.SEVERE)) {
        StringBuilder details = new StringBuilder();
        if (socketChannel != null) {
          details.append(" of client ");
          details.append(socketChannel.socket().getRemoteSocketAddress());
        }
        logger.log(Level.SEVERE,String.format(FAILED_ACCEPTING_CONNECTION, details.toString()), e);
      }
      this.exceptionRaised(e);
    } catch (OutOfMemoryError e) {
      if (logger.isLoggable(Level.SEVERE)) {
        logger.log(Level.SEVERE, NO_RESOURCES_AVAILABLE, e);
      }
    }

    if (socketChannel != null) {
      SocketAddress clientAddress = socketChannel.socket().getRemoteSocketAddress();
      try {
        socketChannel.close();
        if (logger.isLoggable(Level.WARNING)) {
          logger.warning(String.format(DISCARDING_CLIENT_AFTER_FAILURE, clientAddress));
        }
      } catch (IOException e) {
        if (logger.isLoggable(Level.SEVERE)) {
          logger.log(Level.SEVERE, String.format(FAILED_DISCARDING_CONNECTION_AFTER_FAILURE, clientAddress), e);
        }
        this.exceptionRaised(e);
      }
    }
    return;
  }

  /**
   * Trata uma tentativa de leitura em um {@link SocketChannel}
   *
   * @param key A chave selecionada.
   */
  private void read(SelectionKey key) {
    Session session = (Session) key.attachment();
    State currentState = null;
    try {
      currentState = session.getCurrentState();
      if ((currentState == null) || (!currentState.read(session))) {
        if (logger.isLoggable(Level.WARNING)) {
          logger.warning(String.format(CONNECTION_CLOSED_DUE_TO_CHANNEL_ERROR, getAddress(session), getStateId(currentState)));
        }
        this.stopConnection(key, session, ChannelClosedReason.CHANNEL_ERROR);
      }
    } catch (Exception e) {
      String currentStateId = getStateId(currentState);
      if (session != null && session.getFileChannelInfo() != null) {
        byte[] fileId = session.getFileChannelInfo().getFileId();
        if (logger.isLoggable(Level.SEVERE)) {
          logger.log(Level.SEVERE, String.format(UNEXPECTED_EXCEPTION_DURING_STATE_READ_OPERATION_FILE_ID,
            currentStateId, session.getChannel().socket().getRemoteSocketAddress(), hexString(fileId)), e);
        }
        this.exceptionRaised(e, fileId);
      } else {
        if (logger.isLoggable(Level.SEVERE)) {
          logger.log(Level.SEVERE, String.format(
            UNEXPECTED_EXCEPTION_DURING_STATE_READ_OPERATION, currentStateId, getAddress(session)), e);
        }
        this.exceptionRaised(e);
      }
      this.stopConnection(key, session, ChannelClosedReason.CHANNEL_ERROR);
    }
  }

  /**
   * Trata uma tentativa de escrita em um {@link SocketChannel}
   *
   * @param key A chave selecionada.
   */
  private void write(SelectionKey key) {
    Session session = (Session) key.attachment();
    State currentState = null;
    try {
      currentState = session.getCurrentState();
      if ((currentState == null) || (!currentState.write(session))) {
        if (logger.isLoggable(Level.WARNING)) {
          logger.warning(String.format(CONNECTION_CLOSED_DUE_TO_CHANNEL_ERROR, getAddress(session), getStateId(currentState)));
        }
        this.stopConnection(key, session, ChannelClosedReason.CHANNEL_ERROR);
      }
    } catch (Exception e) {
      String currentStateId = getStateId(currentState);
      if (session != null && session.getFileChannelInfo() != null) {
        byte[] fileId = session.getFileChannelInfo().getFileId();
        if (logger.isLoggable(Level.SEVERE)) {
          logger.log(Level.SEVERE, String.format(UNEXPECTED_EXCEPTION_DURING_STATE_WRITE_OPERATION_FILE_ID,
            currentStateId, session.getChannel().socket().getRemoteSocketAddress(), hexString(fileId)), e);
        }
        this.exceptionRaised(e, fileId);
      } else {
        if (logger.isLoggable(Level.SEVERE)) {
          logger.log(Level.SEVERE, String.format(
            UNEXPECTED_EXCEPTION_DURING_STATE_WRITE_OPERATION, currentStateId, getAddress(session)), e);
        }
        this.exceptionRaised(e);
      }
      this.stopConnection(key, session, ChannelClosedReason.CHANNEL_ERROR);
    }
  }

  private String getStateId(State currentState) {
    return (currentState != null)
      ? currentState.getClass().getCanonicalName()
      : "<unknown>";
  }

  private String getAddress(Session session) {
    return (session != null)
      ? session.getChannel().socket().getRemoteSocketAddress().toString()
      : "<no client attached>";
  }

  /**
   * Finaliza uma conexo com um cliente.
   *
   * @param key A chave de seleo do canal do cliente.
   * @param session A sesso do cliente.
   * @param reason Razo pela qual o canal foi fechado.
   */
  private void stopConnection(SelectionKey key, Session session,
      ChannelClosedReason reason) {
    if (session != null) {
      session.close(reason);
    } else {
      try {
        key.channel().close();
      } catch (IOException e) {
        if (logger.isLoggable(Level.WARNING)) {
          logger.log(Level.WARNING, String.format(FAILED_DISCARDING_CONNECTION_AFTER_FAILURE, "<no session associated>"), e);
        }
        this.exceptionRaised(e);
      }
    }
    key.attach(null);
    key.cancel();
  }

  /**
   * Retorna Objeto com as configuraes utilizadas pelo servidor
   *
   * @return config
   */
  public FileServerConfig getConfig() {
    return this.config;
  }

  /**
   * @param config Objeto com as configuraes utilizadas pelo servidor
   */
  public void setConfig(FileServerConfig config) {
    this.config = config;
  }

  /**
   * @return O exception handler cadastrado
   */
  public FileServerExceptionHandler getExceptionHandler() {
    return this.exceptionHandler;
  }

  /**
   * Cadastra um exception handler para receber as excecoes do servidor.
   *
   * @param exceptionHandler
   */
  public void setExceptionHandler(FileServerExceptionHandler exceptionHandler) {
    this.exceptionHandler = exceptionHandler;
  }
}
