package tecgraf.ftc_1_4.server;

import java.io.IOException;
import java.net.SocketAddress;
import java.nio.ByteBuffer;
import java.nio.channels.SocketChannel;
import java.util.logging.Level;
import java.util.logging.Logger;

import tecgraf.ftc_1_4.common.IDataChannel;
import tecgraf.ftc_1_4.server.states.State;
import tecgraf.ftc_1_4.server.states.SupportedProtocols;
import tecgraf.ftc_1_4.server.states.v1_4.VersionHandshakeState;

import static tecgraf.ftc_1_4.server.ErrorMessages.FAILED_DISCARDING_CONNECTION_AFTER_FAILURE;
import static tecgraf.ftc_1_4.server.ErrorMessages.FAILED_DISCARDING_CONNECTION_AFTER_FAILURE_FILE_ID;
import static tecgraf.ftc_1_4.server.ErrorMessages.hexString;

/**
 * Representa uma sesso de acesso a um arquivo no servidor.
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class Session {
  /**
   * O canal de comunicao com o cliente.
   */
  private SocketChannel channel = null;
  /**
   * O <i>buffer</i> para a comunicao com o cliente.
   */
  private ByteBuffer buffer = null;
  /**
   * O servidor de arquivos.
   */
  private FileServer fileServer = null;
  /**
   * Protocolo utilizado pelo cliente atual
   */
  private SupportedProtocols protocol = SupportedProtocols.V1_4;
  /**
   * O estado atual da sesso.
   */
  private State currentState = null;
  /**
   * Informaes sobre a requisio do canal de arquivos.
   */
  private FileChannelRequestInfo fileChannelInfo = null;
  /**
   * O canal do arquivo.
   */
  private IDataChannel fileChannel = null;
  /**
   * Indica se o arquivo foi aberto como somente leitura ou no.
   */
  private boolean readOnly = true;
  /**
   * Marca o tempo da ultima atividade em milisegundos.
   */
  private long lastActivity = 0;
  /**
   * Indica se o numero maximo de clientes foi atingido nessa sessao
   */
  private boolean maxClients = false;
  /**
   * Indica se a sesso  valida.
   */
  private boolean valid = true;
  /**
   * Objeto responsvel por registrar as atividades do servidor.
   */
  private final static Logger logger = Logger.getLogger("tecgraf.ftc");
  /**
   * Cria uma sesso.
   * 
   * @param channel O canal de comunicao com o cliente.
   * @param fileServer O servidor de arquivos.
   */
  Session(SocketChannel channel, FileServer fileServer) {
    this.channel = channel;
    this.buffer = ByteBuffer.allocateDirect(fileServer.getConfig()
        .getClientBufferSize());
    this.fileServer = fileServer;
    this.currentState = new VersionHandshakeState();
  }

  /**
   * Obtm o canal de comunicao com o cliente.
   * 
   * @return O canal de comunicao com o cliente.
   */
  public SocketChannel getChannel() {
    return this.channel;
  }

  /**
   * Obtm o <i>buffer</i> para comunicao com o cliente.
   * 
   * @return O <i>buffer</i> para a comunicao com o cliente.
   */
  public ByteBuffer getBuffer() {
    return this.buffer;
  }

  /**
   * Obtm o servidor de arquivos.
   * 
   * @return O servidor de arquivos.
   */
  public FileServer getFileServer() {
    return this.fileServer;
  }

  /**
   * Obtm o estado corrente.
   * 
   * @return O estado corrente.
   */
  public State getCurrentState() {
    return this.currentState;
  }

  /**
   * Define o estado corrente.
   * 
   * @param state O estado corrente.
   */
  public void setCurrentState(State state) {

    this.currentState = state;
  }

  /**
   * Obtm informaes sobre a requisio do canal de arquivos.
   * 
   * @return Informaes sobre a requisio do canal de arquivos.
   */
  public FileChannelRequestInfo getFileChannelInfo() {
    return this.fileChannelInfo;
  }

  /**
   * Define as informaes sobre a requisio do canal de arquivos.
   * 
   * @param fileChannelInfo Informaes sobre a requisio do canal de arquivos.
   */
  public void setFileChannelInfo(FileChannelRequestInfo fileChannelInfo) {
    this.fileChannelInfo = fileChannelInfo;
  }

  /**
   * Obtm o canal do arquivo.
   * 
   * @return O canal do arquivo, ou {@code null}, caso no tenha sido definido.
   */
  public IDataChannel getFileChannel() {
    return this.fileChannel;
  }

  /**
   * Define o canal do arquivo.
   * 
   * @param fileChannel O canal do arquivo.
   */
  public void setFileChannel(IDataChannel fileChannel) {
    this.fileChannel = fileChannel;
  }

  /**
   * Verifica se o canal do arquivo foi aberto como somente leitura.
   * 
   * @return {@code true} caso o canal do arquivo tenha sido aberto como somente
   *         leitura, ou {@code false}.
   */
  public boolean isReadOnly() {
    return this.readOnly;
  }

  /**
   * Define se o canal do arquivo foi aberto como somente leitura.
   * 
   * @param readOnly {@code true} caso o canal do arquivo tenha sido aberto como
   *          somente leitura, ou {@code false}.
   */
  public void setReadOnly(boolean readOnly) {
    this.readOnly = readOnly;
  }

  /**
   * Fecha a sesso.
   *
   * @param reason Razo do fechamento da sesso.
   */
  public void close(ChannelClosedReason reason) {
    this.closeFileChannel(reason);
    try {
      this.channel.close();
    } catch (IOException e) {
      SocketAddress clientAddress = this.channel.socket().getRemoteSocketAddress();
      if (this.fileChannelInfo != null) {
        byte[] fileId = this.fileChannelInfo.getFileId();
        if (logger.isLoggable(Level.WARNING)) {
          logger.log(Level.WARNING, String.format(FAILED_DISCARDING_CONNECTION_AFTER_FAILURE_FILE_ID,
            clientAddress, hexString(fileId)), e);
        }
        this.fileServer.exceptionRaised(e, fileId);
      } else {
        if (logger.isLoggable(Level.WARNING)) {
          logger.log(Level.WARNING, String.format(FAILED_DISCARDING_CONNECTION_AFTER_FAILURE, clientAddress), e);
        }
        this.fileServer.exceptionRaised(e);
      }
    }
    this.valid = false;
    this.currentState = null;
  }

  /**
   * Fecha o canal do arquivo e notifica ao servidor de arquivos sobre este
   * fechamento.
   * 
   * @param reason Razo do fechamento do canal.
   * 
   * @return {@code true} caso o arquivo tenha sido aberto anteriormente ou
   *         {@code false}, caso contrrio.
   */
  public boolean closeFileChannel(ChannelClosedReason reason) {
    if (this.fileChannel == null) {
      return false;
    }
    try {
      this.fileChannel.close();
    } catch (IOException e) {
      if (this.fileChannelInfo != null) {
        this.fileServer.exceptionRaised(e, this.fileChannelInfo.getFileId());
      } else {
        this.fileServer.exceptionRaised(e);
      }
    } finally {
      this.fileChannel = null;
      this.fileServer.getDataProvider().fileChannelClosed(
          this.fileChannelInfo.getRequester(),
          this.fileChannelInfo.getFileId(), reason);
    }
    return true;
  }

  /**
   * Retorna o hora (milisegundos) da ultima atividade realizada.
   * 
   * @return lastActivity
   */
  public long getLastActivity() {
    return this.lastActivity;
  }

  /**
   * marca a hora da ultima atividade
   */
  public void markLastActivity() {
    this.lastActivity = System.currentTimeMillis();
  }

  /**
   * Retorna o procolo utilizado nessa sessao
   * 
   * @return protocolo
   */
  public SupportedProtocols getProtocol() {
    return this.protocol;
  }

  /**
   * Define o protocolo utilizado pelo cliente nessa sessao.
   * 
   * @param protocol
   */
  public void setProtocol(SupportedProtocols protocol) {
    this.protocol = protocol;
  }

  /**
   * Define se essa sessao deve ser fechada por ter atingido o numero maximo de
   * clientes.
   * 
   * @param value
   */
  public void setMaxClientsReached(boolean value) {
    this.maxClients = value;
  }

  /**
   * Indica se essa sessao deve ser fechada por ter atingido o numero maximo de
   * clientes.
   * 
   * @return True de se atingiu
   */
  public boolean getMaxClientsReached() {
    return this.maxClients;
  }

  /**
   * Indica se o servidor esta interessado em eventos de escrita desse canal.
   * 
   * @return boolean se estiver interessado
   */
  public boolean isWriting() {
    if (this.currentState == null) {
      return false;
    }
    return this.currentState.isWriting();
  }

  /**
   * Indica se a sesso  valida (nao foi fechada).
   * 
   * @return true se a sesso for valida
   */
  public boolean isValid() {
    return this.valid;
  }
}
