package tecgraf.ftc_1_4.server.states.v1_1;

import java.io.IOException;
import java.net.SocketAddress;
import java.nio.ByteBuffer;
import java.nio.channels.SocketChannel;
import java.util.logging.Level;
import java.util.logging.Logger;

import tecgraf.ftc_1_4.common.IDataChannel;
import tecgraf.ftc_1_4.common.logic.ErrorCode;
import tecgraf.ftc_1_4.common.logic.PrimitiveTypeSize;
import tecgraf.ftc_1_4.server.Session;
import tecgraf.ftc_1_4.server.states.State;

import static tecgraf.ftc_1_4.server.ErrorMessages.DATA_CHANNEL_DOESNT_SUPPORT_GET_POSITION;
import static tecgraf.ftc_1_4.server.ErrorMessages.FAILED_TO_READ_DATA_CHANNEL_POSITION;
import static tecgraf.ftc_1_4.server.ErrorMessages.OPERATION_GET_POSITION_RESULT_BUFFERED;
import static tecgraf.ftc_1_4.server.ErrorMessages.OPERATION_GET_POSITION_RESULT_SENT;
import static tecgraf.ftc_1_4.server.ErrorMessages.hexString;

/**
 * Operao de obteno da posio atual do arquivo.
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class GetPositionState implements State {
  /**
   * Representa os estados internos desta operao.
   * 
   * @author Tecgraf/PUC-Rio
   */
  private enum InternalState {
    /**
     * O estado inicial.
     */
    INITIAL,
    /**
     * Estado que indica que a posio do arquivo j foi lida.
     */
    POSITION_READ;
  }

  /**
   * O estado atual da operao.
   */
  private InternalState currentState = InternalState.INITIAL;

  /**
   * A posio do arquivo.
   */
  private long position = -1;

  /**
   * Objeto responsvel por registrar as atividades do servidor.
   */
  private final static Logger logger = Logger.getLogger("tecgraf.ftc");

  /**
   * Indica se o estado esta interessado em eventos de escrita desse canal.
   */
  private boolean writing = true;

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean read(Session session) {
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean write(Session session) throws IOException {
    ByteBuffer buffer = session.getBuffer();
    SocketChannel channel = session.getChannel();
    SocketAddress clientAddress = channel.socket().getRemoteSocketAddress();
    switch (this.currentState) {
      case INITIAL:
        IDataChannel byteChannel = session.getFileChannel();
        if ((byteChannel.supportedOperations() & IDataChannel.OP_GET_POSITION) != 0) {
          try {
            this.position = byteChannel.getPosition();
          } catch (Exception e) {
            byte[] fileId = session.getFileChannelInfo().getFileId();
            if (logger.isLoggable(Level.SEVERE)) {
              logger.log(Level.SEVERE, String.format(FAILED_TO_READ_DATA_CHANNEL_POSITION,
                clientAddress, hexString(fileId)), e);
            }
            session.getFileServer().exceptionRaised(e, fileId);
          }
        } else {
          if (logger.isLoggable(Level.WARNING)) {
            logger.warning(String.format(DATA_CHANNEL_DOESNT_SUPPORT_GET_POSITION,
              clientAddress, hexString(session.getFileChannelInfo().getFileId())));
          }
        }
        buffer.limit(PrimitiveTypeSize.LONG.getSize());
        buffer.putLong(this.position);
        buffer.flip();
        if (logger.isLoggable(Level.FINEST)) {
          logger.finest(String.format(OPERATION_GET_POSITION_RESULT_BUFFERED,
            (this.position != -1) ? ErrorCode.OK : ErrorCode.FAILURE, clientAddress));
        }
        this.currentState = InternalState.POSITION_READ;

      case POSITION_READ:
        if (channel.write(buffer) > 0) {
          session.markLastActivity();
        }
        if (buffer.hasRemaining()) {
          return true;
        }
        this.writing = false;
        buffer.clear();
        if (logger.isLoggable(Level.FINEST)) {
          logger.finest(String.format(OPERATION_GET_POSITION_RESULT_SENT,
            ((this.position != -1) ? ErrorCode.OK : ErrorCode.FAILURE) + " value: " + this.position, clientAddress));
        }
        session.setCurrentState(new GetOperationState());
        return true;
      default:
        return true;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isWriting() {
    return this.writing;
  }
}
