package tecgraf.ftc_1_4.server.states.v1_1;

import java.io.IOException;
import java.net.SocketAddress;
import java.nio.ByteBuffer;
import java.nio.channels.SocketChannel;
import java.util.logging.Level;
import java.util.logging.Logger;

import tecgraf.ftc_1_4.common.IDataChannel;
import tecgraf.ftc_1_4.common.exception.FailureException;
import tecgraf.ftc_1_4.common.exception.PermissionException;
import tecgraf.ftc_1_4.common.logic.ErrorCode;
import tecgraf.ftc_1_4.common.logic.PrimitiveTypeSize;
import tecgraf.ftc_1_4.server.FileChannelRequestInfo;
import tecgraf.ftc_1_4.server.FileServer;
import tecgraf.ftc_1_4.server.Session;
import tecgraf.ftc_1_4.server.states.State;

import static tecgraf.ftc_1_4.server.ErrorMessages.OPERATION_OPEN_OLD_FILE_ID_ARRAY_RECEIVED;
import static tecgraf.ftc_1_4.server.ErrorMessages.OPERATION_OPEN_OLD_FILE_ID_LENGTH_RECEIVED;
import static tecgraf.ftc_1_4.server.ErrorMessages.OPERATION_OPEN_RESULT_BUFFERED;
import static tecgraf.ftc_1_4.server.ErrorMessages.OPERATION_OPEN_RESULT_SENT_SUCCESSFULLY;
import static tecgraf.ftc_1_4.server.ErrorMessages.hexString;

/**
 * Operao de abertura de um arquivo.
 * 
 * @author Tecgraf/PUC-Rio
 */
public abstract class OpenState implements State {
  /**
   * Representa os estados internos desta operao.
   * 
   * @author Tecgraf/PUC-Rio
   */
  protected enum InternalState {
    /**
     * O estado inicial.
     */
    INITIAL,
    /**
     * Estado que indica que o tamanho do identificador do arquivo j foi
     * recebido.
     */
    IDENTIFIER_SIZE_RECEIVED,
    /**
     * Estado que indica que o identificador do arquivo j foi recebido.
     */
    IDENTIFIER_RECEIVED;
  }

  /**
   * O estado atual da operao.
   */
  private InternalState currentState = InternalState.INITIAL;
  /**
   * O tamanho do identificador do arquivo.
   */
  private int fileIdSize;
  /**
   * O identificador do arquivo.
   */
  private byte[] fileId;
  /**
   * O cdigo de erro.
   */
  private ErrorCode errorCode;

  /**
   * Indica se o arquivo ser aberto somente para leitura ou se ser aberto para
   * leitura e gravao.
   */
  private boolean readOnly;

  /**
   * Objeto responsvel por registrar as atividades do servidor.
   */
  private final static Logger logger = Logger.getLogger("tecgraf.ftc");

  /**
   * Indica se o estado esta interessado em eventos de escrita desse canal.
   */
  private boolean writing = false;

  /**
   * Cria uma operao de abertura de arquivo.
   * 
   * @param readOnly Indica se o arquivo ser aberto somente para leitura ou se
   *          ser aberto para leitura e gravao.
   */
  protected OpenState(boolean readOnly) {
    this.readOnly = readOnly;
    if (logger.isLoggable(Level.FINEST)) {
      logger.finest(OpenState.class.getCanonicalName() + " readOnly=" + readOnly);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean read(Session session) throws IOException {
    ByteBuffer buffer = session.getBuffer();
    SocketChannel channel = session.getChannel();
    SocketAddress clientAddress = channel.socket().getRemoteSocketAddress();
    switch (this.currentState) {
      case INITIAL:
        buffer.limit(PrimitiveTypeSize.BYTE.getSize());
        if (channel.read(buffer) < 0) {
          return false;
        } else {
          session.markLastActivity();
        }
        if (buffer.hasRemaining()) {
          return true;
        }
        buffer.flip();
        this.fileIdSize = buffer.get() & 0xFF;
        buffer.clear();
        if (logger.isLoggable(Level.FINEST)) {
          logger.finest(String.format(OPERATION_OPEN_OLD_FILE_ID_LENGTH_RECEIVED, this.fileIdSize, clientAddress));
        }
        this.currentState = InternalState.IDENTIFIER_SIZE_RECEIVED;
      case IDENTIFIER_SIZE_RECEIVED:
        buffer.limit(this.fileIdSize);
        if (channel.read(buffer) < 0) {
          return false;
        } else {
          session.markLastActivity();
        }
        if (buffer.hasRemaining()) {
          return true;
        }
        buffer.flip();
        this.fileId = new byte[this.fileIdSize];
        buffer.get(this.fileId);
        buffer.clear();
        if (logger.isLoggable(Level.FINEST)) {
          logger.finest(String.format(OPERATION_OPEN_OLD_FILE_ID_ARRAY_RECEIVED, hexString(this.fileId), clientAddress));
        }
        FileChannelRequestInfo fileChannelInfo = session.getFileChannelInfo();
        FileServer fileServer = session.getFileServer();
        try {
          IDataChannel fileChannel = fileServer.getDataProvider()
              .createFileChannel(fileChannelInfo.getRequester(),
                  fileChannelInfo.getFileId(), this.readOnly);
          if (fileChannel == null) {
            this.errorCode = ErrorCode.FILE_NOT_FOUND;
          } else {
            this.errorCode = ErrorCode.OK;
            session.setFileChannel(fileChannel);
          }
        } catch (FailureException e) {
          this.errorCode = ErrorCode.FAILURE;
        } catch (PermissionException e) {
          this.errorCode = ErrorCode.NO_PERMISSION;
        }
        buffer.limit(PrimitiveTypeSize.BYTE.getSize());
        buffer.put(this.errorCode.getCode());
        buffer.flip();
        if (logger.isLoggable(Level.FINEST)) {
          logger.finest(String.format(OPERATION_OPEN_RESULT_BUFFERED, this.errorCode, clientAddress));
        }
        this.writing = true;
        this.currentState = InternalState.IDENTIFIER_RECEIVED;
        return true;
      default:
        return true;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean write(Session session) throws IOException {
    ByteBuffer buffer = session.getBuffer();
    SocketChannel channel = session.getChannel();
    switch (this.currentState) {
      case IDENTIFIER_RECEIVED:
        if (channel.write(buffer) > 0) {
          session.markLastActivity();
        }
        if (buffer.hasRemaining()) {
          return true;
        }
        this.writing = false;
        buffer.clear();
        if (logger.isLoggable(Level.FINEST)) {
          SocketAddress clientAddress = channel.socket().getRemoteSocketAddress();
          logger.finest(String.format(OPERATION_OPEN_RESULT_SENT_SUCCESSFULLY, this.errorCode, clientAddress));
        }
        session.setCurrentState(new GetOperationState());
        if (!this.errorCode.equals(ErrorCode.OK)) {
          return false;
        }
        return true;
      default:
        return true;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isWriting() {
    return this.writing;
  }
}
