package tecgraf.ftc_1_4.server.states.v1_1;

import java.io.IOException;
import java.net.SocketAddress;
import java.nio.ByteBuffer;
import java.nio.channels.SocketChannel;
import java.util.logging.Level;
import java.util.logging.Logger;

import tecgraf.ftc_1_4.common.IDataChannel;
import tecgraf.ftc_1_4.common.exception.DataChannelException;
import tecgraf.ftc_1_4.common.logic.PrimitiveTypeSize;
import tecgraf.ftc_1_4.server.FileServer;
import tecgraf.ftc_1_4.server.Session;
import tecgraf.ftc_1_4.server.states.State;

import static tecgraf.ftc_1_4.server.ErrorMessages.OPERATION_READ_COUNT_REQUESTED;
import static tecgraf.ftc_1_4.server.ErrorMessages.OPERATION_READ_START_POSITION;
import static tecgraf.ftc_1_4.server.ErrorMessages.OPERATION_READ_STATS;
import static tecgraf.ftc_1_4.server.ErrorMessages.OPERATION_READ_USING_TRANSFER_TO;

/**
 * Operao para leitura de dados a partir de uma determinada posio.
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class ReadState implements State {
  /**
   * Representa os estados internos desta operao.
   * 
   * @author Tecgraf/PUC-Rio
   */
  private enum InternalState {
    /**
     * O estado inicial.
     */
    INITIAL,
    /**
     * Estado que indica que a posio a partir da qual os dados sero lidos j
     * foi lida.
     */
    POSITION_READ,
    /**
     * Estado que indica que a quantidade de bytes que sero lidos j foi lida.
     */
    BYTE_COUNT_READ,
    /**
     * Estado que indica que todos os bytes solicitados j foram enviados.
     */
    BYTES_SENT;
  }

  /**
   * O estado atual da operao.
   */
  private InternalState currentState = InternalState.INITIAL;

  /**
   * A posio a partir da qual os dados sero lidos do arquivo.
   */
  private long position;
  /**
   * Quantidade de bytes que sero lidos do arquivo.
   */
  private long count;
  /**
   * Indica se metodo transferTo deve ser utilizado.
   */
  private boolean readUsesTransferTo = false;

  /**
   * Objeto responsvel por registrar as atividades do servidor.
   */
  private final static Logger logger = Logger.getLogger("tecgraf.ftc");

  /**
   * Tamanho maximo a ser transmitido na chamada do metodo transferTo do
   * fileChannel.
   */
  public static final long MAX_BYTES = Integer.MAX_VALUE;

  /**
   * A quantidade de bytes enviados ao cliente.
   */
  private long bytesSent = 0;

  /**
   * Indica se o estado esta interessado em eventos de escrita desse canal.
   */
  private boolean writing = false;

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean read(Session session) throws IOException {
    ByteBuffer buffer = session.getBuffer();
    SocketChannel channel = session.getChannel();
    IDataChannel dataChannel = session.getFileChannel();
    SocketAddress clientAddress = channel.socket().getRemoteSocketAddress();
    switch (this.currentState) {
      case INITIAL:
        buffer.limit(PrimitiveTypeSize.LONG.getSize());
        if (channel.read(buffer) < 0) {
          return false;
        } else {
          session.markLastActivity();
        }

        if (buffer.hasRemaining()) {
          return true;
        }

        buffer.flip();
        this.position = buffer.getLong();
        buffer.clear();

        this.currentState = InternalState.POSITION_READ;

        if (logger.isLoggable(Level.FINEST)) {
          logger.finest(String.format(OPERATION_READ_START_POSITION, clientAddress, this.position));
        }
      case POSITION_READ:
        buffer.limit(PrimitiveTypeSize.LONG.getSize());
        if (channel.read(buffer) < 0) {
          return false;
        } else {
          session.markLastActivity();
        }
        if (buffer.hasRemaining()) {
          return true;
        }
        buffer.flip();
        this.count = buffer.getLong();
        buffer.clear();

        this.readUsesTransferTo = ((dataChannel.supportedOperations() & IDataChannel.OP_TRANSFER_TO) != 0)
            && (session.getFileChannelInfo().useTransferTo())
            && ((this.count <= MAX_BYTES) || (!FileServer.PLATAFORM_HAS_TRANSFERTO_BUG));

        this.currentState = InternalState.BYTE_COUNT_READ;

        this.writing = true;

        buffer.limit(0);

        if (logger.isLoggable(Level.FINEST)) {
          logger.finest(String.format(OPERATION_READ_COUNT_REQUESTED, clientAddress, this.count));
        }
        return true;
      default:
        return true;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean write(Session session) throws IOException, DataChannelException {
    SocketChannel channel = session.getChannel();
    SocketAddress clientAddress = channel.socket().getRemoteSocketAddress();
    switch (this.currentState) {
      case BYTE_COUNT_READ:
        ByteBuffer buffer = session.getBuffer();
        IDataChannel fileChannel = session.getFileChannel();
        long bytesToRead = this.count - this.bytesSent;
        long bytesWritten = transferBytesToSocket(fileChannel, channel, buffer, bytesToRead);
        // Na FTC-95 houve melhoria que passou a sinalizar END_OF_FILE para verses 1.2 e 1.4 do protocolo
        // A verso 1.1 no possui essa caracterstica, portanto  preciso retornar falso para fechar o canal com erro
        if (bytesWritten < 0) {
          return false;
        } else {
          session.markLastActivity();
          this.bytesSent += bytesWritten;
        }

        if (logger.isLoggable(Level.FINE)) {
          logger.fine(String.format(OPERATION_READ_STATS, clientAddress, bytesWritten, this.bytesSent, this.count));
        }

        if (this.bytesSent == this.count) {
          this.writing = false;
          buffer.clear();
          this.currentState = InternalState.BYTES_SENT;
          session.setCurrentState(new GetOperationState());
        }
        return true;
      default:
        return true;
    }
  }

  /**
   * Metodo auxiliar para transferir bytes do canal de dados para o socket
   * 
   * @param byteChannel
   * @param channel
   * @param buffer
   * @param bytesToRead
   * 
   * @return
   * 
   * @throws IOException
   * @throws DataChannelException
   * @throws UnsupportedOperationException
   */
  private long transferBytesToSocket(IDataChannel byteChannel,
      SocketChannel channel, ByteBuffer buffer, long bytesToRead)
      throws IOException, DataChannelException {
    long bytesWritten = -1;

    if (this.readUsesTransferTo) {
      logger.finest(String.format(OPERATION_READ_USING_TRANSFER_TO, channel.socket().getRemoteSocketAddress()));
      if (this.position == -1) {
        this.position = byteChannel.getPosition();
      }

      bytesWritten = byteChannel.transferTo(this.position + this.bytesSent,
          bytesToRead, channel);
    } else {

      if (buffer.hasRemaining()) {
        bytesWritten = channel.write(buffer);
      } else {
        if (this.position >= 0) {
          byteChannel.setPosition(this.position + this.bytesSent);
        }

        buffer.clear();
        if (buffer.remaining() > bytesToRead) {
          buffer.limit((int) bytesToRead);
        }

        int bytesRead = byteChannel.read(buffer);

        if (bytesRead == -1) {
          return -1;
        }

        buffer.flip();

        bytesWritten = channel.write(buffer);
      }

    }

    return bytesWritten;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isWriting() {
    return this.writing;
  }
}
