package tecgraf.ftc_1_4.server.states.v1_1;

import java.io.IOException;
import java.net.SocketAddress;
import java.nio.ByteBuffer;
import java.nio.channels.SocketChannel;
import java.util.logging.Level;
import java.util.logging.Logger;

import tecgraf.ftc_1_4.common.exception.InvalidArraySize;
import tecgraf.ftc_1_4.common.logic.ErrorCode;
import tecgraf.ftc_1_4.common.logic.PrimitiveTypeSize;
import tecgraf.ftc_1_4.server.AccessKey;
import tecgraf.ftc_1_4.server.FileChannelRequestInfo;
import tecgraf.ftc_1_4.server.Session;
import tecgraf.ftc_1_4.server.states.State;

import static tecgraf.ftc_1_4.server.ErrorMessages.OPERATION_KEY_SIZE_RECEIVED;
import static tecgraf.ftc_1_4.server.ErrorMessages.OPERATION_VALIDATE_RESULT_BUFFERED;
import static tecgraf.ftc_1_4.server.ErrorMessages.OPERATION_VALIDATE_RESULT_SENT;
import static tecgraf.ftc_1_4.server.ErrorMessages.REQUEST_NOT_FOUND;
import static tecgraf.ftc_1_4.server.ErrorMessages.REQUEST_RETRIEVED;

/**
 * Operao de obteno e validao de uma chave de acesso.
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class ValidateKeyState implements State {
  /**
   * Representa os estados internos desta operao.
   * 
   * @author Tecgraf/PUC-Rio
   */
  private enum InternalState {
    /**
     * O estado inicial.
     */
    INITIAL,
    /**
     * Estado que indica que o tamanho da chave j foi recebido.
     */
    KEY_SIZE_RECEIVED,
    /**
     * Estado que indica que a chave j foi recebida.
     */
    KEY_RECEIVED,
    /**
     * Estado que indica que o cdigo de erro j foi enviado.
     */
    ERROR_CODE_SENT;
  }

  /**
   * O estado atual da operao.
   */
  private InternalState currentState = InternalState.INITIAL;
  /**
   * O tamanho da chave de acesso.
   */
  private int keySize;
  /**
   * A chave de acesso.
   */
  private byte[] key;
  /**
   * O cdigo de erro.
   */
  private ErrorCode errorCode;

  /**
   * Objeto responsvel por registrar as atividades do servidor.
   */
  private final static Logger logger = Logger.getLogger("tecgraf.ftc");

  /**
   * Indica se o estado esta interessado em eventos de escrita desse canal.
   */
  private boolean writing = false;

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean read(Session session) throws IOException {
    ByteBuffer buffer = session.getBuffer();
    SocketChannel channel = session.getChannel();
    SocketAddress clientAddress = channel.socket().getRemoteSocketAddress();
    switch (this.currentState) {
      case INITIAL:
        buffer.limit(PrimitiveTypeSize.BYTE.getSize());
        if (channel.read(buffer) < 0) {
          return false;
        } else {
          session.markLastActivity();
        }

        if (buffer.hasRemaining()) {
          return true;
        }

        buffer.flip();
        this.keySize = buffer.get() & 0xFF;
        buffer.clear();

        if (logger.isLoggable(Level.FINEST)) {
          logger.finest(String.format(OPERATION_KEY_SIZE_RECEIVED, clientAddress, this.keySize));
        }

        this.currentState = InternalState.KEY_SIZE_RECEIVED;
      case KEY_SIZE_RECEIVED:
        buffer.limit(this.keySize);
        if (channel.read(buffer) < 0) {
          return false;
        } else {
          session.markLastActivity();
        }

        if (buffer.hasRemaining()) {
          return true;
        }

        buffer.flip();
        this.key = new byte[this.keySize];
        buffer.get(this.key);
        buffer.clear();

        this.writing = true;

        try {
          AccessKey accessKey = new AccessKey(this.key);
          long timestamp = System.currentTimeMillis();
          FileChannelRequestInfo fileChannelInfo = session.getFileServer().getFileChannelInfo(accessKey);
          if (fileChannelInfo == null) {
            this.errorCode = ErrorCode.INVALID_KEY;
            if (logger.isLoggable(Level.WARNING)) {
              logger.warning(String.format(REQUEST_NOT_FOUND, timestamp, clientAddress, accessKey));
            }
          } else {
            this.errorCode = ErrorCode.OK;
            session.setFileChannelInfo(fileChannelInfo);
            if (logger.isLoggable(Level.INFO)) {
              logger.info(String.format(REQUEST_RETRIEVED, timestamp, clientAddress, accessKey));
            }
          }
        } catch (InvalidArraySize e) {
          this.errorCode = ErrorCode.INVALID_KEY;
        }

        this.currentState = InternalState.KEY_RECEIVED;
        return true;
      default:
        return true;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean write(Session session) throws IOException {
    ByteBuffer buffer = session.getBuffer();
    SocketChannel channel = session.getChannel();
    SocketAddress clientAddress = channel.socket().getRemoteSocketAddress();
    switch (this.currentState) {
      case KEY_RECEIVED:
        buffer.limit(PrimitiveTypeSize.BYTE.getSize());
        buffer.put(this.errorCode.getCode());
        buffer.flip();
        if (logger.isLoggable(Level.FINEST)) {
          logger.finest(String.format(OPERATION_VALIDATE_RESULT_BUFFERED, this.errorCode, clientAddress));
        }
        if (channel.write(buffer) > 0) {
          session.markLastActivity();
        }
        if (buffer.hasRemaining()) {
          return true;
        }
        this.writing = false;
        buffer.clear();
        if (logger.isLoggable(Level.FINEST)) {
          logger.finest(String.format(OPERATION_VALIDATE_RESULT_SENT, this.errorCode, clientAddress));
        }
        this.currentState = InternalState.ERROR_CODE_SENT;
        session.setCurrentState((this.errorCode.equals(ErrorCode.OK)) ? new GetOperationState() : null);
        return true;
      default:
        return true;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isWriting() {
    return this.writing;
  }
}
