package tecgraf.ftc_1_4.server.states.v1_2;

import java.io.IOException;
import java.net.SocketAddress;
import java.nio.ByteBuffer;
import java.nio.channels.SocketChannel;
import java.util.logging.Level;
import java.util.logging.Logger;

import tecgraf.ftc_1_4.common.logic.ErrorCode;
import tecgraf.ftc_1_4.common.logic.PrimitiveTypeSize;
import tecgraf.ftc_1_4.server.Session;
import tecgraf.ftc_1_4.server.states.State;

import static tecgraf.ftc_1_4.server.ErrorMessages.OPERATION_KEEP_ALIVE_RESULT_SENT;

/**
 * Operao que mantem uma conexao ativa.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class KeepAliveState implements State {

  /**
   * Representa os estados internos desta operao.
   * 
   * @author Tecgraf/PUC-Rio
   */
  private enum InternalState {
    /**
     * O estado inicial.
     */
    INITIAL, RETURN_MSG_WRITTEN
  }

  /**
   * O estado atual da operao.
   */
  private InternalState currentState = InternalState.INITIAL;

  /**
   * Objeto responsvel por registrar as atividades do servidor.
   */
  private final static Logger logger = Logger.getLogger("tecgraf.ftc");

  /**
   * Indica se o estado esta interessado em eventos de escrita desse canal.
   */
  private boolean writing = true;

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean read(Session session) {
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean write(Session session) throws IOException {
    ByteBuffer buffer = session.getBuffer();
    SocketChannel channel = session.getChannel();
    switch (this.currentState) {
      case INITIAL:
        buffer.limit(PrimitiveTypeSize.BYTE.getSize());
        buffer.put(ErrorCode.OK.getCode());
        buffer.flip();
        this.currentState = InternalState.RETURN_MSG_WRITTEN;

      case RETURN_MSG_WRITTEN:
        if (channel.write(buffer) > 0) {
          session.markLastActivity();
        }
        if (buffer.hasRemaining()) {
          return true;
        }
        this.writing = false;
        buffer.clear();
        if (logger.isLoggable(Level.FINEST)) {
          SocketAddress clientAddress = channel.socket().getRemoteSocketAddress();
          logger.finest(String.format(OPERATION_KEEP_ALIVE_RESULT_SENT, ErrorCode.OK, clientAddress));
        }
        session.setCurrentState(new GetOperationState());
        return true;
      default:
        return true;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isWriting() {
    return this.writing;
  }
}
