package tecgraf.ftc_1_4.server.states.v1_2;

import java.io.IOException;
import java.net.SocketAddress;
import java.nio.ByteBuffer;
import java.nio.channels.SocketChannel;
import java.util.logging.Level;
import java.util.logging.Logger;

import tecgraf.ftc_1_4.common.IDataChannel;
import tecgraf.ftc_1_4.common.exception.DataChannelException;
import tecgraf.ftc_1_4.common.logic.ErrorCode;
import tecgraf.ftc_1_4.common.logic.PrimitiveTypeSize;
import tecgraf.ftc_1_4.server.FileServer;
import tecgraf.ftc_1_4.server.Session;
import tecgraf.ftc_1_4.server.states.State;

import static tecgraf.ftc_1_4.server.ErrorMessages.OPERATION_READ_CHUNK_SIZE;
import static tecgraf.ftc_1_4.server.ErrorMessages.OPERATION_READ_COUNT_REQUESTED;
import static tecgraf.ftc_1_4.server.ErrorMessages.OPERATION_READ_RESULT_BUFFERED;
import static tecgraf.ftc_1_4.server.ErrorMessages.OPERATION_READ_RESULT_SENT;
import static tecgraf.ftc_1_4.server.ErrorMessages.OPERATION_READ_START_POSITION;
import static tecgraf.ftc_1_4.server.ErrorMessages.OPERATION_READ_STATS;
import static tecgraf.ftc_1_4.server.ErrorMessages.OPERATION_READ_USING_TRANSFER_TO;

/**
 * Operao para leitura de dados a partir de uma determinada posio.
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class ReadState implements State {
  /**
   * Representa os estados internos desta operao.
   * 
   * @author Tecgraf/PUC-Rio
   */
  private enum InternalState {
    /**
     * O estado inicial.
     */
    INITIAL,
    /**
     * Estado que indica que a posio a partir da qual os dados sero lidos j
     * foi lida.
     */
    POSITION_READ,
    /**
     * Estado que indica que a quantidade de bytes que sero lidos j foi lida.
     */
    BYTE_COUNT_READ,
    /**
     * Estado que indica que o cabealho de resposta ja foi montado.
     */
    CHUNK_HEADER_WRITTEN,
    /**
     * Estado que indica que o cabealho de resposta ao comando ja foi enviado.
     */
    CHUNK_HEADER_SENT,
    /**
     * Estado que indica que todos os bytes solicitados j foram enviados.
     */
    BYTES_SENT;
  }

  /**
   * O estado atual da operao.
   */
  private InternalState currentState = InternalState.INITIAL;
  /**
   * O cdigo de erro.
   */
  private ErrorCode errorCode;
  /**
   * A posio a partir da qual os dados sero lidos do arquivo.
   */
  private long position;
  /**
   * Quantidade de bytes que sero lidos do arquivo.
   */
  private long count;

  /**
   * Indica se metodo transferTo deve ser utilizado.
   */
  private boolean readUsesTransferTo = false;

  /**
   * Objeto responsvel por registrar as atividades do servidor.
   */
  private final static Logger logger = Logger.getLogger("tecgraf.ftc");

  /**
   * Tamanho maximo a ser transmitido na chamada do metodo transferTo do
   * fileChannel.
   */
  public static final long MAX_BYTES = Integer.MAX_VALUE;

  /**
   * A quantidade de bytes enviados ao cliente.
   */
  private long bytesSent = 0;

  /**
   * Informa quantos bytes do chunk atual ja foram enviados.
   */
  private long chunkBytesSent = 0;

  /**
   * Informa o tamanho do chunk atual.
   */
  private long currentChunkSize = 0;

  /**
   * indica se deve trocar o estado atual da sessao.
   */
  private boolean changeState = false;

  /**
   * Indica se o estado esta interessado em eventos de escrita desse canal.
   */
  private boolean writing = false;

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean read(Session session) throws IOException {
    ByteBuffer buffer = session.getBuffer();
    SocketChannel channel = session.getChannel();
    IDataChannel dataChannel = session.getFileChannel();
    SocketAddress clientAddress = channel.socket().getRemoteSocketAddress();
    switch (this.currentState) {
      case INITIAL:
        buffer.limit(PrimitiveTypeSize.LONG.getSize());
        if (channel.read(buffer) < 0) {
          return false;
        } else {
          session.markLastActivity();
        }
        if (buffer.hasRemaining()) {
          return true;
        }

        buffer.flip();
        this.position = buffer.getLong();
        buffer.clear();

        this.currentState = InternalState.POSITION_READ;

        if (logger.isLoggable(Level.FINEST)) {
          logger.finest(String.format(OPERATION_READ_START_POSITION, clientAddress, this.position));
        }

      case POSITION_READ:
        buffer.limit(PrimitiveTypeSize.LONG.getSize());
        if (channel.read(buffer) < 0) {
          return false;
        } else {
          session.markLastActivity();
        }
        if (buffer.hasRemaining()) {
          return true;
        }

        buffer.flip();
        this.count = buffer.getLong();
        buffer.clear();

        this.readUsesTransferTo = ((dataChannel.supportedOperations() & IDataChannel.OP_TRANSFER_TO) != 0)
            && (session.getFileChannelInfo().useTransferTo())
            && ((this.count <= MAX_BYTES) || (!FileServer.PLATAFORM_HAS_TRANSFERTO_BUG));

        this.currentState = InternalState.BYTE_COUNT_READ;

        this.writing = true;

        if (logger.isLoggable(Level.FINEST)) {
          logger.finest(String.format(OPERATION_READ_COUNT_REQUESTED, clientAddress, this.count));
        }
        return true;
      default:
        return true;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean write(Session session) throws IOException, DataChannelException {
    SocketChannel channel = session.getChannel();
    ByteBuffer buffer = session.getBuffer();
    IDataChannel byteChannel = session.getFileChannel();
    SocketAddress clientAddress = channel.socket().getRemoteSocketAddress();
    switch (this.currentState) {
      case BYTE_COUNT_READ:
        // processing first or just another chunk, reset result message
        this.errorCode = ErrorCode.FAILURE;

        buffer.clear();
        short operations = byteChannel.supportedOperations();
        if (((operations & IDataChannel.OP_READ) == 0)
            || ((this.position != -1) && ((operations & IDataChannel.OP_SET_POSITION) == 0))) {

          buffer.limit(PrimitiveTypeSize.BYTE.getSize());
          buffer.put(this.errorCode.getCode());
          buffer.flip();
          this.changeState = true;

        } else {
          if (this.position >= 0) {
            byteChannel.setPosition(this.position + this.bytesSent);
          }

          this.changeState = createChunkHeader(session, buffer, byteChannel);
        }
        if (logger.isLoggable(Level.FINEST)) {
          logger.finest(String.format(OPERATION_READ_RESULT_BUFFERED, this.errorCode, clientAddress));
        }
        this.currentState = InternalState.CHUNK_HEADER_WRITTEN;

      case CHUNK_HEADER_WRITTEN:
        if (channel.write(buffer) > 0) {
          session.markLastActivity();
        }
        if (buffer.hasRemaining()) {
          return true;
        }
        buffer.clear();
        if (logger.isLoggable(Level.FINEST)) {
          logger.finest(String.format(OPERATION_READ_RESULT_SENT, this.errorCode, clientAddress));
        }
        if (this.changeState) {
          session.setCurrentState(new GetOperationState());
          return true;
        }
        buffer.limit(0);
        this.currentState = InternalState.CHUNK_HEADER_SENT;

      case CHUNK_HEADER_SENT:
        long bytesToRead = this.currentChunkSize - this.chunkBytesSent;
        long bytesWritten = transferBytesToSocket(byteChannel, channel, buffer, bytesToRead);
        if (bytesWritten > 0) {
          session.markLastActivity();
          this.bytesSent += bytesWritten;
        }

        // TODO: ???????
        if (bytesWritten == -1) {
          return true;
        }

        if (logger.isLoggable(Level.FINE)) {
          logger.fine(String.format(OPERATION_READ_STATS, clientAddress, bytesWritten, this.bytesSent, this.count));
        }

        // Se todos os bytes desse chunk foram enviados, comecar um novo chunk.
        this.chunkBytesSent += bytesWritten;
        if (this.chunkBytesSent == this.currentChunkSize) {
          buffer.clear();
          this.chunkBytesSent = 0;
          this.currentState = InternalState.BYTE_COUNT_READ;
        }

        // Se todos os bytes pedidos ja foram enviados, esperar nova operacao
        if (this.bytesSent == this.count) {
          this.writing = false;
          buffer.clear();
          this.currentState = InternalState.BYTES_SENT;
          session.setCurrentState(new GetOperationState());
        }
        return true;
      default:
        return true;
    }
  }

  /**
   * Metodo auxiliar para escrever no buffer o cabecalho do pacote (chunk) de dados
   * ou sinalizar o cdigo de erro quando for o caso.
   *
   * @param session a sesso do servidor para identificar o tamanho do pacote (chunk) atualmente configurado
   * @param buffer o buffer que ser usado posteriormente para enviar os dados no socket
   * @param dataChannel o stream de dados que ser usado para calcular quanto resta ser enviado
   *
   * @return {@code true} caso atingiu o final do arquivo e seja necessrio mudar de estado aps envio do erro, ou
   *         {@code false} caso o chunk foi processado e pode prosseguir com o envio do chunk.
   * 
   * @throws IOException
   * @throws DataChannelException
   */
  private boolean createChunkHeader(Session session, ByteBuffer buffer,
      IDataChannel dataChannel) throws IOException, DataChannelException {

    boolean returnValue = false;

    long bytesLeftOnFile = dataChannel.remaining();
    if (bytesLeftOnFile < 0) {
      this.errorCode = ErrorCode.END_OF_FILE;
      buffer.limit(PrimitiveTypeSize.BYTE.getSize());
      buffer.put(this.errorCode.getCode());
      buffer.flip();
      returnValue = true;
    } else {
      long missingBytes = this.count - this.bytesSent;
      this.currentChunkSize = Math.min(
          Math.min(missingBytes, session.getFileServer().getConfig()
              .getChunkSize()), bytesLeftOnFile);

      if (logger.isLoggable(Level.FINEST)) {
        logger.finest(String.format(OPERATION_READ_CHUNK_SIZE, this.currentChunkSize,
          session.getChannel().socket().getRemoteSocketAddress()));
      }

      this.errorCode = ErrorCode.OK;
      buffer.limit(PrimitiveTypeSize.BYTE.getSize()
          + PrimitiveTypeSize.LONG.getSize());
      buffer.put(this.errorCode.getCode());
      buffer.putLong(this.currentChunkSize);
      buffer.flip();
    }

    return returnValue;
  }

  /**
   * Metodo auxiliar para transferir bytes do canal de dados para o socket
   * 
   * @param byteChannel
   * @param channel
   * @param buffer
   * @param bytesToRead
   * 
   * @return
   * 
   * @throws IOException
   * @throws DataChannelException
   * @throws UnsupportedOperationException
   */
  private long transferBytesToSocket(IDataChannel byteChannel,
      SocketChannel channel, ByteBuffer buffer, long bytesToRead)
      throws IOException, UnsupportedOperationException, DataChannelException {
    long bytesWritten = -1;

    if (this.readUsesTransferTo) {
      logger.finest(String.format(OPERATION_READ_USING_TRANSFER_TO, channel.socket().getRemoteSocketAddress()));
      if (this.position == -1) {
        this.position = byteChannel.getPosition();
      }

      bytesWritten = byteChannel.transferTo(this.position + this.bytesSent,
          bytesToRead, channel);
    } else {

      if (buffer.hasRemaining()) {
        bytesWritten = channel.write(buffer);
      } else {
        if (this.position >= 0) {
          byteChannel.setPosition(this.position + this.bytesSent);
        }

        buffer.clear();
        if (buffer.remaining() > bytesToRead) {
          buffer.limit((int) bytesToRead);
        }

        int bytesRead = byteChannel.read(buffer);

        if (bytesRead == -1) {
          return -1;
        }

        buffer.flip();

        bytesWritten = channel.write(buffer);
      }

    }

    return bytesWritten;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isWriting() {
    return this.writing;
  }
}
