package tecgraf.ftc_1_4.server.states.v1_4;

import java.io.IOException;
import java.net.SocketAddress;
import java.nio.ByteBuffer;
import java.nio.channels.SocketChannel;
import java.util.logging.Level;
import java.util.logging.Logger;

import tecgraf.ftc_1_4.common.logic.ErrorCode;
import tecgraf.ftc_1_4.common.logic.ResultMessage;
import tecgraf.ftc_1_4.server.ChannelClosedReason;
import tecgraf.ftc_1_4.server.Session;
import tecgraf.ftc_1_4.server.states.State;
import tecgraf.ftc_1_4.server.states.StateUtil;

import static tecgraf.ftc_1_4.server.ErrorMessages.CLOSE_DATA_CHANNEL_RESULT_BUFFERED;
import static tecgraf.ftc_1_4.server.ErrorMessages.CLOSE_DATA_CHANNEL_RESULT_SENT_SUCCESSFULLY;

/**
 * Operao para fechamento de arquivo.
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class CloseState implements State {
  /**
   * Representa os estados internos desta operao.
   * 
   * @author Tecgraf/PUC-Rio
   */
  private enum InternalState {
    /**
     * O estado inicial.
     */
    INITIAL,
    /**
     * Estado que indica que a operao foi executada e a mensagem de resposta
     * colocada no buffer.
     */
    CHANNEL_CLOSED
  }

  /**
   * O estado atual da operao.
   */
  private InternalState currentState = InternalState.INITIAL;

  /**
   * Mensagem contendo o resultado a ser retornado.
   */
  private ResultMessage result = new ResultMessage();

  /**
   * Objeto responsvel por registrar as atividades do servidor.
   */
  private final static Logger logger = Logger.getLogger("tecgraf.ftc");

  /**
   * Indica se o estado esta interessado em eventos de escrita desse canal.
   */
  private boolean writing = true;

  /**
   * Cria a operao de fechamento de arquivo.
   */
  public CloseState() {
    this(ErrorCode.FAILURE);
  }

  /**
   * Cria a operao de fechamento de arquivo com um cdigo de erro inicial.
   * 
   * @param error um dos valores da enumerao {@link ErrorCode}
   */
  public CloseState(ErrorCode error) {
    this.result.success = ErrorCode.OK.equals(error) ? true : false;
    this.result.code = error;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean read(Session session) {
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean write(Session session) throws IOException {
    ByteBuffer buffer = session.getBuffer();
    SocketChannel channel = session.getChannel();
    SocketAddress clientAddress = channel.socket().getRemoteSocketAddress();
    switch (this.currentState) {
      case INITIAL:
        if (session.closeFileChannel(ChannelClosedReason.REQUESTED_BY_CLIENT)) {
          this.result.success = true;
        }

        StateUtil.writeResultMessage(buffer, this.result);
        buffer.flip();

        if (logger.isLoggable(Level.FINEST)) {
          logger.finest(String.format(CLOSE_DATA_CHANNEL_RESULT_BUFFERED, this.result, clientAddress));
        }
        this.currentState = InternalState.CHANNEL_CLOSED;

      case CHANNEL_CLOSED:
        if (channel.write(buffer) > 0) {
          session.markLastActivity();
        }
        if (buffer.hasRemaining()) {
          return true;
        }
        this.writing = false;
        buffer.clear();
        if (logger.isLoggable(Level.FINEST)) {
          logger.finest(String.format(CLOSE_DATA_CHANNEL_RESULT_SENT_SUCCESSFULLY, this.result, clientAddress));
        }
        session.setCurrentState(null);
        channel.close();

        return true;
      default:
        return false;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isWriting() {
    return this.writing;
  }
}
