package tecgraf.ftc_1_4.server.states.v1_4;

import java.io.IOException;
import java.net.SocketAddress;
import java.nio.ByteBuffer;
import java.nio.channels.SocketChannel;
import java.util.logging.Level;
import java.util.logging.Logger;

import tecgraf.ftc_1_4.common.logic.Operation;
import tecgraf.ftc_1_4.common.logic.PrimitiveTypeSize;
import tecgraf.ftc_1_4.server.Session;
import tecgraf.ftc_1_4.server.states.State;

import static tecgraf.ftc_1_4.server.ErrorMessages.INVALID_PROTOCOL_OPERATION;
import static tecgraf.ftc_1_4.server.ErrorMessages.PROTOCOL_OPERATION_REQUESTED;

/**
 * Responsvel por obter a operao solicitada pelo cliente.
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class GetOperationState implements State {

  /**
   * Objeto responsvel por registrar as atividades do servidor.
   */
  private final static Logger logger = Logger.getLogger("tecgraf.ftc");

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean read(Session session) throws IOException {
    ByteBuffer buffer = session.getBuffer();
    SocketChannel channel = session.getChannel();
    SocketAddress clientAddress = channel.socket().getRemoteSocketAddress();
    byte operationCode;
    buffer.limit(PrimitiveTypeSize.BYTE.getSize());
    if (channel.read(buffer) < 0) {
      return false;
    } else {
      session.markLastActivity();
    }
    if (buffer.hasRemaining()) {
      return true;
    }
    buffer.flip();
    operationCode = buffer.get();
    buffer.clear();

    Operation operation = Operation.valueOf(operationCode);
    if (operation == null) {
      if (logger.isLoggable(Level.SEVERE)) {
        logger.severe(String.format(INVALID_PROTOCOL_OPERATION, operationCode, clientAddress));
      }
      // Se a operao recebida  desconhecida, significa que o protocolo saiu
      // de sincronia e devemos fechar imediatamente.
      return false;
    }

    if (logger.isLoggable(Level.FINEST)) {
      logger.finest(String.format(PROTOCOL_OPERATION_REQUESTED, operation, clientAddress));
    }

    session.setCurrentState(this.getState(operation));
    return true;
  }

  /**
   * Obtm o estado relacionado a uma operao.
   * 
   * @param operation A operao.
   * 
   * @return O estado.
   */
  private State getState(Operation operation) {
    switch (operation) {
      case OPEN_READ_ONLY:
        return new OpenReadOnlyState();
      case OPEN_READ_WRITE:
        return new OpenReadWriteState();
      case CLOSE:
        return new CloseState();
      case SET_SIZE:
        return new SetSizeState();
      case GET_POSITION:
        return new GetPositionState();
      case SET_POSITION:
        return new SetPositionState();
      case GET_SIZE:
        return new GetSizeState();
      case READ:
        return new ReadState();
      case WRITE:
        return new WriteState();
      case KEEP_ALIVE:
        return new KeepAliveState();
      default:
        return null;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean write(Session session) {
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isWriting() {
    return false;
  }
}
