package tecgraf.ftc_1_4.server.states.v1_4;

import java.io.IOException;
import java.net.SocketAddress;
import java.nio.ByteBuffer;
import java.nio.channels.SocketChannel;
import java.util.logging.Level;
import java.util.logging.Logger;

import tecgraf.ftc_1_4.common.IDataChannel;
import tecgraf.ftc_1_4.common.logic.ErrorCode;
import tecgraf.ftc_1_4.common.logic.PrimitiveTypeSize;
import tecgraf.ftc_1_4.common.logic.ResultMessage;
import tecgraf.ftc_1_4.server.Session;
import tecgraf.ftc_1_4.server.states.State;
import tecgraf.ftc_1_4.server.states.StateUtil;

import static tecgraf.ftc_1_4.server.ErrorMessages.DATA_CHANNEL_DOESNT_SUPPORT_GET_SIZE;
import static tecgraf.ftc_1_4.server.ErrorMessages.FAILED_TO_READ_DATA_CHANNEL_SIZE;
import static tecgraf.ftc_1_4.server.ErrorMessages.OPERATION_GET_SIZE_RESULT_BUFFERED;
import static tecgraf.ftc_1_4.server.ErrorMessages.OPERATION_GET_SIZE_RESULT_SENT;
import static tecgraf.ftc_1_4.server.ErrorMessages.hexString;

/**
 * Operao de obteno do tamanho do arquivo.
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class GetSizeState implements State {
  /**
   * Representa os estados internos desta operao.
   * 
   * @author Tecgraf/PUC-Rio
   */
  private enum InternalState {
    /**
     * O estado inicial.
     */
    INITIAL,
    /**
     * Estado que indica que o tamanho do arquivo j foi lido.
     */
    SIZE_READ,
    /**
     * Estado que a mensagem de retorno ja foi escrita no buffer.
     */
    RESULT_MESSAGE_WRITTEN;
  }

  /**
   * O estado atual da operao.
   */
  private InternalState currentState = InternalState.INITIAL;

  /**
   * O tamanho do arquivo.
   */
  private long size = -1;

  /**
   * Mensagem contendo o resultado a ser retornado.
   */
  private ResultMessage result = new ResultMessage();

  /**
   * Objeto responsvel por registrar as atividades do servidor.
   */
  private final static Logger logger = Logger.getLogger("tecgraf.ftc");

  /**
   * Indica se o estado esta interessado em eventos de escrita desse canal.
   */
  private boolean writing = true;

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean read(Session session) {
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean write(Session session) throws IOException {
    ByteBuffer buffer = session.getBuffer();
    SocketChannel channel = session.getChannel();
    SocketAddress clientAddress = channel.socket().getRemoteSocketAddress();
    switch (this.currentState) {
      case INITIAL:
        IDataChannel byteChannel = session.getFileChannel();
        if ((byteChannel.supportedOperations() & IDataChannel.OP_GET_SIZE) != 0) {
          try {
            this.size = byteChannel.getSize();
            this.result.success = true;
          } catch (Exception e) {
            byte[] fileId = session.getFileChannelInfo().getFileId();
            if (logger.isLoggable(Level.SEVERE)) {
              logger.log(Level.SEVERE, String.format(FAILED_TO_READ_DATA_CHANNEL_SIZE,
                clientAddress, hexString(fileId)), e);
            }
            session.getFileServer().exceptionRaised(e, fileId);
            this.result.code = ErrorCode.FAILURE;
            this.result.message = e.getMessage();
          }
        } else {
          if (logger.isLoggable(Level.WARNING)) {
            logger.warning(String.format(DATA_CHANNEL_DOESNT_SUPPORT_GET_SIZE, clientAddress,
              hexString(session.getFileChannelInfo().getFileId())));
          }
          this.result.code = ErrorCode.UNSUPPORTED_OPERATION;
        }
        this.currentState = InternalState.SIZE_READ;

      case SIZE_READ:
        StateUtil.writeResultMessage(buffer, this.result);
        if (this.result.success) {
          // append effective size in the result message
          buffer.limit(buffer.limit() + PrimitiveTypeSize.LONG.getSize());
          buffer.putLong(this.size);
        }
        buffer.flip();
        if (logger.isLoggable(Level.FINEST)) {
          logger.finest(String.format(OPERATION_GET_SIZE_RESULT_BUFFERED, this.result, clientAddress));
        }
        this.currentState = InternalState.RESULT_MESSAGE_WRITTEN;

      case RESULT_MESSAGE_WRITTEN:
        if (channel.write(buffer) > 0) {
          session.markLastActivity();
        }
        if (buffer.hasRemaining()) {
          return true;
        }
        this.writing = false;
        buffer.clear();
        if (logger.isLoggable(Level.FINEST)) {
          logger.finest(String.format(OPERATION_GET_SIZE_RESULT_SENT,
            this.result + " value: " + this.size, clientAddress));
        }
        session.setCurrentState(new GetOperationState());
        return true;
      default:
        return true;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isWriting() {
    return this.writing;
  }
}
