package tecgraf.ftc_1_4.server.states.v1_4;

import java.io.IOException;
import java.net.SocketAddress;
import java.nio.ByteBuffer;
import java.nio.channels.SocketChannel;
import java.util.Formatter;
import java.util.logging.Level;
import java.util.logging.Logger;

import tecgraf.ftc_1_4.common.IDataChannel;
import tecgraf.ftc_1_4.common.exception.FailureException;
import tecgraf.ftc_1_4.common.exception.PermissionException;
import tecgraf.ftc_1_4.common.logic.ErrorCode;
import tecgraf.ftc_1_4.common.logic.PrimitiveTypeSize;
import tecgraf.ftc_1_4.common.logic.ResultMessage;
import tecgraf.ftc_1_4.server.FileChannelRequestInfo;
import tecgraf.ftc_1_4.server.FileServer;
import tecgraf.ftc_1_4.server.Session;
import tecgraf.ftc_1_4.server.states.State;
import tecgraf.ftc_1_4.server.states.StateUtil;

import static tecgraf.ftc_1_4.server.ErrorMessages.DATA_CHANNEL_SUPPORTED_OPERATIONS;
import static tecgraf.ftc_1_4.server.ErrorMessages.NO_DATA_CHANNEL_PROVIDED;
import static tecgraf.ftc_1_4.server.ErrorMessages.OPERATION_OPEN_RESULT_BUFFERED;
import static tecgraf.ftc_1_4.server.ErrorMessages.OPERATION_OPEN_RESULT_SENT_SUCCESSFULLY;

/**
 * Operao de abertura de um arquivo.
 * 
 * @author Tecgraf/PUC-Rio
 */
public abstract class OpenState implements State {

  /**
   * Representa os estados internos desta operao.
   * 
   * @author Tecgraf/PUC-Rio
   */
  protected enum InternalState {
    /**
     * O estado inicial.
     */
    INITIAL,
    /**
     * Estado que indica que a tentativa de abertura do arquivo j foi feita.
     */
    FILE_OPEN_RESULT,
    /**
     * Estado que a mensagem de retorno ja foi escrita no buffer.
     */
    RESULT_MESSAGE_WRITTEN;
  }

  /**
   * O estado atual da operao.
   */
  private InternalState currentState = InternalState.INITIAL;
  /**
   * O cdigo de erro.
   */
  private ResultMessage result = new ResultMessage();

  /**
   * Indica se o arquivo ser aberto somente para leitura ou se ser aberto para
   * leitura e gravao.
   */
  private boolean readOnly;

  /**
   * Objeto responsvel por registrar as atividades do servidor.
   */
  private final static Logger logger = Logger.getLogger("tecgraf.ftc");

  /**
   * Indica se o estado esta interessado em eventos de escrita desse canal.
   */
  private boolean writing = true;

  /**
   * Cria uma operao de abertura de arquivo.
   * 
   * @param readOnly Indica se o arquivo ser aberto somente para leitura ou se
   *          ser aberto para leitura e gravao.
   */
  protected OpenState(boolean readOnly) {
    this.readOnly = readOnly;
    if (logger.isLoggable(Level.FINEST)) {
      logger.finest(OpenState.class.getCanonicalName() + " readOnly=" + readOnly);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean read(Session session) {
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean write(Session session) throws IOException {
    ByteBuffer buffer = session.getBuffer();
    SocketChannel channel = session.getChannel();
    SocketAddress clientAddress = channel.socket().getRemoteSocketAddress();
    switch (this.currentState) {
      case INITIAL:
        FileChannelRequestInfo fileChannelInfo = session.getFileChannelInfo();
        FileServer fileServer = session.getFileServer();
        try {
          IDataChannel fileChannel = fileServer.getDataProvider()
            .createFileChannel(fileChannelInfo.getRequester(),
              fileChannelInfo.getFileId(), this.readOnly);
          if (fileChannel == null) {
            this.result.code = ErrorCode.FILE_NOT_FOUND;
            this.result.message = NO_DATA_CHANNEL_PROVIDED;
          } else {
            this.result.success = true;
            session.setFileChannel(fileChannel);
          }
        } catch (FailureException e) {
          this.result.code = ErrorCode.FAILURE;
          this.result.message = e.getMessage();
        } catch (PermissionException e) {
          this.result.code = ErrorCode.NO_PERMISSION;
          this.result.message = e.getMessage();
        }
        this.currentState = InternalState.FILE_OPEN_RESULT;

      case FILE_OPEN_RESULT:
        StateUtil.writeResultMessage(buffer, this.result);
        if (this.result.success) {
          // append supported operations mark
          buffer.limit(buffer.limit() + PrimitiveTypeSize.SHORT.getSize());
          short ops = session.getFileChannel().supportedOperations();
          buffer.putShort(ops);
          if (logger.isLoggable(Level.FINEST)) {
            logger.finest(String.format(DATA_CHANNEL_SUPPORTED_OPERATIONS, clientAddress, new Formatter().format("0x%X", ops)));
          }
        }
        buffer.flip();
        if (logger.isLoggable(Level.FINEST)) {
          logger.finest(String.format(OPERATION_OPEN_RESULT_BUFFERED, this.result, clientAddress));
        }
        this.currentState = InternalState.RESULT_MESSAGE_WRITTEN;

      case RESULT_MESSAGE_WRITTEN:
        if (channel.write(buffer) > 0) {
          session.markLastActivity();
        }
        if (buffer.hasRemaining()) {
          return true;
        }
        this.writing = false;
        buffer.clear();
        if (logger.isLoggable(Level.FINEST)) {
          logger.finest(String.format(OPERATION_OPEN_RESULT_SENT_SUCCESSFULLY, this.result, clientAddress));
        }
        session.setCurrentState(new GetOperationState());
        if (!this.result.success) {
          return false;
        }
        return true;
      default:
        return true;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isWriting() {
    return this.writing;
  }
}
