package tecgraf.ftc_1_4.server.states.v1_4;

import java.io.IOException;
import java.net.SocketAddress;
import java.nio.ByteBuffer;
import java.nio.channels.SocketChannel;
import java.util.logging.Level;
import java.util.logging.Logger;

import tecgraf.ftc_1_4.common.IDataChannel;
import tecgraf.ftc_1_4.common.logic.ErrorCode;
import tecgraf.ftc_1_4.common.logic.PrimitiveTypeSize;
import tecgraf.ftc_1_4.common.logic.ResultMessage;
import tecgraf.ftc_1_4.server.Session;
import tecgraf.ftc_1_4.server.states.State;
import tecgraf.ftc_1_4.server.states.StateUtil;

import static tecgraf.ftc_1_4.server.ErrorMessages.DATA_CHANNEL_DOESNT_SUPPORT_SET_SIZE;
import static tecgraf.ftc_1_4.server.ErrorMessages.FAILED_TO_SET_DATA_CHANNEL_SIZE;
import static tecgraf.ftc_1_4.server.ErrorMessages.OPERATION_SET_SIZE_RECEIVED;
import static tecgraf.ftc_1_4.server.ErrorMessages.OPERATION_SET_SIZE_RESULT_BUFFERED;
import static tecgraf.ftc_1_4.server.ErrorMessages.OPERATION_SET_SIZE_RESULT_SENT;
import static tecgraf.ftc_1_4.server.ErrorMessages.hexString;

/**
 * Operao para truncar o arquivo para um novo tamanho.
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class SetSizeState implements State {
  /**
   * Representa os estados internos desta operao.
   * 
   * @author Tecgraf/PUC-Rio
   */
  private enum InternalState {
    /**
     * O estado inicial.
     */
    INITIAL,
    /**
     * Estado que indica que o novo tamanho do arquivo j foi lido.
     */
    SIZE_READ,
    /**
     * Estado que a mensagem de retorno ja foi escrita no buffer.
     */
    RESULT_MESSAGE_WRITTEN;
  }

  /**
   * O estado atual da operao.
   */
  private InternalState currentState = InternalState.INITIAL;

  /**
   * Mensagem contendo o resultado a ser retornado.
   */
  private ResultMessage result = new ResultMessage();

  /**
   * Objeto responsvel por registrar as atividades do servidor.
   */
  private final static Logger logger = Logger.getLogger("tecgraf.ftc");

  /**
   * Indica se o estado esta interessado em eventos de escrita desse canal.
   */
  private boolean writing = false;

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean read(Session session) throws IOException {
    ByteBuffer buffer = session.getBuffer();
    SocketChannel channel = session.getChannel();
    SocketAddress clientAddress = channel.socket().getRemoteSocketAddress();
    switch (this.currentState) {
      case INITIAL:
        buffer.limit(PrimitiveTypeSize.LONG.getSize());
        if (channel.read(buffer) < 0) {
          return false;
        } else {
          session.markLastActivity();
        }
        if (buffer.hasRemaining()) {
          return true;
        }
        buffer.flip();
        long size = buffer.getLong();
        buffer.clear();
        this.writing = true;
        IDataChannel byteChannel = session.getFileChannel();
        if ((byteChannel.supportedOperations() & IDataChannel.OP_SET_SIZE) == 0) {
          if (logger.isLoggable(Level.WARNING)) {
            logger.warning(String.format(DATA_CHANNEL_DOESNT_SUPPORT_SET_SIZE, clientAddress,
              hexString(session.getFileChannelInfo().getFileId())));
          }
          this.result.code = ErrorCode.UNSUPPORTED_OPERATION;
        } else {
          try {
            if (logger.isLoggable(Level.FINEST)) {
              logger.finest(String.format(OPERATION_SET_SIZE_RECEIVED, size, clientAddress));
            }
            byteChannel.setSize(size);
            this.result.success = true;
          } catch (Exception e) {
            byte[] fileId = session.getFileChannelInfo().getFileId();
            if (logger.isLoggable(Level.SEVERE)) {
              logger.log(Level.SEVERE, String.format(FAILED_TO_SET_DATA_CHANNEL_SIZE,
                clientAddress, hexString(fileId)), e);
            }
            session.getFileServer().exceptionRaised(e, fileId);
            this.result.code = ErrorCode.FAILURE;
            this.result.message = e.getMessage();
          }
        }
        this.currentState = InternalState.SIZE_READ;
        return true;
      default:
        return true;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean write(Session session) throws IOException {
    ByteBuffer buffer = session.getBuffer();
    SocketChannel channel = session.getChannel();
    SocketAddress clientAddress = channel.socket().getRemoteSocketAddress();
    switch (this.currentState) {
      case SIZE_READ:
        StateUtil.writeResultMessage(buffer, this.result);
        buffer.flip();
        if (logger.isLoggable(Level.FINEST)) {
          logger.finest(String.format(OPERATION_SET_SIZE_RESULT_BUFFERED, this.result, clientAddress));
        }
        this.currentState = InternalState.RESULT_MESSAGE_WRITTEN;

      case RESULT_MESSAGE_WRITTEN:
        if (channel.write(buffer) > 0) {
          session.markLastActivity();
        }
        if (buffer.hasRemaining()) {
          return true;
        }
        this.writing = false;
        buffer.clear();
        if (logger.isLoggable(Level.FINEST)) {
          logger.finest(String.format(OPERATION_SET_SIZE_RESULT_SENT, this.result, clientAddress));
        }
        session.setCurrentState(new GetOperationState());
        return true;
      default:
        return true;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isWriting() {
    return this.writing;
  }
}
