package tecgraf.ftc_1_4.server.states.v1_4;

import java.io.IOException;
import java.net.SocketAddress;
import java.nio.ByteBuffer;
import java.nio.channels.SocketChannel;
import java.util.logging.Level;
import java.util.logging.Logger;

import tecgraf.ftc_1_4.common.logic.ErrorCode;
import tecgraf.ftc_1_4.common.logic.PrimitiveTypeSize;
import tecgraf.ftc_1_4.common.logic.ProtocolVersion;
import tecgraf.ftc_1_4.common.logic.ResultMessage;
import tecgraf.ftc_1_4.server.Session;
import tecgraf.ftc_1_4.server.states.State;
import tecgraf.ftc_1_4.server.states.StateUtil;
import tecgraf.ftc_1_4.server.states.SupportedProtocols;

import static tecgraf.ftc_1_4.server.ErrorMessages.CONNECTION_IS_USING_PROTOCOL_VERSION;
import static tecgraf.ftc_1_4.server.ErrorMessages.INVALID_PROTOCOL_MESSAGE_RECEIVED;
import static tecgraf.ftc_1_4.server.ErrorMessages.UNSUPPORTED_PROTOCOL_VERSION;
import static tecgraf.ftc_1_4.server.ErrorMessages.VERSION_HANDSHAKE_RESULT_BUFFERED;
import static tecgraf.ftc_1_4.server.ErrorMessages.VERSION_HANDSHAKE_RESULT_SENT;

/**
 * Classe que implementa o estado de handshake inicial. Nesse handshake 
 * determinado qual verso do protocolo sera utilizada.
 * 
 * @author Tecgraf
 */
public class VersionHandshakeState implements State {

  /**
   * Representa os estados internos desta operao.
   * 
   * @author Tecgraf/PUC-Rio
   */
  private enum InternalState {
    /**
     * O estado inicial.
     */
    INITIAL,
    /**
     * Estado que criado para suportar clientes antigos que fala protocolo 1.1
     */
    FTC_1_1_PROTOCOL_CHECKED,
    /**
     * Estado que indica que o identificador de protocolo foi recebido.
     */
    PROTOCOL_ID_RECEIVED,
    /**
     * Estado que indica que o numero da verso ja foi recebido.
     */
    VERSION_RECEIVED,
    /**
     * Estado que a mensagem de retorno ja foi escrita no buffer.
     */
    RESULT_MESSAGE_WRITTEN,
    /**
     * Estado que indica que o cdigo de erro j foi enviado.
     */
    RESULT_MESSAGE_SENT;
  }

  /**
   * O estado atual da operao.
   */
  private InternalState currentState = InternalState.INITIAL;

  /**
   * Mensagem contendo o resultado a ser retornado.
   */
  private ResultMessage result = new ResultMessage();

  /**
   * Objeto responsvel por registrar as atividades do servidor.
   */
  private final static Logger logger = Logger.getLogger("tecgraf.ftc");

  /**
   * Indica se o estado esta interessado em eventos de escrita desse canal.
   */
  private boolean writing = false;

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean read(Session session) throws IOException {
    ByteBuffer buffer = session.getBuffer();
    SocketChannel channel = session.getChannel();
    SocketAddress clientAddress = channel.socket().getRemoteSocketAddress();
    switch (this.currentState) {
      case INITIAL:
        buffer.limit(PrimitiveTypeSize.BYTE.getSize());
        if (channel.read(buffer) < 0) {
          return false;
        } else {
          session.markLastActivity();
        }
        if (buffer.hasRemaining()) {
          return true;
        }
        // Workaround para suportar clientes com verso 1.1
        if (buffer.get(0) != 0) {
          session.setProtocol(SupportedProtocols.V1_1);

          if (session.getMaxClientsReached()) {
            // Por conta do workaround, o buffer ainda contm o tamanho da chave
            //  preciso limpar para garantir o funcionamento do CloseState 1.1
            buffer.clear();
            session
                .setCurrentState(new tecgraf.ftc_1_4.server.states.v1_1.CloseState(
                    ErrorCode.MAX_CLIENTS_REACHED));
          } else {
            session
                .setCurrentState(new tecgraf.ftc_1_4.server.states.v1_1.ValidateKeyState());
          }
          return true;
        }
        this.currentState = InternalState.FTC_1_1_PROTOCOL_CHECKED;

      case FTC_1_1_PROTOCOL_CHECKED:
        buffer.limit(PrimitiveTypeSize.INTEGER.getSize());
        if (channel.read(buffer) < 0) {
          return false;
        } else {
          session.markLastActivity();
        }
        if (buffer.hasRemaining()) {
          return true;
        }

        buffer.flip();
        int protocolId = buffer.getInt();
        buffer.clear();

        if (protocolId != ProtocolVersion.PROTOCOL_IDENTIFICATION) {
          if (logger.isLoggable(Level.WARNING)) {
            logger.warning(String.format(INVALID_PROTOCOL_MESSAGE_RECEIVED, clientAddress));
          }
          this.currentState = InternalState.VERSION_RECEIVED;
          this.result.code = ErrorCode.INVALID_PROTOCOL_MESSAGE;
          return true;
        }

        this.currentState = InternalState.PROTOCOL_ID_RECEIVED;

      case PROTOCOL_ID_RECEIVED:
        buffer.limit(PrimitiveTypeSize.INTEGER.getSize());
        if (channel.read(buffer) < 0) {
          return false;
        } else {
          session.markLastActivity();
        }
        if (buffer.hasRemaining()) {
          return true;
        }

        buffer.flip();
        buffer.get();// padding
        byte major_version = buffer.get();
        byte minor_version = buffer.get();
        byte patch_version = buffer.get();
        buffer.clear();

        this.writing = true;

        this.currentState = InternalState.VERSION_RECEIVED;

        if ((major_version == ProtocolVersion.MAJOR_VERSION)
            && ((minor_version == ProtocolVersion.MINOR_VERSION) || (minor_version == ProtocolVersion.V2_MINOR_VERSION))) {
          this.result.success = true;

          if (minor_version == ProtocolVersion.V2_MINOR_VERSION) {
            session.setProtocol(SupportedProtocols.V1_2);
          }
          if (logger.isLoggable(Level.INFO)) {
            logger.info(String.format(CONNECTION_IS_USING_PROTOCOL_VERSION, clientAddress, major_version
              + "." + minor_version + "." + patch_version));
          }
        } else {
          if (logger.isLoggable(Level.SEVERE)) {
            logger.severe(String.format(UNSUPPORTED_PROTOCOL_VERSION, clientAddress, major_version
              + "." + minor_version + "." + patch_version));
          }
          this.result.code = ErrorCode.INVALID_VERSION;
        }

        if (session.getMaxClientsReached()) {
          State state;

          switch (session.getProtocol()) {
            case V1_2:
              state = new tecgraf.ftc_1_4.server.states.v1_2.CloseState(
                  ErrorCode.MAX_CLIENTS_REACHED);
              break;
            case V1_4:
              state = new tecgraf.ftc_1_4.server.states.v1_4.CloseState(
                  ErrorCode.MAX_CLIENTS_REACHED);
              break;
            default:
              state = new tecgraf.ftc_1_4.server.states.v1_4.CloseState(
                  ErrorCode.FAILURE);
              break;
          }
          session.setCurrentState(state);
          return true;
        }

        return true;
      default:
        return true;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean write(Session session) throws IOException {
    ByteBuffer buffer = session.getBuffer();
    SocketChannel channel = session.getChannel();
    SocketAddress clientAddress = channel.socket().getRemoteSocketAddress();
    switch (this.currentState) {
      case VERSION_RECEIVED:

        StateUtil.writeResultMessage(buffer, this.result);
        buffer.flip();

        if (logger.isLoggable(Level.FINEST)) {
          logger.finest(String.format(VERSION_HANDSHAKE_RESULT_BUFFERED, this.result, clientAddress));
        }

        this.currentState = InternalState.RESULT_MESSAGE_WRITTEN;

      case RESULT_MESSAGE_WRITTEN:
        if (channel.write(buffer) > 0) {
          session.markLastActivity();
        }
        if (buffer.hasRemaining()) {
          return true;
        }
        this.writing = false;

        buffer.clear();

        if (logger.isLoggable(Level.FINEST)) {
          logger.finest(String.format(VERSION_HANDSHAKE_RESULT_SENT, this.result, clientAddress));
        }

        this.currentState = InternalState.RESULT_MESSAGE_SENT;

        if (!this.result.success) {
          return false;
        }

        State state;
        switch (session.getProtocol()) {
          case V1_2:
            state = new tecgraf.ftc_1_4.server.states.v1_2.ValidateKeyState();
            break;
          case V1_4:
            state = new tecgraf.ftc_1_4.server.states.v1_4.ValidateKeyState();
            break;
          default:
            state = new tecgraf.ftc_1_4.server.states.v1_4.CloseState();
            break;
        }
        session.setCurrentState(state);
        return true;
      default:
        return true;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isWriting() {
    return this.writing;
  }
}
