package tecgraf.ftc_1_4.server.states.v1_4;

import java.io.IOException;
import java.net.SocketAddress;
import java.nio.ByteBuffer;
import java.nio.channels.SocketChannel;
import java.util.logging.Level;
import java.util.logging.Logger;

import tecgraf.ftc_1_4.common.IDataChannel;
import tecgraf.ftc_1_4.common.exception.DataChannelException;
import tecgraf.ftc_1_4.common.logic.ErrorCode;
import tecgraf.ftc_1_4.common.logic.PrimitiveTypeSize;
import tecgraf.ftc_1_4.common.logic.ResultMessage;
import tecgraf.ftc_1_4.server.FileChannelRequestInfo;
import tecgraf.ftc_1_4.server.Session;
import tecgraf.ftc_1_4.server.states.State;
import tecgraf.ftc_1_4.server.states.StateUtil;

import static tecgraf.ftc_1_4.server.ErrorMessages.OPERATION_WRITE_RESULT_BUFFERED;
import static tecgraf.ftc_1_4.server.ErrorMessages.OPERATION_WRITE_RESULT_SENT;
import static tecgraf.ftc_1_4.server.ErrorMessages.OPERATION_WRITE_STATS;
import static tecgraf.ftc_1_4.server.ErrorMessages.OPERATION_WRITE_USING_TRANSFER_FROM;

/**
 * Operao para escrita de dados a partir de uma determinada posio.
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class WriteState implements State {

  /**
   * Representa os estados internos desta operao.
   * 
   * @author Tecgraf/PUC-Rio
   */
  private enum InternalState {
    /**
     * O estado inicial.
     */
    INITIAL,
    /**
     * Estado que indica que a posio a partir da qual os dados sero escritos
     * j foi lida.
     */
    POSITION_READ,
    /**
     * Estado que indica que a quantidade de bytes que sero escritos j foi
     * lida.
     */
    BYTE_COUNT_READ,
    /**
     * Estado que indica que todos os bytes solicitados j foram recebidos.
     */
    BYTES_RECEIVED,
    /**
     * Estado que a mensagem de retorno ja foi escrita no buffer.
     */
    RESULT_MESSAGE_WRITTEN,
    /**
     * Estado que indica que o cdigo de erro j foi enviado.
     */
    RESULT_MESSAGE_SENT;
  }

  /**
   * O estado atual da operao.
   */
  private InternalState currentState = InternalState.INITIAL;

  /**
   * A posio a partir da qual os dados sero escritos no arquivo.
   */
  private long position;
  /**
   * Quantidade de bytes que sero escritos no arquivo.
   */
  private long count;
  /**
   * A quantidade de bytes recebidos do cliente.
   */
  private long bytesReceived;
  /**
   * Indica se o metodo transferFrom deve ser utilizado.
   */
  private boolean writeUseTransferFrom = false;

  /**
   * Mensagem contendo o resultado a ser retornado.
   */
  private ResultMessage result = new ResultMessage();

  /**
   * Objeto responsvel por registrar as atividades do servidor.
   */
  private final static Logger logger = Logger.getLogger("tecgraf.ftc");

  /**
   * Indica se o estado esta interessado em eventos de escrita desse canal.
   */
  private boolean writing = false;

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean read(Session session) throws IOException, DataChannelException {
    ByteBuffer buffer = session.getBuffer();
    SocketChannel channel = session.getChannel();
    IDataChannel byteChannel = session.getFileChannel();

    switch (this.currentState) {
      case INITIAL:
        buffer.limit(PrimitiveTypeSize.LONG.getSize());
        if (channel.read(buffer) < 0) {
          return false;
        } else {
          session.markLastActivity();
        }
        if (buffer.hasRemaining()) {
          return true;
        }
        buffer.flip();
        this.position = buffer.getLong();
        buffer.clear();

        this.currentState = InternalState.POSITION_READ;

      case POSITION_READ:
        buffer.limit(PrimitiveTypeSize.LONG.getSize());
        if (channel.read(buffer) < 0) {
          return false;
        } else {
          session.markLastActivity();
        }
        if (buffer.hasRemaining()) {
          return true;
        }

        buffer.flip();
        this.count = buffer.getLong();
        buffer.clear();

        this.writing = true;

        this.currentState = InternalState.BYTE_COUNT_READ;
        return true;
      case RESULT_MESSAGE_SENT:
        long remainingBytes = (this.count - this.bytesReceived);
        long bytesRead = transferBytesFromSocket(byteChannel, channel, buffer, remainingBytes);
        if (bytesRead > 0) {
          session.markLastActivity();
        }
        // Na FTC-95 houve melhoria que passou a sinalizar END_OF_FILE para verses 1.2 e 1.4 do protocolo
        // Os estados internos BYTE_COUNT_READ e RESULT_MESSAGE_WRITTEN j realizaram a sinalizao do erro
        if (bytesRead == -1) {
          return true;
        }

        this.bytesReceived += bytesRead;

        if (logger.isLoggable(Level.FINE)) {
          SocketAddress clientAddress = channel.socket().getRemoteSocketAddress();
          logger.fine(String.format(OPERATION_WRITE_STATS, clientAddress, bytesRead, this.bytesReceived, this.count));
        }

        if (this.bytesReceived == this.count) {
          buffer.clear();
          this.currentState = InternalState.BYTES_RECEIVED;
          session.setCurrentState(new GetOperationState());
        }
        return true;
      default:
        return true;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean write(Session session) throws IOException {
    ByteBuffer buffer = session.getBuffer();
    SocketChannel channel = session.getChannel();
    IDataChannel fileChannel = session.getFileChannel();
    SocketAddress clientAddress = channel.socket().getRemoteSocketAddress();
    switch (this.currentState) {
      case BYTE_COUNT_READ:
        FileChannelRequestInfo fileChannelInfo = session.getFileChannelInfo();
        short operations = fileChannel.supportedOperations();

        if (((operations & IDataChannel.OP_WRITE) == 0)
            || ((this.position != -1) && ((operations & IDataChannel.OP_SET_POSITION) == 0))) {
          this.result.code = ErrorCode.UNSUPPORTED_OPERATION;
        } else if (session
            .getFileServer()
            .getDataProvider()
            .isLocked(fileChannelInfo.getRequester(),
                fileChannelInfo.getFileId())) {
          this.result.code = ErrorCode.FILE_LOCKED;
        } else {
          this.result.success = true;
        }

        StateUtil.writeResultMessage(buffer, this.result);

        if (logger.isLoggable(Level.FINEST)) {
          logger.finest(String.format(OPERATION_WRITE_RESULT_BUFFERED, this.result, clientAddress));
        }

        buffer.flip();
        this.currentState = InternalState.RESULT_MESSAGE_WRITTEN;

      case RESULT_MESSAGE_WRITTEN:
        if (channel.write(buffer) > 0) {
          session.markLastActivity();
        }
        if (buffer.hasRemaining()) {
          return true;
        }
        this.writing = false;

        buffer.clear();

        if (logger.isLoggable(Level.FINEST)) {
          logger.finest(String.format(OPERATION_WRITE_RESULT_SENT, this.result, clientAddress));
        }

        if (this.result.success) {
          this.currentState = InternalState.RESULT_MESSAGE_SENT;

          this.writeUseTransferFrom = ((fileChannel.supportedOperations() & IDataChannel.OP_TRANSFER_FROM) != 0)
              && (session.getFileChannelInfo().useTransferTo());

          buffer.limit(0);
        } else {
          session.setCurrentState(new GetOperationState());
        }
        return true;
      default:
        return true;
    }
  }

  /**
   * @param byteChannel
   * @param channel
   * @param buffer
   * @return
   * @throws IOException
   * @throws DataChannelException
   * @throws UnsupportedOperationException
   */
  private long transferBytesFromSocket(IDataChannel byteChannel,
      SocketChannel channel, ByteBuffer buffer, long remainingBytes)
      throws IOException, UnsupportedOperationException, DataChannelException {

    long bytesRead = -1;

    if (this.writeUseTransferFrom) {
      if (this.position == -1) {
        this.position = byteChannel.getPosition();
      }
      if (logger.isLoggable(Level.FINEST)) {
        logger.finest(String.format(OPERATION_WRITE_USING_TRANSFER_FROM, channel.socket().getRemoteSocketAddress()));
      }
      bytesRead = byteChannel.transferFrom(channel, this.position
          + this.bytesReceived, this.count - this.bytesReceived);
    } else {

      if (buffer.hasRemaining()) {
        int bytesWritten = byteChannel.write(buffer);

        bytesRead = bytesWritten;
      } else {
        if (this.position >= 0) {
          byteChannel.setPosition(this.position + this.bytesReceived);
        }

        buffer.clear();
        if (buffer.remaining() > remainingBytes) {
          buffer.limit((int) remainingBytes);
        }

        if (channel.read(buffer) == -1) {
          return -1;
        }

        buffer.flip();

        int bytesWritten = byteChannel.write(buffer);
        bytesRead = bytesWritten;
      }
    }

    return bytesRead;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isWriting() {
    return this.writing;
  }
}
