/*
 * $Id: IOUtils.java 88412 2009-02-13 22:13:31Z vfusco $
 */
package tecgraf.ftc_1_4.utils;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.WritableByteChannel;

/**
 * Classe de mtodos utilitrios para operaes de I/O.
 * 
 * @author Tecgraf
 */
public class IOUtils {

  /**
   * Sobrecarga do mtodo
   * {@link #transferTo(FileChannel, long, long, WritableByteChannel, ByteBuffer)}
   * que j especifica um buffer <i>default</i> de 1 MB.
   * 
   * @param srcChannel FileChannel de origem para o arquivo a ser transmitido.
   * @param position posio a partir da qual o arquivo dever ser transmitido.
   * @param count nmero de bytes a serem transmitidos.
   * @param dstChannel WritableByteChannel de destino para a transferncia.
   * 
   * @return nmero de bytes efetivamente transmitidos.
   * 
   * @throws IOException
   */
  public static long transferTo(FileChannel srcChannel, long position,
    long count, WritableByteChannel dstChannel) throws IOException {
    ByteBuffer buffer = ByteBuffer.allocate(1024 * 1024);
    return transferTo(srcChannel, position, count, dstChannel, buffer);
  }

  /**
   * <p>
   * Mtodo para transferncia de bytes de um {@code FileChannel} para um
   * {@code WritableByteChannel}. Este mtodo procura suprir a deficincia do
   * {@link FileChannel#transferTo(long, long, WritableByteChannel)} em lidar
   * com arquivos acima de 2GB. Vide
   * "http://bugs.sun.com/bugdatabase/view_bug.do?bug_id=6253145" .
   * </p>
   * <p>
   * Procurou-se seguir a assinatura e comportamento esperado do mtodo
   * original. Para maiores referncias, consulte a API da Sun.
   * </p>
   * 
   * @param srcChannel FileChannel de origem para o arquivo a ser transmitido.
   * @param position posio a partir da qual o arquivo dever ser transmitido.
   * @param count nmero de bytes a serem transmitidos.
   * @param dstChannel WritableByteChannel de destino para a transferncia.
   * @param buffer buffer a ser usado para a transferncia.
   * 
   * @return nmero de bytes efetivamente transmitidos.
   * 
   * @throws IOException
   */
  public static long transferTo(FileChannel srcChannel, long position,
    long count, WritableByteChannel dstChannel, ByteBuffer buffer)
    throws IOException {
    if (srcChannel == null) {
      throw new IllegalArgumentException("srcChannel == null");
    }
    if (position < 0) {
      throw new IllegalArgumentException("position < 0");
    }
    if (count < 0) {
      throw new IllegalArgumentException("count < 0");
    }
    if (dstChannel == null) {
      throw new IllegalArgumentException("dstChannel == null");
    }
    if (buffer == null) {
      throw new IllegalArgumentException("nioBuffer == null");
    }
    long countRemaining = count;
    long totalBytesWritten = 0;
    long positionBackup = srcChannel.position();
    srcChannel.position(position);
    while (countRemaining > 0) {
      buffer.clear();
      if (buffer.remaining() > countRemaining) {
        buffer.limit((int) countRemaining);
      }
      int bytesRead = srcChannel.read(buffer);
      if (bytesRead == -1) {
        srcChannel.position(positionBackup);
        return totalBytesWritten;
      }
      buffer.flip();
      int bytesWritten = dstChannel.write(buffer);
      if (bytesWritten == -1) {
        srcChannel.position(positionBackup);
        return totalBytesWritten;
      }
      countRemaining -= bytesWritten;
      totalBytesWritten += bytesWritten;
      if (buffer.hasRemaining()) {
        srcChannel.position(positionBackup);
        return totalBytesWritten;
      }
    }
    srcChannel.position(positionBackup);
    return totalBytesWritten;
  }

  /**
   * <p>
   * Mtodo para transferncia de bytes de um {@code FileChannel} para um
   * {@code WritableByteChannel}. Este mtodo procura suprir a deficincia do
   * {@link FileChannel#transferTo(long, long, WritableByteChannel)} em lidar
   * com arquivos acima de 2GB. Vide
   * "http://bugs.sun.com/bugdatabase/view_bug.do?bug_id=6253145" .
   * </p>
   * <p>
   * Procurou-se seguir a assinatura e comportamento esperado do mtodo
   * original. Para maiores referncias, consulte a API da Sun.
   * </p>
   * 
   * @param srcChannel FileChannel de origem para o arquivo a ser transmitido.
   * @param position posio a partir da qual o arquivo dever ser transmitido.
   * @param count nmero de bytes a serem transmitidos.
   * @param dstChannel WritableByteChannel de destino para a transferncia.
   * @param buffer buffer a ser usado para a transferncia.
   * 
   * @return nmero de bytes efetivamente transmitidos.
   * 
   * @throws IOException
   */
  public static long transferToNonBlock(FileChannel srcChannel, long position,
    long count, WritableByteChannel dstChannel, ByteBuffer buffer)
    throws IOException {

    if (srcChannel == null) {
      throw new IllegalArgumentException("srcChannel == null");
    }
    if (position < 0) {
      throw new IllegalArgumentException("position < 0");
    }
    if (count < 0) {
      throw new IllegalArgumentException("count < 0");
    }
    if (dstChannel == null) {
      throw new IllegalArgumentException("dstChannel == null");
    }
    if (buffer == null) {
      throw new IllegalArgumentException("nioBuffer == null");
    }

    long positionBackup = srcChannel.position();
    srcChannel.position(position);

    buffer.clear();
    if (buffer.remaining() > count) {
      buffer.limit((int) count);
    }
    int bytesRead = srcChannel.read(buffer);
    if (bytesRead == -1) {
      srcChannel.position(positionBackup);
      return -1;
    }
    buffer.flip();
    int bytesWritten = dstChannel.write(buffer);

    srcChannel.position(positionBackup);
    return bytesWritten;

  }

  public static long transferFromNonBlock(FileChannel dstChannel,
    long position, long count, ReadableByteChannel srcChannel, ByteBuffer buffer)
    throws IOException {

    if (srcChannel == null) {
      throw new IllegalArgumentException("srcChannel == null");
    }
    if (position < 0) {
      throw new IllegalArgumentException("position < 0");
    }
    if (count < 0) {
      throw new IllegalArgumentException("count < 0");
    }
    if (dstChannel == null) {
      throw new IllegalArgumentException("dstChannel == null");
    }
    if (buffer == null) {
      throw new IllegalArgumentException("nioBuffer == null");
    }

    long totalBytesWritten = 0;
    long positionBackup = dstChannel.position();
    dstChannel.position(position);

    buffer.clear();
    if (buffer.remaining() > count) {
      buffer.limit((int) count);
    }
    int bytesRead = srcChannel.read(buffer);
    if (bytesRead == -1) {
      dstChannel.position(positionBackup);
      return totalBytesWritten;
    }
    buffer.flip();
    int bytesWritten = dstChannel.write(buffer);
    if (bytesWritten == -1) {
      dstChannel.position(positionBackup);
      return totalBytesWritten;
    }

    totalBytesWritten += bytesWritten;

    dstChannel.position(positionBackup);
    return totalBytesWritten;
  }

}
