package br.pucrio.tecgraf.soma.serviceapi.persistence.specification.impl;

import java.lang.reflect.Field;

import jakarta.persistence.criteria.CriteriaBuilder;
import jakarta.persistence.criteria.Predicate;
import jakarta.persistence.criteria.Root;

import br.pucrio.tecgraf.soma.serviceapi.persistence.specification.JPASpecification;

public class GenericSpecification<T> extends JPASpecification<T> {

	private String fieldName;
	private ComparisonOperator comparisonOperator;
	private Object value;
	private Class<T> typeParameterClass;

	public GenericSpecification(Class<T> typeParameterClass, String fieldName, ComparisonOperator comparisonOperator, Object value) {
		this.typeParameterClass = typeParameterClass;
		this.fieldName = fieldName;
		this.comparisonOperator = comparisonOperator;
		this.value = value;
	}

	@SuppressWarnings({ "rawtypes", "unchecked" })
	@Override
	public Predicate toPredicate(Root<T> root, CriteriaBuilder cb) {
		switch (comparisonOperator) {
		case GT:
			return cb.and(cb.greaterThan(root.get(fieldName), (Comparable) value));
		case LT:
			return cb.and(cb.lessThan(root.get(fieldName), (Comparable) value));
		case GTE:
			return cb.and(cb.greaterThanOrEqualTo(root.get(fieldName), (Comparable) value));
		case LTE:
			return cb.and(cb.lessThanOrEqualTo(root.get(fieldName), (Comparable) value));
		case EQ:
			return cb.and(cb.equal(root.get(fieldName), value));
		case NE:
			return cb.and(cb.notEqual(root.get(fieldName), value));
		case LIKE:
			return cb.and(cb.like(root.get(fieldName), "%" + value + "%"));
		default:
			return null;
		}
	}

	@SuppressWarnings({ "unchecked", "rawtypes" })
	@Override
	public boolean isSatisfiedBy(T element) {
		try {

			Field field = typeParameterClass.getDeclaredField(fieldName);
			field.setAccessible(true);

			switch (comparisonOperator) {
			case GT:
				return ((Comparable) field.get(element)).compareTo(value) > 0;
			case LT:
				return ((Comparable) field.get(element)).compareTo(value) < 0;
			case GTE:
				return ((Comparable) field.get(element)).compareTo(value) >= 0;
			case LTE:
				return ((Comparable) field.get(element)).compareTo(value) <= 0;
			case EQ:
				return field.get(element).equals(value);
			case NE:
				return !field.get(element).equals(value);
			case LIKE:
				return ((String) field.get(element)).contains((String) value);
			default:
				return false;
			}
		} catch (Exception e) {
			e.printStackTrace();
			return false;
		}

	}

	public enum ComparisonOperator {
		GT, LT, GTE, LTE, EQ, NE, LIKE;

	}
	
}
