package br.pucrio.tecgraf.soma.serviceapi.persistence.specification.impl;

import com.github.tennaito.rsql.misc.ArgumentFormatException;
import com.github.tennaito.rsql.misc.DefaultArgumentParser;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.math.BigDecimal;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeFormatterBuilder;
import java.time.format.DateTimeParseException;
import java.time.temporal.ChronoField;
import java.util.Date;
import java.util.logging.Level;
import java.util.logging.Logger;

public class RSQLArgumentParser extends DefaultArgumentParser {
  private static final Logger LOG = Logger.getLogger(RSQLArgumentParser.class.getName());
  private final DateTimeFormatter formatter;

  public RSQLArgumentParser() {
      /*
       Allowed patterns: (ISO 8601)
       - "yyyy/MM/ddx",
       - "yyyy/MM/dd HH:mmx",
       - "yyyy/MM/dd HH:mm:ssx",
       - "yyyy/MM/dd HH:mm:ss.SSSx"
       x  zone-offset. Allowed patterns:  -hh; -hhmm; +hh; +hhmm
       Date-hour separators can be " " or "T".
       */
    this.formatter = new DateTimeFormatterBuilder()
            .appendPattern("yyyy/MM/dd[[' ']['T']HH:mm[:ss][.SSS]]x")
            .parseDefaulting(ChronoField.HOUR_OF_DAY, 0)
            .parseDefaulting(ChronoField.MINUTE_OF_HOUR, 0)
            .parseDefaulting(ChronoField.SECOND_OF_MINUTE, 0)
            .toFormatter();
  }

  @Override
  public <T> T parse(String argument, Class<T> type) throws ArgumentFormatException, IllegalArgumentException {

    LOG.log(Level.INFO, "Parsing argument ''{0}'' as type {1}, thread {2}", new Object[]{argument, type.getSimpleName(), Thread.currentThread().getName()});

    // Nullable object
    if (argument == null || "null".equals(argument.trim().toLowerCase())) {
      return null;
    }

    // common types
    try {
      if (type.equals(String.class)) return (T) argument;
      if (type.equals(Integer.class) || type.equals(int.class))
        return (T) Integer.valueOf(argument);
      if (type.equals(Boolean.class) || type.equals(boolean.class))
        return (T) Boolean.valueOf(argument);
      if (type.isEnum()) return (T) Enum.valueOf((Class<Enum>) type, argument);
      if (type.equals(Float.class) || type.equals(float.class)) return (T) Float.valueOf(argument);
      if (type.equals(Double.class) || type.equals(double.class))
        return (T) Double.valueOf(argument);
      if (type.equals(Long.class) || type.equals(long.class)) return (T) Long.valueOf(argument);
      if (type.equals(BigDecimal.class)) return (T) new BigDecimal(argument);
    }
    catch (IllegalArgumentException ex) {
      throw new ArgumentFormatException(argument, type);
    }

    if (type.equals(Date.class)) {
      return (T) Date.from(parseDate(argument));
    }
    if (type.equals(LocalDateTime.class)) {
      return (T) LocalDateTime.ofInstant(parseDate(argument), ZoneOffset.UTC);
    }

    // try to parse via valueOf(String s) method
    try {
      LOG.log(Level.INFO, "Trying to get and invoke valueOf(String s) method on {0}", type);
      Method method = type.getMethod("valueOf", String.class);
      return (T) method.invoke(type, argument);
    }
    catch (InvocationTargetException ex) {
      throw new ArgumentFormatException(argument, type);
    }
    catch (ReflectiveOperationException ex) {
      LOG.log(Level.WARNING, "{0} does not have method valueOf(String s) or method is inaccessible", type);
      throw new IllegalArgumentException("Cannot parse argument type " + type);
    }
  }

  private Instant parseDate(String argument) {
    try {
      return ZonedDateTime.parse(argument, formatter).toInstant();
    }
    catch (DateTimeParseException ex) {
      throw new ArgumentFormatException(argument, ZonedDateTime.class);
    }
  }
}
