package br.pucrio.tecgraf.soma.serviceapi.persistence.unitofwork.impl;

import java.io.Closeable;
import java.io.IOException;
import java.util.Map;

import jakarta.persistence.EntityManager;
import jakarta.persistence.EntityManagerFactory;
import jakarta.persistence.FlushModeType;
import jakarta.persistence.RollbackException;
import jakarta.persistence.SynchronizationType;

import br.pucrio.tecgraf.soma.serviceapi.persistence.unitofwork.UnitOfWork;

public abstract class JPAUnitOfWork implements UnitOfWork, Closeable {

	private EntityManager entityManager;
	private EntityManagerFactory factory;
	private FlushModeType flushModeType;
	private SynchronizationType synchronizationType;
	private Map<?, ?> map;

	public JPAUnitOfWork(EntityManagerFactory factory) {
		this.factory = factory;
		start();
	}

	public void configure(FlushModeType flushModeType, SynchronizationType synchronizationType, Map<?, ?> map) {
		this.flushModeType = flushModeType;
		this.synchronizationType = synchronizationType;
		this.map = map;
	}

	@Override
	public void start() {
		if (entityManager != null) {
			throw new IllegalStateException("Unit of work is already started");
		}
		if (synchronizationType != null || map != null) {
			if (synchronizationType != null && map != null)
				entityManager = factory.createEntityManager(synchronizationType, map);
			else if (map != null)
				entityManager = factory.createEntityManager(map);
			else
				entityManager = factory.createEntityManager(synchronizationType);
		} else {
			entityManager = factory.createEntityManager();
		}
		if (flushModeType != null)
			entityManager.setFlushMode(flushModeType);
		entityManager.getTransaction().begin();
	}

	/**
     * Commit the current resource transaction, writing any 
     * unflushed changes to the database.  
     * @throws IllegalStateException if <code>isActive()</code> is false
     * @throws RollbackException if the commit fails
     */
	@Override
	public void end(){
		if (!entityManager.isOpen())
			throw new IllegalStateException("Unit of work was not started");
		try {
			entityManager.getTransaction().commit();
		} finally {
			entityManager.close();
		}
	}

	/**
     * Mark the current resource transaction so that the only 
     * possible outcome of the transaction is for the transaction 
     * to be rolled back. 
     * @throws IllegalStateException if <code>isActive()</code> is false
     */
	public void cancel() {
		entityManager.getTransaction().setRollbackOnly();
	}

	@Override
	public void close() throws IOException {
		end();
	}
	
	protected EntityManager getEntityManager() {
		return entityManager;
	}

}
