/*
 * (c) 2003-2005, 2009, 2010 ThoughtWorks Ltd
 * All rights reserved.
 *
 * The software in this package is published under the terms of the BSD
 * style license a copy of which has been included with this distribution in
 * the LICENSE.txt file.
 * 
 * Created on 11-May-2004
 */
package com.thoughtworks.proxy.toys.multicast;

import java.lang.reflect.Array;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.List;

import com.thoughtworks.proxy.Invoker;
import com.thoughtworks.proxy.ProxyFactory;
import com.thoughtworks.proxy.kit.ReflectionUtils;

/**
 * A {@link Invoker} implementation that multicasts calls to multiple targets. Proxies generated by this class will
 * forward all method invocations to an array of underlying objects. The behavior is recursive, so return values will
 * also be multicasting objects.
 *
 * @author Aslak Helles&oslash;y
 * @author Chris Stevenson
 * @author J&ouml;rg Schaible
 * @since 0.1
 */
public class MulticastingInvoker<T> implements Invoker {
    private static final long serialVersionUID = 1L;
    private static final Method multicastTargetsDirect;
    private static final Method multicastTargetsIndirect;
    private static final Method getTargetsInArray;
    private static final Method getTargetsInTypedArray;

    static {
        try {
            multicastTargetsDirect = Multicast.class.getMethod("multicastTargets", Method.class, Object[].class);
            multicastTargetsIndirect = Multicast.class.getMethod("multicastTargets", Class.class, String.class, Object[].class);
            getTargetsInArray = Multicast.class.getMethod("getTargetsInArray");
            getTargetsInTypedArray = Multicast.class.getMethod("getTargetsInArray", Class.class);
        } catch (NoSuchMethodException e) {
            throw new ExceptionInInitializerError(e.toString());
        }
    }

    private Class<?>[] types;
    private ProxyFactory proxyFactory;
    private Object[] targets;

    /**
     * Construct a MulticastingInvoker.
     *
     * @param type         the implemented types
     * @param proxyFactory the {@link ProxyFactory} to use
     * @param targets      the target instances where the proxy delegates a call
     * @since 0.1
     */
    public MulticastingInvoker(final Class<?>[] type, final ProxyFactory proxyFactory, final Object[] targets) {
        this.types = type;
        this.proxyFactory = proxyFactory;
        this.targets = targets;
    }

    /**
     * Create a proxy for this Invoker.
     *
     * @return the new proxy
     * @since 0.1
     */
    public T proxy() {
        final Class<?>[] classes;
        int i;
        i = types.length;
        while (--i >= 0 && types[i] != Multicast.class) {
        }
        if (i < 0) {
            classes = new Class[types.length + 1];
            if (classes.length > 1) {
                System.arraycopy(types, 0, classes, 1, types.length);
            }
            classes[0] = Multicast.class;
        } else {
            classes = types;
        }
        return proxyFactory.<T>createProxy(this, classes);
    }

    public Object invoke(final Object proxy, Method method, Object[] args) throws Throwable {
        if (getTargetsInArray.equals(method)) {
            return targets;
        } else if (getTargetsInTypedArray.equals(method)) {
            final Object[] elements = Object[].class.cast(Array.newInstance(Class.class.cast(args[0]), targets.length));
            System.arraycopy(targets, 0, elements, 0, targets.length);
            return elements;
        } else if (multicastTargetsDirect.equals(method)) {
            method = (Method) args[0];
            args = (Object[]) args[1];
        } else if (multicastTargetsIndirect.equals(method)) {
            final Object[] newArgs = args[2] == null ? new Object[0] : Object[].class.cast(args[2]);
            method = ReflectionUtils.getMatchingMethod(Class.class.cast(args[0]), String.class.cast(args[1]), newArgs);
            args = newArgs;
        }
        final List<Object> invocationResults = new ArrayList<Object>();
        for (Object target : targets) {
            if (method.getDeclaringClass().isInstance(target)) {
                Object result = method.invoke(target, args);
                if (result != null) {
                    invocationResults.add(result);
                }
            }
        }
        if (invocationResults.size() == 0) {
            return null;
        } else if (invocationResults.size() == 1) {
            return invocationResults.get(0);
        } else if (method.getReturnType().equals(byte.class)) {
            return addBytes(invocationResults.toArray());
        } else if (method.getReturnType().equals(char.class)) {
            return addChars(invocationResults.toArray());
        } else if (method.getReturnType().equals(short.class)) {
            return addShorts(invocationResults.toArray());
        } else if (method.getReturnType().equals(int.class)) {
            return addIntegers(invocationResults.toArray());
        } else if (method.getReturnType().equals(long.class)) {
            return addLongs(invocationResults.toArray());
        } else if (method.getReturnType().equals(float.class)) {
            return addFloats(invocationResults.toArray());
        } else if (method.getReturnType().equals(double.class)) {
            return addDoubles(invocationResults.toArray());
        } else if (method.getReturnType().equals(boolean.class)) {
            return andBooleans(invocationResults.toArray());
        } else {
            return Multicasting.proxy(invocationResults.toArray()).build(proxyFactory);
        }
    }

    private static Byte addBytes(final Object[] args) {
        byte result = 0;
        for (Object arg : args) {
            result += Byte.class.cast(arg);
        }
        return result;
    }

    private static Character addChars(final Object[] args) {
        char result = 0;
        for (Object arg : args) {
            result += Character.class.cast(arg);
        }
        return result;
    }

    private static Short addShorts(final Object[] args) {
        short result = 0;
        for (Object arg : args) {
            result += Short.class.cast(arg);
        }
        return result;
    }

    private static Integer addIntegers(final Object[] args) {
        int result = 0;
        for (Object arg : args) {
            result += Integer.class.cast(arg);
        }
        return result;
    }

    private static Long addLongs(final Object[] args) {
        long result = 0;
        for (Object arg : args) {
            result += Long.class.cast(arg);
        }
        return result;
    }

    private static Float addFloats(final Object[] args) {
        float result = 0;
        for (Object arg : args) {
            result += Float.class.cast(arg);
        }
        return result;
    }

    private static Double addDoubles(final Object[] args) {
        double result = 0;
        for (Object arg : args) {
            result += Double.class.cast(arg);
        }
        return result;
    }

    private static Boolean andBooleans(final Object[] args) {
        for (Object arg : args) {
            if (!Boolean.class.cast(arg)) {
                return Boolean.FALSE;
            }
        }
        return Boolean.TRUE;
    }
}