package io.quarkus.deployment.builditem;

import java.io.IOException;
import java.nio.file.FileSystem;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import io.quarkus.bootstrap.model.PathsCollection;
import io.quarkus.builder.item.SimpleBuildItem;
import io.quarkus.fs.util.ZipUtils;
import io.quarkus.paths.PathCollection;
import io.quarkus.paths.PathList;

public final class ArchiveRootBuildItem extends SimpleBuildItem {

    public static class Builder {

        private List<Path> archiveRoots = new ArrayList<>();
        private Collection<Path> excludedFromIndexing;

        private Builder() {
        }

        public Builder addArchiveRoot(Path root) {
            this.archiveRoots.add(root);
            return this;
        }

        public Builder addArchiveRoots(PathCollection paths) {
            paths.forEach(archiveRoots::add);
            return this;
        }

        public Builder setExcludedFromIndexing(Collection<Path> excludedFromIndexing) {
            this.excludedFromIndexing = excludedFromIndexing;
            return this;
        }

        @Deprecated
        public Builder setArchiveLocation(Path archiveLocation) {
            this.archiveRoots.clear();
            this.archiveRoots.add(archiveLocation);
            return this;
        }

        public ArchiveRootBuildItem build(QuarkusBuildCloseablesBuildItem buildCloseables) throws IOException {
            return new ArchiveRootBuildItem(this, buildCloseables);
        }
    }

    public static Builder builder() {
        return new Builder();
    }

    private final Path archiveRoot;
    private final Collection<Path> excludedFromIndexing;
    private final PathCollection rootDirs;
    private final PathCollection paths;

    public ArchiveRootBuildItem(Path appClassesDir) {
        this(appClassesDir, appClassesDir);
    }

    @Deprecated
    public ArchiveRootBuildItem(Path archiveLocation, Path archiveRoot) {
        this(archiveLocation, archiveRoot, Collections.emptySet());
    }

    private ArchiveRootBuildItem(Path archiveLocation, Path archiveRoot, Collection<Path> excludedFromIndexing) {
        if (!Files.isDirectory(archiveRoot)) {
            throw new IllegalArgumentException(archiveRoot + " does not point to the application output directory");
        }
        this.rootDirs = PathList.of(archiveRoot);
        this.paths = PathList.of(archiveLocation);
        this.archiveRoot = archiveRoot;
        this.excludedFromIndexing = excludedFromIndexing;
    }

    private ArchiveRootBuildItem(Builder builder, QuarkusBuildCloseablesBuildItem buildCloseables) throws IOException {
        this.excludedFromIndexing = builder.excludedFromIndexing;
        if (!builder.archiveRoots.isEmpty()) {
            final PathList.Builder rootDirs = PathList.builder();
            final PathList.Builder paths = PathList.builder();
            for (Path root : builder.archiveRoots) {
                paths.add(root);
                if (Files.isDirectory(root)) {
                    rootDirs.add(root);
                } else {
                    final FileSystem fs = buildCloseables.add(ZipUtils.newFileSystem(root));
                    fs.getRootDirectories().forEach(rootDirs::add);
                }
            }
            this.rootDirs = rootDirs.build();
            this.paths = paths.build();
            this.archiveRoot = this.rootDirs.iterator().next();
        } else {
            this.paths = this.rootDirs = PathsCollection.of();
            this.archiveRoot = null;
        }
    }

    /**
     * If this archive is a jar file it will return the path to the jar file on the file system,
     * otherwise it will return the directory that this corresponds to.
     *
     * @deprecated in favor of {@link #getResolvedPaths()}
     */
    @Deprecated
    public Path getArchiveLocation() {
        final Iterator<Path> i = paths.iterator();
        Path last = i.next();
        while (i.hasNext()) {
            last = i.next();
        }
        return last;
    }

    /**
     *
     * Returns a path representing the archive root. Note that if this is a jar archive this is not the path to the
     * jar, but rather a path to the root of the mounted {@link com.sun.nio.zipfs.ZipFileSystem}
     *
     * @return The archive root.
     * @deprecated in favor of {@link #getRootDirectories()}
     */
    @Deprecated
    public Path getArchiveRoot() {
        return archiveRoot;
    }

    /**
     * Collection of path representing the archive's root directories. If there is a JAR among the paths
     * (returned by {@link #getResolvedPaths()} this method will return the path to the root of the mounted
     * {@link java.nio.file.ZipFileSystem}
     * instead.
     *
     * @deprecated in favor of {@link #getRootDirectories()}
     *
     * @return Collection of path representing the archive's root directories.
     */
    @Deprecated
    public PathsCollection getRootDirs() {
        return PathsCollection.from(rootDirs);
    }

    /**
     * Collection of path representing the archive's root directories. If there is a JAR among the paths
     * (returned by {@link #getResolvedPaths()} this method will return the path to the root of the mounted
     * {@link java.nio.file.ZipFileSystem}
     * instead.
     *
     * @return Collection of path representing the archive's root directories.
     */
    public PathCollection getRootDirectories() {
        return rootDirs;
    }

    /**
     * Collection of paths that collectively constitute the application archive's content.
     *
     * @deprecated in favor of {@link #getResolvedPaths()}
     *
     * @return collection of paths that collectively constitute the application archive content.
     */
    @Deprecated
    public PathsCollection getPaths() {
        return PathsCollection.from(paths);
    }

    /**
     * Collection of paths that collectively constitute the application archive's content.
     *
     * @return collection of paths that collectively constitute the application archive content.
     */
    public PathCollection getResolvedPaths() {
        return paths;
    }

    public boolean isExcludedFromIndexing(Path p) {
        return excludedFromIndexing.contains(p);
    }
}
