/*
 * Decompiled with CFR 0.152.
 */
package net.intelie.disq;

import java.io.IOException;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import net.intelie.disq.Buffer;
import net.intelie.disq.DataFileReader;
import net.intelie.disq.DataFileWriter;
import net.intelie.disq.Lenient;
import net.intelie.disq.RawQueue;
import net.intelie.disq.StateFile;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DiskRawQueue
implements RawQueue {
    public static final int FAILED_READ_THRESHOLD = 64;
    private static final Logger LOGGER = LoggerFactory.getLogger(DiskRawQueue.class);
    private final long maxSize;
    private final long dataFileLimit;
    private final Lenient lenient;
    private final boolean flushOnRead;
    private final boolean flushOnWrite;
    private final boolean deleteOldestOnOverflow;
    private boolean temp;
    private Path directory;
    private boolean closed = false;
    private StateFile state;
    private DataFileReader reader;
    private DataFileWriter writer;
    private int failedReads = 0;
    private int oldFailedReads = 0;
    private long flushCount = 0L;

    public DiskRawQueue(Path directory, long maxSize) {
        this(directory, maxSize, true, true, true);
    }

    public DiskRawQueue(Path directory, long maxSize, boolean flushOnPop, boolean flushOnPush, boolean deleteOldestOnOverflow) {
        this.directory = directory;
        this.maxSize = Math.max(Math.min(maxSize, 259845521287L), 61952L);
        this.dataFileLimit = Math.max(512L, this.maxSize / 121L + (long)(this.maxSize % 121L > 0L ? 1 : 0));
        this.lenient = new Lenient(this);
        this.flushOnRead = flushOnPop;
        this.flushOnWrite = flushOnPush;
        this.deleteOldestOnOverflow = deleteOldestOnOverflow;
        this.temp = false;
        this.reopen();
    }

    @Override
    public synchronized void reopen() {
        this.internalClose();
        this.closed = false;
    }

    public Path path() {
        return this.directory;
    }

    private void internalOpen() throws IOException {
        this.internalClose();
        if (this.directory == null) {
            this.directory = Files.createTempDirectory("disq", new FileAttribute[0]);
            this.temp = true;
        }
        Files.createDirectories(this.directory, new FileAttribute[0]);
        this.state = new StateFile(this.directory.resolve("state"));
        this.writer = null;
        this.reader = null;
        this.gc();
    }

    @Override
    public synchronized void touch() throws IOException {
        if (this.state == null) {
            this.internalOpen();
        }
    }

    private void checkNotClosed() {
        if (this.closed) {
            throw new IllegalStateException("This queue is already closed.");
        }
    }

    @Override
    public synchronized long bytes() {
        this.checkNotClosed();
        return this.lenient.performSafe(() -> this.state.getBytes(), 0L);
    }

    @Override
    public synchronized long count() {
        this.checkNotClosed();
        return this.lenient.performSafe(() -> this.state.getCount(), 0L);
    }

    public synchronized long files() {
        this.checkNotClosed();
        return this.lenient.performSafe(() -> this.state.getNumberOfFiles(), 0L);
    }

    @Override
    public synchronized long remainingBytes() {
        this.checkNotClosed();
        return this.lenient.performSafe(() -> this.maxSize - this.state.getBytes(), 0L);
    }

    public long flushCount() {
        return this.flushCount;
    }

    @Override
    public synchronized long remainingCount() {
        this.checkNotClosed();
        return this.lenient.performSafe(() -> {
            if (this.state.getCount() == 0L) {
                return this.maxSize / 4L;
            }
            double bytesPerElement = (double)this.state.getBytes() / (double)this.state.getCount();
            return (long)((double)(this.maxSize - this.state.getBytes()) / bytesPerElement);
        }, 0L);
    }

    @Override
    public synchronized void clear() throws IOException {
        this.checkNotClosed();
        this.lenient.perform(() -> {
            this.state.clear();
            this.internalFlush();
            this.reopen();
            return 1L;
        });
    }

    @Override
    public synchronized boolean pop(Buffer buffer) throws IOException {
        this.checkNotClosed();
        return this.lenient.perform(() -> {
            if (!this.checkFailedReads()) {
                return 0L;
            }
            if (this.checkReadEOF()) {
                return 0L;
            }
            int read = this.innerRead(buffer);
            this.state.addReadCount(read);
            if (this.flushOnRead) {
                this.internalFlush();
            }
            this.checkReadEOF();
            return 1L;
        }) > 0L;
    }

    private boolean checkFailedReads() throws IOException {
        if (this.failedReads >= 64) {
            LOGGER.info("Detected corrupted file #{}, backing up and moving on.", (Object)this.state.getReadFile());
            boolean wasSame = this.state.sameFileReadWrite();
            this.deleteOldestFile(true);
            if (wasSame) {
                this.clear();
                return false;
            }
        }
        return true;
    }

    private int innerRead(Buffer buffer) throws IOException {
        try {
            int read = this.reader().read(buffer);
            this.oldFailedReads = this.failedReads;
            this.failedReads = 0;
            return read;
        }
        catch (Throwable e) {
            ++this.failedReads;
            throw e;
        }
    }

    @Override
    public synchronized boolean peek(Buffer buffer) throws IOException {
        this.checkNotClosed();
        return this.lenient.perform(() -> {
            if (this.checkReadEOF()) {
                return 0L;
            }
            this.reader().peek(buffer);
            return 1L;
        }) > 0L;
    }

    private void deleteOldestFile(boolean renameFile) throws IOException {
        int currentFile = this.state.getReadFile();
        this.state.advanceReadFile(this.reader().size());
        this.reader.close();
        this.failedReads = 0;
        this.internalFlush();
        this.reader = null;
        this.tryDeleteFile(currentFile, renameFile);
    }

    @Override
    public synchronized void notifyFailedRead() {
        this.failedReads = this.oldFailedReads + 1;
    }

    @Override
    public synchronized boolean push(Buffer buffer) throws IOException {
        this.checkNotClosed();
        return this.lenient.perform(() -> {
            this.checkWriteEOF();
            if (this.checkFutureQueueOverflow(buffer.count())) {
                return 0L;
            }
            int written = this.writer().write(buffer);
            this.state.addWriteCount(written);
            if (this.flushOnWrite) {
                this.internalFlush();
            }
            this.checkWriteEOF();
            return 1L;
        }) > 0L;
    }

    private boolean checkFutureQueueOverflow(int count) throws IOException {
        if (this.deleteOldestOnOverflow) {
            while (!this.state.sameFileReadWrite() && this.willOverflow(count)) {
                this.deleteOldestFile(false);
            }
            return false;
        }
        return this.willOverflow(count);
    }

    @Override
    public synchronized void flush() throws IOException {
        this.checkNotClosed();
        this.lenient.perform(this::internalFlush);
    }

    private long internalFlush() throws IOException {
        if (this.writer != null) {
            this.writer.flush();
        }
        this.state.flush();
        ++this.flushCount;
        return 1L;
    }

    @Override
    public synchronized void close() {
        this.closed = true;
        this.internalClose();
    }

    private void internalClose() {
        this.lenient.safeClose(this.reader);
        this.reader = null;
        this.lenient.safeClose(this.writer);
        this.writer = null;
        this.lenient.safeClose(this.state);
        this.state = null;
        if (this.temp) {
            this.lenient.safeDelete(this.directory);
            this.directory = null;
            this.temp = false;
        }
    }

    private boolean willOverflow(int count) throws IOException {
        return this.bytes() + (long)count + 4L > this.maxSize || this.files() >= 121L;
    }

    private boolean checkReadEOF() throws IOException {
        while (!this.state.sameFileReadWrite() && this.state.readFileEof()) {
            this.deleteOldestFile(false);
        }
        if (this.state.needsFlushBeforePop()) {
            this.internalFlush();
        }
        return this.state.getCount() == 0L;
    }

    private DataFileReader reader() throws IOException {
        return this.reader != null ? this.reader : (this.reader = this.openReader());
    }

    private void checkWriteEOF() throws IOException {
        if (this.state.getWritePosition() >= this.dataFileLimit) {
            this.advanceWriteFile();
        }
    }

    private DataFileWriter writer() throws IOException {
        return this.writer != null ? this.writer : (this.writer = this.openWriter());
    }

    private void advanceWriteFile() throws IOException {
        this.writer().close();
        this.state.advanceWriteFile();
        this.internalFlush();
        this.writer = null;
    }

    private void gc() throws IOException {
        Path file = this.makeDataPath(this.state.getReadFile());
        boolean shouldFlush = false;
        while (!Files.exists(file, new LinkOption[0]) && !this.state.sameFileReadWrite()) {
            this.state.advanceReadFile(0L);
            file = this.makeDataPath(this.state.getReadFile());
            shouldFlush = true;
        }
        long totalBytes = 0L;
        long totalCount = 0L;
        for (int i = 0; i < 121; ++i) {
            Path path = this.makeDataPath(i);
            if (!Files.exists(path, new LinkOption[0])) continue;
            if (!this.state.isInUse(i)) {
                this.tryDeleteFile(i, false);
                continue;
            }
            totalBytes += Files.size(path);
            totalCount += (long)this.state.getFileCount(i);
        }
        if (shouldFlush |= this.state.fixCounts(totalCount, totalBytes)) {
            this.internalFlush();
        }
    }

    private void tryDeleteFile(int file, boolean renameFile) {
        Path from = this.makeDataPath(file);
        try {
            if (renameFile) {
                Path to = this.makeCorruptedPath(file);
                LOGGER.info("Backing up {} as {}", (Object)from, (Object)to);
                Files.move(from, to, new CopyOption[0]);
            } else {
                Files.delete(from);
            }
        }
        catch (Exception e) {
            LOGGER.info("Unable to delete file {}", (Object)from);
            LOGGER.info("Stacktrace", (Throwable)e);
        }
    }

    private Path makeDataPath(int state) {
        return this.directory.resolve(String.format("data%02x", state));
    }

    private Path makeCorruptedPath(int state) {
        return this.directory.resolve(String.format("data%02x.%d.corrupted", state, System.currentTimeMillis()));
    }

    private DataFileReader openReader() throws IOException {
        return new DataFileReader(this.makeDataPath(this.state.getReadFile()), this.state.getReadPosition());
    }

    private DataFileWriter openWriter() throws IOException {
        Files.createDirectories(this.directory, new FileAttribute[0]);
        return new DataFileWriter(this.makeDataPath(this.state.getWriteFile()), this.state.getWritePosition());
    }
}

