/*
 * @(#)XmlOutStream.java	1.13 98/12/17
 * 
 * Copyright (c) 1998 Sun Microsystems, Inc. All Rights Reserved.
 * 
 * This software is the confidential and proprietary information of Sun
 * Microsystems, Inc. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sun.
 * 
 * SUN MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SUN SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 */

package com.sun.xml.io;

import java.beans.*;
import java.io.*;
import java.lang.reflect.*;
import java.util.Hashtable;

import com.sun.xml.util.XmlChars;


/**
 * Publishes data conforming to an experimental XML DTD, which can be read
 * back by <code>XmlInStream</code>.  This is not the same model as object
 * serialization.  Rather than storing private state unless it's explicitly
 * tagged as <code>transient</code>, and requiring classes to implement the
 * <code>Serializable</code> interface, it works on all objects supporting
 * two common features.  Those features are present in JavaBeans:  <OL>
 *
 *	<LI> The object's class must have a public default constructor.
 *
 *	<LI> Properties needed to reconstitute the object must be
 *	available through public get/set methods visible through the
 *	class's BeanInfo.
 *
 *	</UL>
 *
 * <P> Objects, including arrays, are sent only once: this means that all
 * finite graphs are linearized on transmission.  Note that the only way
 * in which such graphs will be visible is through object properties.
 * (Or perhaps later, objects which externalize themselves.)
 *
 * <P> However, <em>infinite graphs can't be sent.</em>  There is a
 * settable limit on the number of objects that will be sent in a given
 * document.  Objects which create new objects when they are asked for
 * their properties can potentially create infinite graphs, and should
 * in general be avoided.  Similarly, objects which can't correctly be
 * used as keys in Hashtables.  (Perhaps they don't implement both the
 * <em>hashCode</em> and <em>equals</em> methods, which would be a
 * basic object model bug in that such a class.)
 *
 * <P> The XML formatted data is pretty-printed, so that it can be read
 * and edited with a normal text editor.  An XML DTD for this data format
 * is available for use when validating input or output, as a Java
 * resource in this package:  <em>com/sun/xml/io/stream.dtd</em>.
 *
 * <P> <em>Note that the Writer may be responsible for incorrectly
 * encoding characters, usually as <em>?</em> rather than a character
 * reference (such as <em>&amp;#1234;</em>).  Try to use UTF-8 writers
 * (or UTF-16 ones) to avoid this problem. </em>
 *
 * @see XmlInStream
 *
 * @author David Brownell
 * @version 1.13
 */
public class XmlOutStream implements ObjectOutput
{
    // XXX should register this...

    /**
     * This is the public ID for the stream DTD being used.
     */
    static final public String		publicId =
	    "+//Sun Microsystems Inc.//DTD Marshaling:0.5//EN";

    private Writer		out;
    private int			indentLevel = 0;

    // objects show only up once in the output stream; if they're
    // referred to more than once, references after the first are
    // XML IDREF attributes.  thus, we linearize all finite graphs.
    private int			id;
    private Hashtable		mapping = new Hashtable ();

    // infinite graphs can't be detected except when they try to
    // blow past some size limit.
    private int			sizeLimit = 10000;


    /**
     * Constructs an XML output stream built on top of the Writer.
     * The XML stream <em>must</em> be closed in order to propertly
     * terminate the XML output.
     *
     * @param w the writer to which the XML text is emitted.  If
     *	this is an OutputStreamWriter, the encoding in the XML
     *	declaration is derived from the encoding of that stream.
     *	<em>You are strongly advised to use UTF-8 output writers!</em>
     * @param jarURL optional URL for the JAR file holding classes
     *	that may be needed to associate behaviour with tree nodes.
     */
    public XmlOutStream (Writer w, String jarURL)
    throws IOException
    {
	String	encoding = null;

	if (w instanceof OutputStreamWriter) 
	    encoding = java2std (((OutputStreamWriter)w).getEncoding ());
	out = w;

	out.write ("<?xml version=\"1.0\"");
	if (encoding != null) {
	    out.write (" encoding=\"");
	    out.write (encoding);
	    out.write ("\"");
	}
	out.write ("?>\n");
	out.write ("<!DOCTYPE STREAM PUBLIC\n");
	    out.write ("    \"" + publicId + "\"\n");
	    out.write ("    \"stream.dtd\"\n");
	out.write ("    >\n");

	out.write ("<!-- experimental XML ObjectOutput implementation -->\n\n");
	out.write ("<STREAM");
	if (jarURL != null) {
	    out.write (" ARCHIVE=\"");
	    out.write (urlEncode (jarURL));
	    out.write ('\"');
	}
	out.write (">");
	indentLevel++;
    }

    private static boolean isAscii (byte c)
    {
	if ((c > 0x7f) || (c <= 0x20))
	    return true;
	return false;
    }

    private String urlEncode (String url)
    throws UnsupportedEncodingException
    {
	byte	bytes [] = url.getBytes ("UTF8");	// UTF-8
	boolean	doMunge = false;

	for (int i = 0; i < bytes.length && !doMunge; i++) {
	    if (isAscii (bytes [i]))
		doMunge = true;
	}
	if (doMunge) {
	    // at least one non-printable ASCII byte;
	    // gotta munge the bytes
	    StringBuffer	buf = new StringBuffer ();

	    for (int i = 0; i < bytes.length; i++)
		if (isAscii (bytes [i]))
		    buf.append ((char) bytes [i]);
		else {
		    buf.append ("%");
		    buf.append (Integer.toHexString (bytes [i]));
		}
	    return buf.toString ();
	} else
	    return url;
    }

    // try some of the common conversions from Java's internal
    // names (fit in class names) to standard ones.

    // package private
    static String java2std (String encodingName)
    {
	if (encodingName == null)
	    return null;

	if (encodingName.startsWith ("ISO8859_"))	// JDK 1.2
	    return "ISO-8859-" + encodingName.charAt (8);
	if (encodingName.startsWith ("8859_"))		// JDK 1.1
	    return "ISO-8859-" + encodingName.charAt (5);

	// JDK 1.1.5 doesn't accept all the standard names.
	// Includes:  "us-ascii", "UTF-8", "UTF-16", "EUC-JP".
	// XML parsers are expected to deal with this.

	// ... unclear just how much JDK 1.2 changed!

	if ("ASCII7".equalsIgnoreCase (encodingName)
		|| "ASCII".equalsIgnoreCase (encodingName))
	    return "US-ASCII";
	
	if ("UTF8".equalsIgnoreCase (encodingName))
	    return "UTF-8";
	if (encodingName.startsWith ("Unicode"))
	    return "UTF-16";
	
	if ("SJIS".equalsIgnoreCase (encodingName))
	    return "Shift_JIS";
	if ("JIS".equalsIgnoreCase (encodingName))
	    return "ISO-2022-JP";
	if ("EUCJIS".equalsIgnoreCase (encodingName))
	    return "EUC-JP";

	// UTF-16 == ISO-10646-UCS-2 plus surrogates.
	// The sun.io "Unicode" encoders/decoders don't check.

	// hope this works ...
	return encodingName;
    }

    /**
     * Assigns the size limit used to detect attempted serialization
     * of infinite graphs.  Documents that attempt to include more
     * than this many distinct objects will be rejected.
     */
    public void setSizeLimit (int maximum) 
	{ sizeLimit = maximum; }
    
    /**
     * Returns the size limit used to detect attempted serialization
     * of infinite graphs.
     */
    public int getSizeLimit ()
	{ return sizeLimit; }


    /**
     * You <em>must</em> close this stream after writing the XML
     * data out.  This is an artifact of XML's syntax, which permits
     * data (such as processing directives) after the end of the
     * document element in an XML stream; only an end-of-file truly
     * signifies the end of XML data.
     *
     * <P> <em> At this time, this causes the underlying writer to
     * close too! </em>
     */
    public void close () throws IOException
    {
	if (indentLevel != 1)
	    throw new XmlStreamException ("unbalanced");
	out.write ("\n</STREAM>\n");
	out.write ("<!-- end of document -->\n");

	// XXX check whether we can realistically not close the writer
	// underlying this -- will that violate all conventions?

	out.close ();
	mapping = null;
    }

    /**
     * Writes any buffered data out.
     */
    public void flush () throws IOException
    {
	out.flush ();
    }

    private void indent () throws IOException
    {
	int	temp = indentLevel;

	out.write ('\n');
	while (temp >= 8) {
	    out.write ('\t');
	    temp -= 8;
	}
	while (temp-- > 0)
	    out.write (' ');
    }

    //	BOOLEAN

    /** Writes a boolean value. */
    public void writeBoolean (boolean b) throws IOException
    {
	indent ();
	if (b)
	    out.write ("<boolean V=\"1\"/>");
	else
	    out.write ("<boolean/>");
    }

    // INTEGRAL TYPES

    /** Writes a byte value. */
    public void writeByte (int b) throws IOException
    {
	write (b);
    }

    /** Writes a byte value. */
    public void write (int b) throws IOException
    {
	indent ();
	out.write ("<i1");
	if (b != 0) {
	    out.write (" V=\"");
	    out.write (Byte.toString ((byte)b));
	    out.write ('"');
	}
	out.write ("/>");
    }

    /** Writes a short value. */
    public void writeShort (int s) throws IOException
    {
	indent ();
	out.write ("<i2");
	if (s != 0) {
	    out.write (" V=\"");
	    out.write (Short.toString ((short)s));
	    out.write ('"');
	}
	out.write ("/>");
    }

    /** Writes an integer value. */
    public void writeInt (int i) throws IOException
    {
	indent ();
	out.write ("<i4");
	if (i != 0) {
	    out.write (" V=\"");
	    out.write (Integer.toString (i));
	    out.write ('"');
	}
	out.write ("/>");
    }

    /** Writes a long value. */
    public void writeLong (long l) throws IOException
    {
	indent ();
	out.write ("<i8");
	if (l != 0) {
	    out.write (" V=\"");
	    out.write (Long.toString (l));
	    out.write ('"');
	}
	out.write ("/>");
    }

    // FLOATING POINT TYPES

    /** Writes a single precision floating point  value. */
    public void writeFloat (float f) throws IOException
    {
	indent ();
	out.write ("<r4");
	if (f != 0.0) {
	    out.write (" V=\"");
	    out.write (Float.toString (f));
	    out.write ('"');
	}
	out.write ("/>");
    }

    /** Writes a double precision floating point  value. */
    public void writeDouble (double d) throws IOException
    {
	indent ();
	out.write ("<r8");
	if (d != 0.0) {
	    out.write (" V=\"");
	    out.write (Double.toString (d));
	    out.write ('"');
	}
	out.write ("/>");
    }

    // CHARACTER TYPES

    /** Writes a character value. */
    public void writeChar (int c) throws IOException
    {
	// NOTE:  Needs to be numeric (in general) due to
	// XML disallowing control characters and standalone
	// surrogates.
	indent ();
	out.write ("<c V=\"");
	out.write (Short.toString ((short) c));
	out.write ("\"/>");
    }

    private void outChar (char c) throws IOException
    {
	if (c == '<')
	    out.write ("&lt;");
	else if (c == '&')
	    out.write ("&amp;");
	else if (!XmlChars.isChar (c)) {
	    out.write ("<c V=\"");
	    out.write (Short.toString ((short) c));
	    out.write ("\"/>");
	} else
	    // Expect the output writer doesn't mangle this...
	    out.write (c);
    }


    /**
     * DataOutput method which discards parts of the characters in
     * the most common implementations.
     * @deprecated Use writeChars, which never discards any upper
     *	bytes of the characters in the string.
     */
    public void writeBytes (String s) throws IOException
    {
	// NOTE: this is a "discard high byte" conversion according
	// to DataOutputStream; not according to DataOutput.
	writeChars (s);
    }

    /** Writes a string value. */
    public void writeChars (String s) throws IOException
    {
	indent ();

	if (s == null) {
	    out.write ("<NULL/>");
	    return;
	}

	int	len = s.length ();

	out.write ("<STRING>");
	for (int i = 0; i < len; i++)
	    outChar ((char) s.charAt (i));
	out.write ("</STRING>");
    }

    /**
     * DataOutput method which refuses to encode long strings.
     * @deprecated Use writeChars instead, since it doesn't
     *	refuse to encode longish strings.
     */
    public void writeUTF (String s) throws IOException
    {
	//
	// This call's supposed to be an error for strings
	// that take 2^16 bytes or more to write, in their
	// UTF-8 encoded form.  Kind of awkward to do in
	// general; we do the best we can easily do (testing
	// for longer than 2^16 characters, fails if any
	// non-ASCII characters are in the string).
	//
	if (s != null && s.length () > 65535)
	    throw new UTFDataFormatException ();

	writeChars (s);
    }

    /**
     * Writes opaque data from the buffer.  The recipient must
     * understand how to interpret this.
     */
    public void write (byte buf [], int off, int len) throws IOException
    {
	if (len < 0)
	    throw new IllegalArgumentException ("len < 0");

	Base64Encoder	encoder = new Base64Encoder (out);

	// note that the receiving end may need to be able to
	// group bunches of opaque data together... supporting
	// reads which bundle these together, as well as ones
	// split an opaque block into several reads.

	indent ();
	out.write ("<OPAQUE LENGTH=\"");
	out.write (Integer.toString (len));
	out.write ("\">\n");
	encoder.write (buf, off, len);
	encoder.close ();
	indent ();
	out.write ("</OPAQUE>");
    }

    /**
     * Shorthand for <code>write (buf, 0, buf.length)</code>.
     */
    public void write (byte buf []) throws IOException
    {
	write (buf, 0, buf.length);
    }

    /**
     * Writes an object.  At this time the object must be either
     * null or a bean which can be reconstructed from its properties.
     * When there is reason to suspect that the recipient of this
     * XML message will not have access to the class of such objects,
     * you should construct the stream with the URL of a JAR file
     * holding those classes.
     *
     * <P> This will report an exception if too many objects are
     * sent to the stream, exceeding the specified size limit.
     * That mechanism exists to prevent attempted encoding of
     * infinite graphs.
     *
     * <P> Future work here should support 
     * the <code>Externalizable</code> interface to allow
     * objects to affect how they are published.
     */
    public void writeObject (Object o) throws IOException
    {
	Class	objClass;
	String	objId;

	if (o == null) {
	    out.write ("<NULL/>");
	    return;
	}
	objClass = o.getClass ();

	//
	// If we already externalized this one, refer to its ID ...
	// relies on objects to support "hashCode" and "equals" correctly.
	// Works with arrays and other objects.
	//
	if ((objId = (String) mapping.get (o)) != null) {
	    indent ();
	    out.write ("<OBJECT IDREF=\"");
	    out.write (objId);
	    out.write ("\"/>");
	    return;
	}

	//
	// If it publishes intelligently, use that ... we don't use any
	// of the serialization infrastructure at this time, though perhaps
	// externalizable would be OK.
	// 
	if (o instanceof Externalizable) {
	    // XXX support self-publishing...
	}

	//
	// Defend against infinite graphs, e.g. property "get" functions
	// which create new objects with different hashCode values each
	// time, and/or which aren't equal to each other.
	//
	if (mapping.size () >= sizeLimit)
	    throw new XmlStreamException (
		"writeObject size limit exceeded: infinite graph?");

	//
	// OK, we've got to actually do the work.
	//
	try {
	    objClass = o.getClass ();
	    if (objClass.isArray ())
		writeArray (o, objClass);
	    else
		writeBean (o, objClass);

	} catch (IntrospectionException e) {
	    e.printStackTrace ();
	    throw new XmlStreamException ("writeObject introspection: "
		+ e.getMessage ());
	
	} catch (InvocationTargetException e) {
	    e.getTargetException ().printStackTrace ();
	    throw new XmlStreamException ("writeObject invocation: "
		+ e.getTargetException ().getMessage ());
	
	} catch (IllegalAccessException e) {
	    e.printStackTrace ();
	    throw new XmlStreamException ("writeObject access: "
		+ e.getMessage ());
	}
    }

    private void writeBean (Object bean, Class objClass)
    throws IOException, IntrospectionException,
	IllegalAccessException, InvocationTargetException
    {
	BeanInfo		info = Introspector.getBeanInfo (objClass);

	// MethodDescriptor	methods [] = info.getMethodDescriptors ();
	// if no public default constructor ... error!

	//
	// Emit a bunch of tagged properties
	//
	PropertyDescriptor	props [] = info.getPropertyDescriptors ();
	int			savedProps = 0;
	String			idString = "bean" + id++;

	indent ();
	out.write ("<BEAN CLASS='");
	out.write (objClass.getName ());
	out.write ("' ID='");
	out.write (idString);
	out.write ("'>");
	indentLevel++;

	// store mapping early, properties may refer to it ...
	mapping.put (bean, idString);

	for (int i = 0; i < props.length; i++) {
	    Method		getter = props [i].getReadMethod ();
	    Method		putter = props [i].getWriteMethod ();

	    if (getter == null || putter == null
		    || !Modifier.isPublic (getter.getModifiers ())
		    || !Modifier.isPublic (putter.getModifiers ())
		    )
		continue;
	    indent ();
	    out.write ("<PROPERTY NAME=\"" + props [i].getName () + "\">");
	    writeProperty (bean, getter);
	    out.write ("</PROPERTY>");
	    savedProps++;
	}
	if (savedProps == 0)
	    throw new XmlStreamException ("Can't XML-ize this bean:  " + bean);
	indent ();
	out.write ("</BEAN>");
	indentLevel--;
    }

    private void writeProperty (Object bean, Method getter)
    throws IOException, IllegalAccessException,
	IllegalArgumentException, InvocationTargetException
    {
	Class	propType = getter.getReturnType ();
	Object	value = getter.invoke (bean, null);

	indentLevel++;

	if (propType.equals (boolean.class)) {
	    if (value == Boolean.TRUE)
		writeBoolean (true);
	    else
		writeBoolean (false);
	}

	else if (propType.equals (byte.class))
	    writeByte (((Byte)value).byteValue ());
	else if (propType.equals (short.class))
	    writeShort (((Short)value).shortValue ());
	else if (propType.equals (int.class))
	    writeInt (((Integer)value).intValue ());
	else if (propType.equals (long.class))
	    writeLong (((Long)value).longValue ());

	else if (propType.equals (float.class))
	    writeFloat (((Float)value).floatValue ());
	else if (propType.equals (double.class))
	    writeDouble (((Double)value).doubleValue ());

	else if (propType.equals (char.class))
	    writeChar (((Character)value).charValue ());
	else if (propType.equals (String.class))
	    writeChars ((String) value);

	else
	    writeObject (value);

	indentLevel--;
    }

    private void writeArray (Object array, Class arrayClass)
    throws IOException, IntrospectionException,
	IllegalAccessException, InvocationTargetException
    {
	String		idString = "array" + id++;
	Class		elementType = arrayClass.getComponentType ();
	int		length = Array.getLength (array);

	indent ();
	out.write ("<ARRAY CLASS='");
	out.write (elementType.getName ());
	out.write ("' LENGTH='");
	out.write (Integer.toString (length));
	out.write ("' ID='");
	out.write (idString);
	out.write ("'>");

	// store mapping early...
	mapping.put (array, idString);

	for (int i = 0; i < length; i++) {
	    indent ();
	    out.write ("<ELEMENT INDEX='" + i + "'>");
	    writeElement (elementType, array, i);
	    out.write ("</ELEMENT>");
	}

	indent ();
	out.write ("</ARRAY>");
	indentLevel--;
    }

    private void writeElement (Class elementType, Object array, int index)
    throws IOException, IllegalAccessException,
	IllegalArgumentException, InvocationTargetException
    {
	indentLevel++;

	if (elementType.equals (boolean.class))
	    writeBoolean (Array.getBoolean (array, index));

	else if (elementType.equals (byte.class))
	    writeByte (Array.getByte (array, index));
	else if (elementType.equals (short.class))
	    writeShort (Array.getShort (array, index));
	else if (elementType.equals (int.class))
	    writeInt (Array.getInt (array, index));
	else if (elementType.equals (long.class))
	    writeLong (Array.getLong (array, index));

	else if (elementType.equals (float.class))
	    writeFloat (Array.getFloat (array, index));
	else if (elementType.equals (double.class))
	    writeDouble (Array.getDouble (array, index));

	else if (elementType.equals (char.class))
	    writeChar (Array.getChar (array, index));
	else if (elementType.equals (String.class))
	    writeChars ((String) Array.get (array, index));

	else
	    writeObject (Array.get (array, index));

	indentLevel--;
    }
}
